/* KiscMathScript.cc */
/* Created by Enomoto Sanshiro on 15 July 2008 */
/* Last updated by Enomoto Sanshiro on 15 July 2008 */


#include "ParaParser.hh"
#include "KumaFourierTransform.hh"
#include "KumaDigitalFilter.hh"
#include "KiscMathScript.hh"

using namespace std;
using namespace kuma;


TKiscMathScript::TKiscMathScript(void)
{
}

TKiscMathScript::~TKiscMathScript()
{
}

TParaTokenTable* TKiscMathScript::CreateTokenTable(void)
{
    TParaTokenTable* TokenTable;
    TokenTable = TParaParser::CreateTokenTable();

    return TokenTable;
}

TParaOperatorTable* TKiscMathScript::CreateOperatorTable(void)
{
    TParaOperatorTable* OperatorTable;
    OperatorTable = TParaParser::CreateOperatorTable();

    return OperatorTable;
}

TParaBuiltinFunctionTable* TKiscMathScript::CreateBuiltinFunctionTable(void)
{
    TParaBuiltinFunctionTable* BuiltinFunctionTable;
    BuiltinFunctionTable = TParaParser::CreateBuiltinFunctionTable();

    BuiltinFunctionTable->RegisterAnonymousClass(
	new TKiscSignalProcessingObject()
    );

    return BuiltinFunctionTable;
}

TParaObjectPrototypeTable* TKiscMathScript::CreateObjectPrototypeTable(void)
{
    TParaObjectPrototypeTable* ObjectPrototypeTable;
    ObjectPrototypeTable = TParaParser::CreateObjectPrototypeTable();

    ObjectPrototypeTable->RegisterClass(
	"MovingAverageFilter", new TKiscDigitalFilterObject()
    );
    ObjectPrototypeTable->RegisterClass(
	"WindowedSincFilter", new TKiscDigitalFilterObject()
    );

    ObjectPrototypeTable->RegisterClass(
	"SinglePoleFilter", new TKiscDigitalFilterObject()
    );
    ObjectPrototypeTable->RegisterClass(
	"SinglePoleLowPassFilter", new TKiscDigitalFilterObject()
    );
    ObjectPrototypeTable->RegisterClass(
	"SinglePoleHighPassFilter", new TKiscDigitalFilterObject()
    );

    ObjectPrototypeTable->RegisterClass(
	"ChebychevFilter", new TKiscDigitalFilterObject()
    );
    ObjectPrototypeTable->RegisterClass(
	"ChebychevLowPassFilter", new TKiscDigitalFilterObject()
    );
    ObjectPrototypeTable->RegisterClass(
	"ChebychevHighPassFilter", new TKiscDigitalFilterObject()
    );

    ObjectPrototypeTable->RegisterClass(
	"ButterworthFilter", new TKiscDigitalFilterObject()
    );
    ObjectPrototypeTable->RegisterClass(
	"ButterworthLowPassFilter", new TKiscDigitalFilterObject()
    );
    ObjectPrototypeTable->RegisterClass(
	"ButterworthHighPassFilter", new TKiscDigitalFilterObject()
    );

    return ObjectPrototypeTable;
}



TKiscSignalProcessingObject::TKiscSignalProcessingObject(void)
: TParaObjectPrototype("KiscSignalProcessing")
{
}

TKiscSignalProcessingObject::~TKiscSignalProcessingObject()
{
}

TParaObjectPrototype* TKiscSignalProcessingObject::Clone(void)
{
    return new TKiscSignalProcessingObject();
}

int TKiscSignalProcessingObject::MethodIdOf(const std::string& MethodName)
{
    if (MethodName == "fft") {
	return MethodId_FastFourierTransform;
    }
    else if (MethodName == "ifft") {
	return MethodId_InverseFastFourierTransform;
    }
    else if (MethodName == "parzenWindow") {
	return MethodId_DataWindowParzen;
    }
    else if (MethodName == "hanningWindow") {
	return MethodId_DataWindowHanning;
    }
    else if (MethodName == "hammingWindow") {
	return MethodId_DataWindowHamming;
    }
    else if (MethodName == "welchWindow") {
	return MethodId_DataWindowWelch;
    }
    else if (MethodName == "blackmanWindow") {
	return MethodId_DataWindowBlackman;
    }
    
    return TParaObjectPrototype::MethodIdOf(MethodName);
}

int TKiscSignalProcessingObject::InvokeMethod(int MethodId, std::vector<TParaValue*>& ArgumentList, TParaValue& ReturnValue) throw(TScriptException)
{
    if (MethodId == MethodId_FastFourierTransform) {
	FastFourierTransform(ArgumentList, ReturnValue, MethodId);
    }
    else if (MethodId == MethodId_InverseFastFourierTransform) {
	FastFourierTransform(ArgumentList, ReturnValue, MethodId);
    }
    else if (MethodId == MethodId_DataWindowParzen) {
	DataWindow(ArgumentList, ReturnValue, MethodId);
    }
    else if (MethodId == MethodId_DataWindowHanning) {
	DataWindow(ArgumentList, ReturnValue, MethodId);
    }
    else if (MethodId == MethodId_DataWindowHamming) {
	DataWindow(ArgumentList, ReturnValue, MethodId);
    }
    else if (MethodId == MethodId_DataWindowWelch) {
	DataWindow(ArgumentList, ReturnValue, MethodId);
    }
    else if (MethodId == MethodId_DataWindowBlackman) {
	DataWindow(ArgumentList, ReturnValue, MethodId);
    }
    else {
	return TParaObjectPrototype::InvokeMethod(MethodId, ArgumentList, ReturnValue);
    }

    return 1;
}

int TKiscSignalProcessingObject::FastFourierTransform(std::vector<TParaValue*>& ArgumentList, TParaValue& ReturnValue, int MethodId) throw(TScriptException)
{
    if ((ArgumentList.size() < 1) || (! ArgumentList[0]->IsList())) {
	throw TScriptException(
	    "fft(list data_list)", "invalid argument[s]"
	);
    }

    TKumaComplexVector DataList;

    int n = ArgumentList[0]->AsValueList().size(), m = 2;
    while ((0x0001 << m) < n) {
	m++;
    }
    n = 0x0001 << m;

    const vector<TParaValue>& ValueList = ArgumentList[0]->AsValueList();
    for (unsigned i = 0; i < ValueList.size(); i++) {
	DataList.Add(ValueList[i].AsComplex());
    }
    while (DataList.Size() < n) {
	DataList.Add(complex<double>(0.0, 0.0));
    }

    int Sign = 1;
    double Factor = 1.0;
    if (MethodId == MethodId_InverseFastFourierTransform) {
	Sign = -1;
	Factor = 1.0/DataList.Size();
    }

    try {
	TKumaFastFourierTransformer Transformer;
	Transformer.Transform(DataList.RawArray(), DataList.Size(), Sign);
    }
    catch (TKumaException &e) {
	throw TScriptException("fft(list data_list)",  e.Message());
    }

    ReturnValue = TParaValue(TParaListValue());
    vector<TParaValue>& ReturnValueList = ReturnValue.AsValueList();
    for (int i = 0; i < DataList.Size(); i++) {
	ReturnValueList.push_back(TParaValue(Factor * DataList[i]));
    }

    return 1;
}

int TKiscSignalProcessingObject::DataWindow(std::vector<TParaValue*>& ArgumentList, TParaValue& ReturnValue, int MethodId) throw(TScriptException)
{
    if ((ArgumentList.size() < 1) || (! ArgumentList[0]->IsLong())) {
	throw TScriptException(
	    "dataWindow(int data_length)", "invalid argument[s]"
	);
    }

    long N = ArgumentList[0]->AsLong();
    double* Window = new double[N];
    double SumOfSquared = 0;
    if (MethodId == MethodId_DataWindowParzen) {
	for (int i = 0; i < N; i++) {
	    Window[i] = 1 - fabs(((double) i - (N-1)/2.0) / ((N+1)/2.0));
	    SumOfSquared += sqr(Window[i]);
	}
    }
    else if (MethodId == MethodId_DataWindowHanning) {
	for (int i = 0; i < N; i++) {
	    Window[i] = (1 - cos(2*M_PI * (double) i/(N-1))) / 2.0;
	    SumOfSquared += sqr(Window[i]);
	}
    }
    else if (MethodId == MethodId_DataWindowHamming) {
	for (int i = 0; i < N; i++) {
	    Window[i] = 0.54 - 0.46 * cos(2*M_PI * (double) i/(N-1));
	    SumOfSquared += sqr(Window[i]);
	}
    }
    else if (MethodId == MethodId_DataWindowWelch) {
	for (int i = 0; i < N; i++) {
	    Window[i] = 1 - sqr(((double) i - (N-1)/2.0) / ((N+1)/2.0));
	    SumOfSquared += sqr(Window[i]);
	}
    }
    else if (MethodId == MethodId_DataWindowBlackman) {
	for (int i = 0; i < N; i++) {
	    Window[i] = (
		+ 0.42 
		- 0.5 * cos(2*M_PI * (double) i/(N-1))
		+ 0.08 * cos(4*M_PI * (double) i/(N-1))
	    );
	    SumOfSquared += sqr(Window[i]);
	}
    }

    double Normalization = sqrt(N / SumOfSquared);

    ReturnValue = TParaValue(TParaListValue());
    vector<TParaValue>& ReturnValueList = ReturnValue.AsValueList();

    for (int i = 0; i < N; i++) {
	ReturnValueList.push_back(TParaValue(Window[i] * Normalization));
    }

    delete[] Window;

    return 1;
}



TKiscDigitalFilterObject::TKiscDigitalFilterObject()
: TParaObjectPrototype("DigitalFilter")
{
    _DigitalFilter = 0;
}

TKiscDigitalFilterObject::~TKiscDigitalFilterObject(void)
{
    delete _DigitalFilter;
}

TParaObjectPrototype* TKiscDigitalFilterObject::Clone(void)
{
    return new TKiscDigitalFilterObject();
}

void TKiscDigitalFilterObject::Construct(const std::string& ClassName, std::vector<TParaValue*>& ArgumentList) throw(TScriptException)
{
    if (ArgumentList.size() < 1) {
	throw TScriptException(
	    ClassName + "::" + ClassName + "(float cutoff_freq, ...)", 
	    "too few argument[s]"
	);
    }
    if (! ArgumentList[0]->IsNumeric()) {
	throw TScriptException(
	    ClassName + "::" + ClassName + "(float cutoff_freq, ...)", 
	    "invalid argument[s]"
	);
    }
    double CutoffFrequency = ArgumentList[0]->AsDouble();

    int PassbandType;
    if (ClassName.find("HighPassFilter") != string::npos) {
	PassbandType = TKumaDigitalFilter::PassbandType_HighPass;
    }
    else {
	PassbandType = TKumaDigitalFilter::PassbandType_LowPass;
    }
    
    if (ClassName == "MovingAverageFilter") {
	_DigitalFilter = new TKumaMovingAverageFilter(CutoffFrequency);
    }
    else if (ClassName == "WindowedSincFilter") {
	int KernelLength = 0;
	if (ArgumentList.size() >= 2) {
	    if (! ArgumentList[1]->IsLong()) {
		throw TScriptException(
		    ClassName + "::" + ClassName +
		    "(float cutoff_freq, int kernel_length)", 
		    "invalid argument[s]"
		);
	    }
	    KernelLength = ArgumentList[1]->AsLong();
	}
	_DigitalFilter = new TKumaWindowedSincFilter(
	    CutoffFrequency, KernelLength
	);
    }
    else if (
	(ClassName == "SinglePoleFilter") ||
	(ClassName == "SinglePoleLowPassFilter") ||
	(ClassName == "SinglePoleHighPassFilter")
    ){
	_DigitalFilter = new TKumaSinglePoleFilter(
	    CutoffFrequency, PassbandType
	);
    }
    else if (
	(ClassName == "ChebychevFilter") ||
	(ClassName == "ChebychevLowPassFilter") ||
	(ClassName == "ChebychevHighPassFilter") 
    ){
	if (ArgumentList.size() < 3) {
	    throw TScriptException(
		ClassName + "::" + ClassName + 
		"(float cutoff_freq, int n_of_poles, float max_ripple)", 
		"too few argument[s]"
	    );
	}
	if (
	    (! ArgumentList[1]->IsLong()) || (! ArgumentList[2]->IsNumeric())
	){
	    throw TScriptException(
		ClassName + "::" + ClassName +
		"(float cutoff_freq, int n_of_poles, float max_ripple)", 
		"invalid argument[s]"
	    );
	}
	int NumberOfPoles = ArgumentList[1]->AsLong();
	double MaxRipple = ArgumentList[2]->AsDouble();
	_DigitalFilter = new TKumaChebychevFilter(
	    CutoffFrequency, NumberOfPoles, MaxRipple, PassbandType
	);
    }
    else if (
	(ClassName == "ButterworthFilter")  ||
	(ClassName == "ButterworthLowPassFilter")  ||
	(ClassName == "ButterworthHighPassFilter") 
    ){
	if (ArgumentList.size() < 2) {
	    throw TScriptException(
		ClassName + "::" + ClassName + 
		"(float cutoff_freq, int n_of_poles)", 
		"too few argument[s]"
	    );
	}
	if (! ArgumentList[1]->IsLong()) {
	    throw TScriptException(
		ClassName + "::" + ClassName +
		"(float cutoff_freq, int n_of_poles)", 
		"invalid argument[s]"
	    );
	}
	int NumberOfPoles = ArgumentList[1]->AsLong();
	_DigitalFilter = new TKumaButterworthFilter(
	    CutoffFrequency, NumberOfPoles, PassbandType
	);
    }
    else {
	throw TScriptException("unknown class: " + ClassName);
    }
}

int TKiscDigitalFilterObject::MethodIdOf(const std::string& MethodName)
{
    if (MethodName == "applyTo") {
	return MethodId_ApplyTo;
    }

    return TParaObjectPrototype::MethodIdOf(MethodName);
}

int TKiscDigitalFilterObject::InvokeMethod(int MethodId, std::vector<TParaValue*>& ArgumentList, TParaValue& ReturnValue) throw(TScriptException)
{
    if (MethodId == MethodId_ApplyTo) {
	ApplyTo(ArgumentList, ReturnValue);
    }
    else {
	return TParaObjectPrototype::InvokeMethod(MethodId, ArgumentList, ReturnValue);
    }

    return 1;
}

void TKiscDigitalFilterObject::ApplyTo(std::vector<TParaValue*>& ArgumentList, TParaValue& ReturnValue) throw(TScriptException)
{
    if ((ArgumentList.size() != 1) || (! ArgumentList[0]->IsList())) {
        throw TScriptException(
	    "DititalFiter::applyTo()", "invalid argument[s]"
	);
    }
    
    TParaListValue& ListValue = ArgumentList[0]->AsList();
    vector<TParaValue>& ValueList = ListValue.ValueList();
    unsigned ListLength = ValueList.size();

    double* DataList = new double[ListLength];
    for (unsigned i = 0; i < ListLength; i++) {
	DataList[i] = ValueList[i].AsDouble();
    }

    _DigitalFilter->ApplyTo(DataList, ListLength);

    for (unsigned i = 0; i < ListLength; i++) {
	ValueList[i] = TParaValue(DataList[i]);
    }
    delete[] DataList;
}
