/* KorbNetworkHub.cc */
/* Created by Enomoto Sanshiro on 4 December 1999. */
/* Updated by Enomoto Sanshiro on 7 February 2000. */
/* Last updated by Enomoto Sanshiro on 19 January 2010. */


#include <iostream>
#include <fstream>
#include "MushNetworkSocket.hh"
#include "KorbNetworkHub.hh"

using namespace std;


//#define DEBUG
#ifdef DEBUG
static ofstream dumpfile("Dump-KorbNetworkHub", ios::app);
#endif


TKorbNetworkHub::TKorbNetworkHub(int DomainId, int MessageQueueProjectId, const string& KorbPath)
{
    _DomainId = DomainId;
    
    _MessageQueueProjectId = MessageQueueProjectId;
    _KorbPath = KorbPath;

    _RoutingTable = new TKorbNetworkRoutingTable(this);
    _SocketSelector = new TMushSocketSelector();

    _MessageQueue = 0;
}

TKorbNetworkHub::~TKorbNetworkHub()
{
    delete _MessageQueue;
    delete _SocketSelector;
    delete _RoutingTable;
}

int TKorbNetworkHub::DomainId(void) const
{
    return _DomainId;
}

void TKorbNetworkHub::Initialize(void) throw(TKorbException)
{
    try {
	unsigned AccessPermission = 0600;
	bool IsSelfish = true;
	_MessageQueue = new TMushServerMessageQueue(
	    _MessageQueueProjectId, _KorbPath, AccessPermission, IsSelfish
	); 
	_MessageQueue->SetAddress(MessageQueueAddress);
    }
    catch (TSystemCallException &e) {
	throw TKorbException(
	    "TKorbNetworkHub::Initialize()",
	    "system call exception: " + e.Message()
	);
    }
}

void TKorbNetworkHub::Finalize(void) throw(TKorbException)
{
    if (NumberOfPendingPackets() > 0) {
	cerr << "WARNING: KorbNetworkHub: unprocessed packets left" << endl;
    }
}

void TKorbNetworkHub::AddConnection(int RemoteDomainId, TMushSocket* Socket)
{
    int UplinkPortId = Socket->FileDescriptor();

    _SocketTable[UplinkPortId] = Socket;
    _SocketSelector->AddSocket(Socket);
    _RoutingTable->AddConnection(UplinkPortId, RemoteDomainId);

#ifdef DEBUG
    dumpfile << "Add new connection" << endl;
    dumpfile << "  Uplink Port Id: " << UplinkPortId << endl;
    dumpfile << "  Remote Domain Id: " << RemoteDomainId << endl;
    dumpfile << endl;
#endif
}

void TKorbNetworkHub::RemoveConnection(TMushSocket* Socket)
{
    int UplinkPortId = Socket->FileDescriptor();

    _SocketTable.erase(UplinkPortId);
    _SocketSelector->RemoveSocket(Socket);
    //_RoutingTable->RemoveConnection(UplinkPortId);

#ifdef DEBUG
    dumpfile << "Remove connection" << endl;
    dumpfile << "  Uplink Port Id: " << UplinkPortId << endl;
    dumpfile << endl;
#endif
}

int TKorbNetworkHub::NumberOfPendingPackets(void)
{
    int Result = _MessageQueue->NumberOfPendingPackets();
    Result += _WaitingQueue.size();
    Result += _ReceivedPacketQueue.size();
    Result += _LocalDestinationPacketQueue.size();

    return Result;
}

int TKorbNetworkHub::SendToUplinkPort(int UplinkPortId, TKorbNetworkPacket& Packet) throw(TKorbException)
{
    int SendLength = 0;

    if (_SocketTable.count(UplinkPortId) == 0) {
	throw TKorbException(
	    "TKorbNetworkHub::SendToUplinkPort()",
	    "bad uplink port id"
	);
    }

    try {
	_SocketTable[UplinkPortId]->Send(
	    Packet.ContentsAddress(), 
	    Packet.ContentsSize()
	);
    }
    catch (TSystemCallException &e) {
	throw TKorbException(
	    "TKorbNetworkHub::SendToUplinkPort()",
	    "system call exception: " + e.Message()
	);
    }

#ifdef DEBUG
    dumpfile << "Send to Uplink Port " << UplinkPortId << endl;
    Packet.Dump(dumpfile);
#endif
    
    return SendLength;
}

int TKorbNetworkHub::SendToLocalPort(TKorbNetworkPacket& Packet) throw(TKorbException)
{
    TMushMessageQueuePacket* MessageQueuePacket
	= (TMushMessageQueuePacket*) Packet.ContentsAddress();
    MessageQueuePacket->MessageType = Packet.ReceiverAdapterId();
    int MessageLength = Packet.ContentsSize();

    int SentLength;
    try {
	bool NoWait = true;
	SentLength = _MessageQueue->Send(
	    MessageQueuePacket, MessageLength, NoWait
	);
	if (SentLength == 0) {
	    static bool IsWarningDisplayed = false;
	    if (! IsWarningDisplayed) {
		cerr << "WARNING: MessageQueue Full: ";
		cerr << "Max message queue size (kernel parameter) too small?";
		cerr << endl;
		cerr << "Current max queue size: ";
		cerr << _MessageQueue->QueueCapacity() << endl;
		IsWarningDisplayed = true;
	    }
	    return 0;
	}
    }
    catch (TSystemCallException &e) {
	throw TKorbException(
	    "TKorbNetworkHub::SendToLocalPort()",
	    "system call exception: " + e.Message()
	);
    }
    if (SentLength != MessageLength) {
        throw TKorbException(
	    "TKorbNetworkAdapter::Send()",
	    "system call exception: unable to send message"
        );
    }

#ifdef DEBUG
    dumpfile << "Send to Local Port" << endl;
    Packet.Dump(dumpfile);
#endif

    return SentLength;
}

int TKorbNetworkHub::ReceiveFromUplinkPort(TKorbNetworkPacket& Packet, int& UplinkPortId) throw(TKorbException)
{
    int ReceivedLength = 0;

    if (_SocketTable.empty()) {
	return ReceivedLength = 0;
    }

    pair<TMushSocket*, TMushSocketSelector::TSocketEvent> SocketEvent;
    try {
	static const int TimeOut_ms = 0;
	SocketEvent = _SocketSelector->WaitEvent(TimeOut_ms);
    }
    catch (TSystemCallException &e) {
	throw TKorbException(
	    "TKorbNetworkHub::ReceiveFromUplinkPort()",
	    "system call exception: " + e.Message()
	);
    }

    TMushSocket* Socket = SocketEvent.first;
    if (Socket == 0) {
	return ReceivedLength = 0;
    }
    else if (SocketEvent.second == TMushSocketSelector::SocketEvent_HungUp) {
	Socket->Close();
	RemoveConnection(Socket);
	return 0;
    }
    
    UplinkPortId = Socket->FileDescriptor();
    try {
	ReceivedLength = Socket->Receive(
	    Packet.ContentsAddress(), 
	    Packet.ContentsCapacity()
	);
    }
    catch (TSystemCallException &e) {
	throw TKorbException(
	    "TKorbNetworkHub::ReceiveFromUplinkPort()",
	    "system call exception: " + e.Message()
	);
    }

    if (ReceivedLength <= 0) {
	// note that read() is called after poll()'s RDIN //
	Socket->Close();
	RemoveConnection(Socket);
	return 0;
    }

    if (Packet.IsByteOrderReversed()) {
	Packet.ReverseByteOrder();
#ifdef DEBUG
	dumpfile << "[byte order was reversed]" << endl;
#endif
    }
    
#ifdef DEBUG
    dumpfile << "Receive from Uplink Port " << UplinkPortId << endl;
    Packet.Dump(dumpfile);
#endif
    
    return ReceivedLength;
}

int TKorbNetworkHub::ReceiveFromLocalPort(TKorbNetworkPacket& Packet) throw(TKorbException)
{
    int ReceivedLength = 0;

    TMushMessageQueuePacket* MessageQueuePacket
        = (TMushMessageQueuePacket*) Packet.ContentsAddress();
    int MaxMessageLength = Packet.ContentsCapacity() - sizeof(long);
    try {
	bool NoWait = true;
	ReceivedLength = _MessageQueue->Receive(
	    MessageQueuePacket, MaxMessageLength, NoWait
	);
    }
    catch (TSystemCallException &e) {
        throw TKorbException(
	    "TKorbNetworkHub::ReceiveFromLocalPort()",
	    "system call exception: " + e.Message()
        );
    }

    if (Packet.SenderDomainId() == DomainId_Unknown) {
	Packet.SetSender(_DomainId, Packet.SenderAdapterId());
    }

#ifdef DEBUG
    if (ReceivedLength) {
	dumpfile << "Receive from Local Port " << endl;
	Packet.Dump(dumpfile);
    }
#endif

    return ReceivedLength;
}

int TKorbNetworkHub::Send(TKorbNetworkPacket& Packet)
{
    _WaitingQueue.push_back(Packet);

    return 1;
}

int TKorbNetworkHub::Receive(TKorbNetworkPacket& Packet)
{
    if (_ReceivedPacketQueue.size() > 0) {
	Packet = _ReceivedPacketQueue.front();
	_ReceivedPacketQueue.pop_front();

	return 1;
    }

    return 0;
}

int TKorbNetworkHub::DoTransaction(void) throw(TKorbException)
{
    if (_MessageQueue == 0) {
	Initialize();
    }

    int NumberOfTransactions = 0;
    int UplinkPortId;

    while (ReceiveFromUplinkPort(_Packet, UplinkPortId) > 0) {
	ProcessPacket(_Packet, UplinkPortId);
	NumberOfTransactions++;
    }

    while (! _WaitingQueue.empty()) {
	ProcessPacket(_WaitingQueue.front());
	_WaitingQueue.pop_front();
	NumberOfTransactions++;
    }

    while (ReceiveFromLocalPort(_Packet) > 0) {
	ProcessPacket(_Packet);
	NumberOfTransactions++;
    }

    while (! _LocalDestinationPacketQueue.empty()) {
	if (SendToLocalPort(_LocalDestinationPacketQueue.front()) == 0) {
	    break;
	}
	_LocalDestinationPacketQueue.pop_front();
	NumberOfTransactions++;
    }

    return NumberOfTransactions;
}

int TKorbNetworkHub::ProcessPacket(TKorbNetworkPacket& Packet, int UplinkPortId) throw(TKorbException)
{
    int Result = 0;
    switch (Packet.PacketType()) {
      case TKorbNetworkPacket::ptRoutingQuery:
      case TKorbNetworkPacket::ptRoutingReply:
	Result = _RoutingTable->ProcessRoutingPacket(Packet, UplinkPortId);
	break;
      case TKorbNetworkPacket::ptEchoRequest:
	Result = ProcessEchoRequestPacket(Packet);
	break;
      case TKorbNetworkPacket::ptEchoReply:
      case TKorbNetworkPacket::ptMessage:
      default:
	Result = ProcessMessagePacket(Packet);
    }

    return Result;
}

int TKorbNetworkHub::ProcessMessagePacket(TKorbNetworkPacket& Packet) throw(TKorbException)
{
    int RemoteDomainId = Packet.ReceiverDomainId();

    if (RemoteDomainId == _DomainId) {
	if (Packet.ReceiverAdapterId() == _AdapterId) {
	    _ReceivedPacketQueue.push_back(Packet);
	}
	else {
	    if (! _LocalDestinationPacketQueue.empty()) {
		// keep the packet order //
		_LocalDestinationPacketQueue.push_back(Packet);
	    }		
	    else if (SendToLocalPort(Packet) == 0) {
		_LocalDestinationPacketQueue.push_back(Packet);
	    }
	}
    }
    else {
        int UplinkPortId = _RoutingTable->LookupRouteTo(RemoteDomainId);
	if (UplinkPortId >= 0) {
	    SendToUplinkPort(UplinkPortId, Packet);
	}
	else if (UplinkPortId == TKorbNetworkRoutingTable::lrSearching) {
	    _WaitingQueue.push_back(Packet);
	}
	else {
	    Packet.SetDeliverStatus(TKorbNetworkPacket::dsNoRoute);
	    Packet.SetReceiver(
		Packet.SenderDomainId(), Packet.SenderAdapterId()
	    );
	}
    }

    return 0;
}

int TKorbNetworkHub::ProcessEchoRequestPacket(TKorbNetworkPacket& Packet) throw(TKorbException)
{
    int RemoteDomainId = Packet.ReceiverDomainId();
    if (RemoteDomainId == DomainId_Unknown) {
	RemoteDomainId = _DomainId;
    }
    
    if (RemoteDomainId == _DomainId) {
	int ReceiverDomianId = Packet.SenderDomainId();
	int ReceiverAdapterId = Packet.SenderAdapterId();
	
	Packet.SetPacketType(TKorbNetworkPacket::ptEchoReply);
	Packet.SetSender(_DomainId, 0);
	Packet.SetReceiver(ReceiverDomianId, ReceiverAdapterId);
    }

    return ProcessMessagePacket(Packet);
}
