/* KorbObjectAdapter.cc */
/* Created by Enomoto Sanshiro on 27 February 2000. */
/* Last updated by Enomoto Sanshiro on 19 January 2010. */


#include <fstream>
#include <string>
#include "MushSignal.hh"
#include "MushTimer.hh"
#include "KorbOrb.hh"
#include "KorbObjectMessenger.hh"
#include "KorbObjectAdapter.hh"
#include "KorbNamingContext.hh"

using namespace std;


//#define DEBUG
#ifdef DEBUG
static ofstream dumpfile("Dump-KorbObjectAdapter", ios::app);
//#define dumpfile cerr
#endif


static const int InitialObjectId = 0x0100;


TKorbObjectAdapter::TKorbObjectAdapter(TKorbOrb* Orb)
{
    _Orb = Orb;

    _NextObjectId = InitialObjectId;
    _IsStopRequested = false;
}

TKorbObjectAdapter::~TKorbObjectAdapter()
{
}

void TKorbObjectAdapter::AddMessenger(TKorbObjectMessenger* Messenger, const string& ObjectName, int ObjectId) throw(TKorbException)
{
    if (ObjectId == 0) {
	ObjectId = _NextObjectId;
	_NextObjectId++;
    }

    _ObjectIdTable[ObjectName] = ObjectId;
    _MessengerTable[ObjectId] = Messenger;

    _Orb->NamingContext()->RegisterObject(TKorbObjectReference(
	Messenger->ClassName(), ObjectName, 
	_Orb->DomainId(), _Orb->AdapterId(), ObjectId
    ));

#ifdef DEBUG
    dumpfile << _Orb->DomainId() << ".";
    dumpfile << _Orb->AdapterId() << ".";
    dumpfile << ObjectId << ": ";
    dimpfile << ObjectName << endl;
#endif
}

void TKorbObjectAdapter::RemoveMessenger(const string& ObjectName) throw(TKorbException)
{
    if (_ObjectIdTable.count(ObjectName) > 0) {
	int ObjectId = _ObjectIdTable[ObjectName];
	_MessengerTable.erase(ObjectId);
    }

    _Orb->NamingContext()->UnregisterObject(ObjectName);
}

int TKorbObjectAdapter::NumberOfMessengers(void) const
{
    return _MessengerTable.size();
}

void TKorbObjectAdapter::Start(void) throw(TKorbException)
{
    TMushSignalHandler SignalHandler;
    TMushSignalCounter SignalCounter;
    SignalHandler.RegisterClient(SIGINT, &SignalCounter);
    SignalHandler.RegisterClient(SIGTERM, &SignalCounter);
    SignalHandler.RegisterClient(SIGHUP, &SignalCounter);
    SignalHandler.StartHandling();

    _IsStopRequested = false;

    while (! _IsStopRequested) {
	if (SignalCounter.SignalCount() > 0) {
	    cerr << "KorbObjectAdapter(";
	    if (! _ObjectIdTable.empty()) {
		cerr << _ObjectIdTable.begin()->first;
	    }
	    cerr << "): ";
	    cerr << "signaled: terminating..." << endl;
	    break;
	}

	DoTransaction();
    }

    SignalHandler.StartDefaultAction(SIGINT);
    SignalHandler.StartDefaultAction(SIGTERM);
    SignalHandler.StartDefaultAction(SIGHUP);

    _Orb->Finalize();
}

int TKorbObjectAdapter::DoTransaction(void) throw(TKorbException)
{
    TKorbOrbPacket OrbPacket = _Orb->CreateServerOrbPacket();

    int RequestId = _Orb->WaitRequest(OrbPacket);
    if (RequestId != 0) {
        return ProcessRequest(RequestId, OrbPacket);
    }
    
    return 0;
}

int TKorbObjectAdapter::DoNowaitTransaction(void) throw(TKorbException)
{
    TKorbOrbPacket OrbPacket = _Orb->CreateServerOrbPacket();

    int RequestId = _Orb->GetRequest(OrbPacket);
    if (RequestId != 0) {
        return ProcessRequest(RequestId, OrbPacket);
    }

    return 0;
}

int TKorbObjectAdapter::ProcessRequest(int RequestId, TKorbOrbPacket& OrbPacket) throw(TKorbException)
{
    int Result = 0;

    int ObjectId = OrbPacket.ObjectId();    
    bool IsOneWay = OrbPacket.IsOneWay();
    int ClientDomainId = OrbPacket.NetworkPacket()->SenderDomainId();
    int ClientAdapterId = OrbPacket.NetworkPacket()->SenderAdapterId();

#ifdef DEBUG
    dumpfile << "request " << RequestId << ": " << flush;
    int MyDomainId = OrbPacket.NetworkPacket()->ReceiverDomainId();
    int MyAdapterId = OrbPacket.NetworkPacket()->ReceiverAdapterId();
    dumpfile << ClientDomainId << "." << ClientAdapterId << ".?" << " => ";
    dumpfile << MyDomainId << "." << MyAdapterId << "." << ObjectId << ": ";
    dumpfile << "method " << OrbPacket.MethodId() << ", ";
    dumpfile << "request " << RequestId;
    dumpfile << (IsOneWay ? ", oneway" : "") <<endl;
#endif

    if (ObjectId == 0) {
	ProcessAdapterRequest(OrbPacket);
    }
    else if (_MessengerTable.count(ObjectId) > 0) {
	if (! _MessengerTable[ObjectId]->DispatchMethodInvocation(OrbPacket)) {
#ifdef DEBUG
	    dumpfile << "UNKNOWN METHOD" << endl;
#endif
	}
    }
    else {
#ifdef DEBUG
        dumpfile << "UNKNOWN OBJECT" << endl;
#endif
    }

    if (! IsOneWay) {
	_Orb->SendReplyFor(
	    RequestId, ClientDomainId, ClientAdapterId, OrbPacket
	);
    }

    return Result = 1;
}

int TKorbObjectAdapter::ProcessAdapterRequest(TKorbOrbPacket& OrbPacket) throw(TKorbException)
{
    int Result = 0;
    if (OrbPacket.MethodId() == MethodId_Exit) {
	_IsStopRequested = true;
    }

    return Result;
}
