/* KorbSerializer.cc */
/* Created by Enomoto Sanshiro on 21 February 2000. */
/* Last updated by Enomoto Sanshiro on 22 February 2000. */


#include <string>
#include "MushConfig.hh"
#include "KorbDefs.hh"
#include "KorbStreamable.hh"
#include "KorbSerializer.hh"

using namespace std;
using namespace mush;


TKorbSerializer::TKorbSerializer(void)
{
    _Buffer = 0;
    _BufferSize = 0;
    
    _Offset = 0;
    _IsPacketValid = true;
}

TKorbSerializer::TKorbSerializer(void* Buffer, int BufferSize)
{
    _Buffer = Buffer;
    _BufferSize = BufferSize;
    
    _Offset = 0;
    _IsPacketValid = true;
}

TKorbSerializer::TKorbSerializer(const TKorbSerializer& Serializer)
{
    _Buffer = Serializer._Buffer;
    _BufferSize = Serializer._BufferSize;
    _Offset = Serializer._Offset;
    _IsPacketValid = Serializer._IsPacketValid;
}

TKorbSerializer& TKorbSerializer::operator= (const TKorbSerializer& Serializer)
{
    _Buffer = Serializer._Buffer;
    _BufferSize = Serializer._BufferSize;
    _Offset = Serializer._Offset;
    _IsPacketValid = Serializer._IsPacketValid;

    return *this;
}

TKorbSerializer::~TKorbSerializer()
{
}

int TKorbSerializer::ProcessedSize(void) const
{
    return _Offset;
}

bool TKorbSerializer::IsPacketValid(void) const
{
    return _IsPacketValid;
}

void TKorbSerializer::Rewind(void)
{
    _Offset = 0;
    _IsPacketValid = true;
}

int TKorbSerializer::PutInt(int Value)
{
    static const int Size = sizeof(int);

    int Capacity = _BufferSize - _Offset;
    if (Size+1 > Capacity) {
	_IsPacketValid = false;
	return 0;
    }

    ((UInt8*) _Buffer)[_Offset] = Size & 0x00ff;
    _Offset++;

    for (int i = 0; i < Size; i++) {
	((UInt8*) _Buffer)[_Offset + i] = (Value >> (i * 8)) & 0x00ff;
    }
    _Offset += Size;

    return Size + 1;
}

int TKorbSerializer::GetInt(int& Value)
{
    Value = 0;
    
    int Size = ((UInt8*) _Buffer)[_Offset];
    _Offset++;

    for (int i = 0; i < Size; i++) {
	Value |= ((UInt8*) _Buffer)[_Offset + i] << (i * 8);
    }
    _Offset += Size;

    return Size + 1;
}

int TKorbSerializer::PutLong(long Value)
{
    static const int Size = sizeof(long);

    int Capacity = _BufferSize - _Offset;
    if (Size+1 > Capacity) {
	_IsPacketValid = false;
	return 0;
    }

    ((UInt8*) _Buffer)[_Offset] = Size & 0x00ff;
    _Offset++;

    for (int i = 0; i < Size; i++) {
	((UInt8*) _Buffer)[_Offset + i] = (Value >> (i * 8)) & 0x00ff;
    }
    _Offset += Size;

    return Size + 1;
}

int TKorbSerializer::GetLong(long& Value)
{
    Value = 0;

    int Size = ((UInt8*) _Buffer)[_Offset];
    _Offset++;

    for (int i = 0; i < Size; i++) {
	Value |= ((UInt8*) _Buffer)[_Offset + i] << (i * 8);
    }
    _Offset += Size;

    return Size + 1;
}

int TKorbSerializer::PutString(const string& Value)
{
    int Size = Value.size();

    int Capacity = _BufferSize - _Offset;
    if (Size+1 > Capacity) {
	_IsPacketValid = false;
	return 0;
    }

    for (int i = 0; i < Size; i++) {
	((UInt8*) _Buffer)[_Offset + i] = Value[i];
    }
    
    ((UInt8*) _Buffer)[_Offset + Size] = '\0';
    Size++;

    _Offset += Size;

    return Size;
}

int TKorbSerializer::GetString(string& Value)
{
    Value = (const char*) (((UInt8*) _Buffer) + _Offset);
    int Size = Value.size() + 1;  /* includes null terminator */
    _Offset += Size;

    return Size;
}

int TKorbSerializer::PutObject(const TKorbStreamable& StreamableObject)
{
    void* Base = ((UInt8*) _Buffer) + _Offset;
    int Capacity = _BufferSize - _Offset;

    int Size = StreamableObject.WriteTo(Base, Capacity);

    _Offset += Size;

    return Size;
}

int TKorbSerializer::GetObject(TKorbStreamable& StreamableObject)
{
    void* Base = ((UInt8*) _Buffer) + _Offset;
    int Capacity = _BufferSize - _Offset;

    int Size = StreamableObject.ReadFrom(Base, Capacity);

    _Offset += Size;

    return Size;
}
