/* KumaDigitalFilter.cc */
/* Created by Enomoto Sanshiro on 1 December 2007. */
/* Last updated by Enomoto Sanshiro on 1 December 2007. */


#include <iostream>
#include <iomanip>
#include <algorithm>
#include <string>
#include <cmath>
#include <complex>
#include "KumaVector.hh"
#include "KumaFunction.hh"
#include "KumaFourierTransform.hh"
#include "KumaDigitalFilter.hh"

using namespace std;
using namespace kuma;



TKumaFiniteImpulseResponseFilter::TKumaFiniteImpulseResponseFilter(void) 
{
    _LastIndex = 0;
    _CoeffA = 0;
}

TKumaFiniteImpulseResponseFilter::~TKumaFiniteImpulseResponseFilter() 
{
}

void TKumaFiniteImpulseResponseFilter::ApplyTo(double* SampleList, int Length)
{
    double* Y = SampleList;
    double* X = new double[Length];
    memcpy(X, SampleList, Length * sizeof(double));

    for (int i = _LastIndex; i < Length; i++) {
	Y[i] = 0;
	for (int j = _LastIndex; j >= 0; j--) {
	    Y[i] += _CoeffA[j] * X[i-j];
	}
    }

    delete[] X;
}

void TKumaFiniteImpulseResponseFilter::SetCoefficients(double* CoeffA, int LastIndex)
{
    _CoeffA = CoeffA;
    _LastIndex = LastIndex;
}

void TKumaFiniteImpulseResponseFilter::DumpCoefficients(std::ostream& os, const std::string& Header)
{
    os.setf(ios::scientific);
    os << setprecision(10);
    for (int i = 0; i <= _LastIndex; i++) {
	os << Header << i << "   " << _CoeffA[i] << endl;
    }
}



TKumaArbitraryResponseFilter::TKumaArbitraryResponseFilter(TKumaFunction& Response, int ResponseLength)
{
    _ResponseLength = ResponseLength;

    _CoeffA = new double[_ResponseLength];
    for (int i = 0; i < _ResponseLength; i++) {
	_CoeffA[i] = Response((double) i);
    }

    SetCoefficients(_CoeffA, _ResponseLength-1);
}

TKumaArbitraryResponseFilter::TKumaArbitraryResponseFilter(double* Response, int ResponseLength)
{
    _ResponseLength = ResponseLength;

    _CoeffA = new double[_ResponseLength];
    for (int i = 0; i < _ResponseLength; i++) {
	_CoeffA[i] = Response[i];
    }

    SetCoefficients(_CoeffA, _ResponseLength-1);
}

TKumaArbitraryResponseFilter::~TKumaArbitraryResponseFilter()
{
    delete[] _CoeffA;
}



TKumaMovingAverageFilter::TKumaMovingAverageFilter(double CutoffFrequency)
{
    _CutoffFrequency = CutoffFrequency;
    _WindowLength = (int) (1.0 / _CutoffFrequency);

    _CoeffA = new double[_WindowLength];
    
    for (int i = 0; i < _WindowLength; i++) {
	_CoeffA[i] = 1.0 / _WindowLength;
    }

    SetCoefficients(_CoeffA, _WindowLength-1);
}

TKumaMovingAverageFilter::~TKumaMovingAverageFilter()
{
    delete[] _CoeffA;
}



TKumaWindowedSincFilter::TKumaWindowedSincFilter(double CutoffFrequency, int KernelLength)
{
    _CutoffFrequency = CutoffFrequency;
    _KernelLength = KernelLength;

    if (_KernelLength <= 0) {
	_KernelLength = (int) (10 / _CutoffFrequency);
    }
    int CenterIndex = _KernelLength / 2;
    _KernelLength = 2 * CenterIndex;

    _CoeffA = new double[_KernelLength+1];
    
    // construct sinc() table //
    for (int i = 0; i <= _KernelLength; i++) {
	double X = (i-CenterIndex);
	_CoeffA[i] = sinc(X * _CutoffFrequency);	
    }

    // apply Blackman Window //
    double Sum = 0;
    for (int i = 0; i <= _KernelLength; i++) {
	double X = i / (double) _KernelLength;
	double Window = 0.42 - 0.5 * cos(2*M_PI*X) + 0.08 * cos(4*M_PI*X);
	_CoeffA[i] *= Window;
	Sum += _CoeffA[i];
    }

    // normalize //
    for (int i = 0; i <= _KernelLength; i++) {
	_CoeffA[i] /= Sum;
    }

    SetCoefficients(_CoeffA, _KernelLength);
}

TKumaWindowedSincFilter::~TKumaWindowedSincFilter()
{
    delete[] _CoeffA;
}



TKumaRecursiveFilter::TKumaRecursiveFilter(void) 
{
    _LastIndex = 0;
    _CoeffA = 0;
    _CoeffB = 0;
}

TKumaRecursiveFilter::~TKumaRecursiveFilter() 
{
}

void TKumaRecursiveFilter::ApplyTo(double* SampleList, int Length)
{
    double* Y = SampleList;
    double* X = new double[Length];
    memcpy(X, SampleList, Length * sizeof(double));

    for (int i = _LastIndex; i < Length; i++) {
	Y[i] = 0;
	for (int j = _LastIndex; j > 0; j--) {
	    Y[i] += _CoeffB[j] * Y[i-j];
	    Y[i] += _CoeffA[j] * X[i-j];
	}
	Y[i] += _CoeffA[0] * X[i];
    }

    delete[] X;
}

void TKumaRecursiveFilter::SetCoefficients(double* CoeffA, double* CoeffB, int LastIndex)
{
    _CoeffA = CoeffA;
    _CoeffB = CoeffB;
    _LastIndex = LastIndex;
}

void TKumaRecursiveFilter::DumpCoefficients(std::ostream& os, const std::string& Header)
{
    os.setf(ios::scientific);
    os << setprecision(10);
    os << Header << "0   " << _CoeffA[0] << endl;
    for (int i = 1; i <= _LastIndex; i++) {
	os << Header;
	os << i << "   " << _CoeffA[i] << "   " << _CoeffB[i] << endl;
    }
}



TKumaSinglePoleFilter::TKumaSinglePoleFilter(double CutoffFrequency, int PassbandType)
{
    // CutoffFrequency = 2*M_PI / TimeConstant //
    double Decay = exp(-2*M_PI * CutoffFrequency);

    if (PassbandType == PassbandType_HighPass) {
	_CoeffA[0] = (1.0 + Decay) / 2;
	_CoeffA[1] = -(1.0 + Decay) / 2;
	_CoeffB[0] = 0;
	_CoeffB[1] = Decay;
    }
    else {
	_CoeffA[0] = 1.0 - Decay;
	_CoeffA[1] = 0;
	_CoeffB[0] = 0;
	_CoeffB[1] = Decay;
    }

    SetCoefficients(_CoeffA, _CoeffB, 1);
}

TKumaSinglePoleFilter::~TKumaSinglePoleFilter()
{
}



TKumaChebychevFilter::TKumaChebychevFilter(double CutoffFrequency, int NumberOfPoles, double MaxRipple, int PassbandType)
{
    _PassbandType = PassbandType;
    _CutoffFrequency = CutoffFrequency;
    _NumberOfPoles = NumberOfPoles;
    _MaxRipple = MaxRipple;

    _CoeffA = new double[_NumberOfPoles+3];
    _CoeffB = new double[_NumberOfPoles+3];
    
    CalculateCoefficients();
}

TKumaChebychevFilter::~TKumaChebychevFilter()
{
    delete[] _CoeffA;
    delete[] _CoeffB;
}

void TKumaChebychevFilter::CalculateCoefficients(void)
{
    for (int i = 0; i <= _NumberOfPoles+2; i++) {
	_CoeffA[i] = 0;
	_CoeffB[i] = 0;
    }
    _CoeffA[2] = 1.0;
    _CoeffB[2] = 1.0;

    double PrevCoeffA[_NumberOfPoles+3], PrevCoeffB[_NumberOfPoles+3];

    for (int PoleIndex = 0; PoleIndex < _NumberOfPoles/2; PoleIndex++) {
	double A[3], B[3];
	CalculateCoefficientsFor(PoleIndex, A, B);

	// add coefficients to the cascade //
	memcpy(PrevCoeffA, _CoeffA, (_NumberOfPoles+3)*sizeof(double));
	memcpy(PrevCoeffB, _CoeffB, (_NumberOfPoles+3)*sizeof(double));
	for (int i = 2; i <= _NumberOfPoles+2; i++) {
	    _CoeffA[i] = 0;
	    _CoeffB[i] = 0;
	    for (int j = 0; j < 3; j++) {
		_CoeffA[i] += A[j]*PrevCoeffA[i-j];
		_CoeffB[i] -= B[j]*PrevCoeffB[i-j];  
	    }
	}
    }

    // finish combining coefficients //
    _CoeffB[2] = 0;
    for (int i = 0; i <= _NumberOfPoles; i++) {
	_CoeffA[i] = _CoeffA[i+2];
	_CoeffB[i] = -_CoeffB[i+2];
    }

    // normalize gain //
    double SumA = 0, SumB = 0;
    for (int i = 0; i <= _NumberOfPoles; i++) {
	int Sign = +1;
	if (_PassbandType == PassbandType_HighPass) {
	    Sign = ((i % 2) == 0) ? +1 : -1;
	}
	SumA += Sign * _CoeffA[i];
	SumB += Sign * _CoeffB[i];
    }
    double Gain = SumA / (1.0 - SumB);
    for (int i = 0; i <= _NumberOfPoles; i++) {
	_CoeffA[i] /= Gain;
    }

    SetCoefficients(_CoeffA, _CoeffB, _NumberOfPoles);
}

void TKumaChebychevFilter::CalculateCoefficientsFor(int PoleIndex, double A[3], double B[3])
{
    // calculate pole position on an unit circle //
    double Angle = M_PI/_NumberOfPoles * (PoleIndex + 0.5);
    double Px = -cos(Angle);
    double Py = sin(Angle);

    // convert from a circle to an ellipse //
    if (_MaxRipple > 0) {
	double eps = sqrt(sqr(1.0/(1.0-_MaxRipple)) - 1.0);
	double nu = asinh(1.0/eps) / _NumberOfPoles;
	double ka = cosh(acosh(1.0/eps) / _NumberOfPoles);
	Px *= sinh(nu) / ka;
	Py *= cosh(nu) / ka;
    }

    // S-domain to Z-domain conversion //
    const double T = 2.0 * tan(0.5);
    double T2 = sqr(T);
    double a[3], b[3];
    double M = sqr(Px) + sqr(Py);
    double D = 4.0 - 4.0*Px*T + M*T2;
    a[0] = T2/D;
    a[1] = 2 * a[0];
    a[2] = a[0];
    b[0] = 1.0;
    b[1] = (8.0 - 2*M*T2) / D;
    b[2] = (-4.0 - 4.0*Px*T - M*T2) / D;

    // LP, HP conversion //
    double w = 2*M_PI * _CutoffFrequency;
    double k, k2;
    if (_PassbandType == PassbandType_HighPass) {
	k = -cos((w+1.0)/2) / cos((w-1.0)/2);
    }
    else {
	k = sin((1.0-w)/2) / sin((1.0+w)/2);
    }
    k2 = sqr(k);
    D = b[0] + k * b[1] - k2 * b[2];
    A[0] = (a[0] - k * a[1] + k2 * a[2]) / D;
    A[1] = (-2*k * a[0] + (k2+1) * a[1] - 2*k * a[2]) / D;
    A[2] = (k2 * a[0] - k * a[1] + a[2]) / D;
    B[0] = -1.0;
    B[1] = (2*k * b[0] + (k2+1) * b[1] - 2*k * b[2]) / D;
    B[2] = (-k2 * b[0] - k * b[1] + b[2]) / D;
    if (_PassbandType == PassbandType_HighPass) {
	A[1] *= -1;
	B[1] *= -1;
    }
}



TKumaButterworthFilter::TKumaButterworthFilter(double CutoffFrequency, int NumberOfPoles, int PassbandType)
: TKumaChebychevFilter(CutoffFrequency, NumberOfPoles, 0, PassbandType)
{
}

TKumaButterworthFilter::~TKumaButterworthFilter()
{
}



TKumaArbitrarySpectrumFilter::TKumaArbitrarySpectrumFilter(TKumaFunction& Spectrum)
{
    _Spectrum = &Spectrum;
}

TKumaArbitrarySpectrumFilter::~TKumaArbitrarySpectrumFilter()
{
}

void TKumaArbitrarySpectrumFilter::ApplyTo(double* SampleList, int Length)
{
    int n = 1, m = 0;
    while (n < Length) {
	n <<= 1;
	m++;
    }

    complex<double>* DataList = new complex<double>[n];
    for (int i = 0; i < Length; i++) {
	DataList[i] = complex<double>(SampleList[i], 0);
    }
    for (int i = Length; i < n; i++) {
	DataList[i] = complex<double>(0, 0);
    }

    complex<double>* SpectrumList = new complex<double>[n];
    for (int i = 1; i < n/2; i++) {
	SpectrumList[i] = (*_Spectrum)((double)i/n);
	SpectrumList[n-i] = SpectrumList[i];
    }
    SpectrumList[0] = (*_Spectrum)(0);
    SpectrumList[n/2] = (*_Spectrum)(0.5);

#if 0
    TKumaFastFourierTransformer().Transform(SpectrumList, n, -1);
    double Sum = 0;
    for (int i = 0; i < n; i++) {
	double X = (double) (i-n/2) / n;
	if (X < 0) {
	    X += 1;
	}
	double Window = 0.42 - 0.5 * cos(2*M_PI*X) + 0.08 * cos(4*M_PI*X);
	SpectrumList[i] *= Window;
	if (fabs(X-0.5) < 0.2) {
	    SpectrumList[i] = 0;
	}
	Sum += SpectrumList[i].real();
    }
    for (int i = 0; i < n; i++) {
	SpectrumList[i] /= Sum * n;
    }
    TKumaFastFourierTransformer().Transform(SpectrumList, n, +1);
#endif

    TKumaFastFourierTransformer().Transform(DataList, n, +1);
    for (int i = 0; i < n; i++) {
	DataList[i] *= SpectrumList[i];
    }
    TKumaFastFourierTransformer().Transform(DataList, n, -1);

    for (int i = 0; i < Length; i++) {
	SampleList[i] = DataList[i].real() / n;
    }

    delete[] DataList;
    delete[] SpectrumList;
}
