/* KumaDigitalFilter.hh */
/* Created by Enomoto Sanshiro on 1 December 2007. */
/* Last updated by Enomoto Sanshiro on 1 December 2007. */


#ifndef __KumaDigitalFilter_hh
#define __KumaDigitalFilter_hh

#include <iostream>
#include <string>
#include <complex>
#include "KumaFunction.hh"


namespace kuma {


class TKumaDigitalFilter {
  public:
    TKumaDigitalFilter(void) {}
    virtual ~TKumaDigitalFilter() {}
    virtual void ApplyTo(double* SampleList, int Length) = 0;
  public:
    enum TPassbandType {
	PassbandType_LowPass,
	PassbandType_HighPass,
	PassbandType_BandPass,
	PassbandType_BandEliminate,
	PassbandType_Others,
	_NumberOfPassbandTypes
    };
};


class TKumaFiniteImpulseResponseFilter: public TKumaDigitalFilter {
  public:
    TKumaFiniteImpulseResponseFilter(void);
    virtual ~TKumaFiniteImpulseResponseFilter();
    virtual void ApplyTo(double* SampleList, int Length);
    virtual void DumpCoefficients(std::ostream& os, const std::string& Header);
  protected:
    void SetCoefficients(double* CoeffA, int LastIndex);
  private:
    int _LastIndex;
    double* _CoeffA;
};


class TKumaArbitraryResponseFilter: public TKumaFiniteImpulseResponseFilter {
  public:
    TKumaArbitraryResponseFilter(TKumaFunction& Response, int ResponseLength);
    TKumaArbitraryResponseFilter(double* Response, int ResponseLength);
    virtual ~TKumaArbitraryResponseFilter();
  protected:
    double* _CoeffA;
    int _ResponseLength;
};


class TKumaMovingAverageFilter: public TKumaFiniteImpulseResponseFilter {
  public:
    TKumaMovingAverageFilter(double CutoffFrequency);
    virtual ~TKumaMovingAverageFilter();
  private:
    double _CutoffFrequency;
    int _WindowLength;
    double* _CoeffA;
};


class TKumaWindowedSincFilter: public TKumaFiniteImpulseResponseFilter {
  public:
    TKumaWindowedSincFilter(double CutoffFrequency, int KernelLength = 0);
    virtual ~TKumaWindowedSincFilter();
  private:
    double _CutoffFrequency;
    int _KernelLength;
    double* _CoeffA;
};


class TKumaRecursiveFilter: public TKumaDigitalFilter {
  public:
    TKumaRecursiveFilter(void);
    virtual ~TKumaRecursiveFilter();
    virtual void ApplyTo(double* SampleList, int Length);
    virtual void DumpCoefficients(std::ostream& os, const std::string& Header);
  protected:
    void SetCoefficients(double* CoeffA, double* CoeffB, int LastIndex);
  private:
    int _LastIndex;
    double* _CoeffA;
    double* _CoeffB;
};


class TKumaSinglePoleFilter: public TKumaRecursiveFilter {
  public:
    TKumaSinglePoleFilter(double CutoffFrequency, int PassbandType = TKumaDigitalFilter::PassbandType_LowPass);
    virtual ~TKumaSinglePoleFilter();
  private:
    double _CoeffA[2], _CoeffB[2];
};


class TKumaChebychevFilter: public TKumaRecursiveFilter {
  public:
    TKumaChebychevFilter(double CutoffFrequency, int NumberOfPoles, double MaxRipple, int PassbandType = TKumaDigitalFilter::PassbandType_LowPass);
    virtual ~TKumaChebychevFilter();
  protected:
    void CalculateCoefficients(void);
    void CalculateCoefficientsFor(int PoleIndex, double A[3], double B[3]);
  private:
    int _PassbandType;
    double _CutoffFrequency;
    int _NumberOfPoles;
    double _MaxRipple;
    double* _CoeffA;
    double* _CoeffB;
};


class TKumaButterworthFilter: public TKumaChebychevFilter {
  public:
    TKumaButterworthFilter(double CutoffFrequency, int NumberOfPoles, int PassbandType = TKumaDigitalFilter::PassbandType_LowPass);
    virtual ~TKumaButterworthFilter();
};


class TKumaArbitrarySpectrumFilter: public TKumaDigitalFilter {
  public:
    TKumaArbitrarySpectrumFilter(TKumaFunction& Spectrum);
    virtual ~TKumaArbitrarySpectrumFilter();
    virtual void ApplyTo(double* SampleList, int Length);
  protected:
    TKumaFunction* _Spectrum;
};


template<class T> inline T sinc(const T& x) {
    if (fabs(x) < 1.0e-6) {
	return 1;
    }
    else {
	return sin(M_PI*x)/(M_PI*x);
    }
}


}

#endif
