/* KumaFunction.hh */
/* Created by Enomoto Sanshiro on 18 January 2001. */
/* Last updated by Enomoto Sanshiro on 26 May 2002. */


#ifndef __KumaFunction_hh__
#define __KumaFunction_hh__


#include <complex>
#include "KumaVector.hh"
#include "KumaPrimitive.hh"

namespace kuma {


template<class T = double> class TKumaFunctionBase {
  public:
    TKumaFunctionBase(void) {}
    virtual ~TKumaFunctionBase() {}
    virtual T ValueOf(const T* InputValueList, int InputListSize) = 0;
    inline T operator() (const T& x) { 
	return ValueOf(&x, 1); 
    }
    inline T operator() (const T& x, const T& y) {
	T InputValueList[2];
	InputValueList[0] = x;
	InputValueList[1] = y;
	return ValueOf(InputValueList, 2); 
    }
    inline T operator() (const T& x, const T& y, const T& z) {
	T InputValueList[3];
	InputValueList[0] = x;
	InputValueList[1] = y;
	InputValueList[2] = z;
	return ValueOf(InputValueList, 3); 
    }
};


typedef TKumaFunctionBase<double> TKumaFunction;
typedef TKumaFunctionBase<std::complex<double> > TKumaComplexFunction;


template<class T = double> class TKumaParameterizedFunctionBase: public TKumaFunctionBase<T> {
  protected:
    TKumaVectorBase<TKumaPrimitiveBase<T> > _ParameterList;
  public:
    TKumaParameterizedFunctionBase(void) {}
    virtual ~TKumaParameterizedFunctionBase(void) {}
    virtual TKumaVectorBase<TKumaPrimitiveBase<T> >& ParameterList(void) {
	return _ParameterList;
    }
    inline TKumaPrimitiveBase<T>& operator[] (int PatameterIndex) {
	return _ParameterList[PatameterIndex];
    }
};


typedef TKumaParameterizedFunctionBase<double> TKumaParameterizedFunction;
typedef TKumaParameterizedFunctionBase<std::complex<double> > TKumaComplexParameterizedFunction;


template<class T = double> class TKumaPolynominalFunctionBase: public TKumaParameterizedFunctionBase<T> {
  public:
    TKumaPolynominalFunctionBase(int MaxOrder) {
	_MaxOrder = MaxOrder;
	for (int i = 0; i <= _MaxOrder; i++) {
	    TKumaParameterizedFunctionBase<T>::_ParameterList.Add(TKumaPrimitiveBase<T>(0));
	}
    }

    TKumaPolynominalFunctionBase(const TKumaVectorBase<TKumaPrimitiveBase<T> >& CoefficientList) {
	_MaxOrder = CoefficientList.size() - 1;
	TKumaParameterizedFunctionBase<T>::_ParameterList = CoefficientList;
    }

    virtual ~TKumaPolynominalFunctionBase() {
    }

    virtual int MaxOrder(void) const {
	return _MaxOrder;
    }

    virtual T ValueOf(const T* InputValueList, int InputListSize) {
	T Term = 1;
	T Result = 0;
	//... the following caluclation should be improved
	for (int i = 0; i <= _MaxOrder; i++) {
	    Result += TKumaParameterizedFunctionBase<T>::_ParameterList[i] * Term;
	    Term *= InputValueList[0];
	}
	return Result;
    }

  protected:
    int _MaxOrder;
};

typedef TKumaPolynominalFunctionBase<double> TKumaPolynominalFunction;
typedef TKumaPolynominalFunctionBase<std::complex<double> > TKumaComplexPolynominalFunction;


}

#endif
