/* KumaMatrix.hh */
/* Created by Enomoto Sanshiro on 18 January 2001 */
/* Last updated by Enomoto Sanshiro on 18 January 2001. */


#ifndef __KumaMatrix_hh__
#define __KumaMatrix_hh__


#include <complex>

namespace kuma {


template<class T = double> class TKumaMatrixBase {
  public:
    inline TKumaMatrixBase(void);
    inline TKumaMatrixBase(int NumberOfRows, int NumberOfColumns);
    inline TKumaMatrixBase(const TKumaMatrixBase<T>& Matrix);
    inline ~TKumaMatrixBase();
    inline TKumaMatrixBase<T>& operator=(const TKumaMatrixBase<T> Matrix);
    inline int NumberOfRows(void) const;
    inline int NumberOfColumns(void) const;
    inline void Resize(int NumberOfRows, int NumberOfColumns);
    inline void Fill(const T& Value);
    inline T* RawArray(void);
    inline const T* RawArray(void) const;
    inline T** RawMatrix(void);
    inline const T** RawMatrix(void) const;
  public:
    inline T* operator[] (int RowIndex);
    inline const T* operator[] (int RowIndex) const;
    inline operator T*();
    inline operator const T*() const;
    inline operator T**();
    inline operator const T**() const;
  protected:
    int _NumberOfRows, _NumberOfColumns;
    int _Capacity;
    T* _RawArray;
    mutable T** _RawMatrix;
};


typedef TKumaMatrixBase<double> TKumaMatrix;
typedef TKumaMatrixBase<std::complex<double> > TKumaComplexMatrix;


template<class T> inline TKumaMatrixBase<T>::TKumaMatrixBase(void)
{
    _Capacity = 0;
    _RawArray = 0;
    _RawMatrix = 0;
    
    _NumberOfRows = 0;
    _NumberOfColumns = 0;
}

template<class T> inline TKumaMatrixBase<T>::TKumaMatrixBase(int NumberOfRows, int NumberOfColumns)
{
    _Capacity = 0;
    _RawArray = 0;
    _RawMatrix = 0;

    Resize(NumberOfRows, NumberOfColumns);
}

template<class T> inline TKumaMatrixBase<T>::TKumaMatrixBase(const TKumaMatrixBase<T>& Matrix)
{
    _Capacity = 0;
    _RawArray = 0;
    _RawMatrix = 0;

    Resize(Matrix._NumberOfRows, Matrix._NumberOfColumns);

    for (unsigned i = 0; i < _NumberOfRows * _NumberOfColumns; i++) {
	_RawArray[i] = Matrix.RawArray[i];
    }
}

template<class T> inline TKumaMatrixBase<T>::~TKumaMatrixBase()
{
    delete[] _RawMatrix;
    delete[] _RawArray;
}

template<class T> inline TKumaMatrixBase<T>& TKumaMatrixBase<T>::operator=(const TKumaMatrixBase<T> Matrix)
{
    Resize(Matrix._NumberOfRows, Matrix._NumberOfColumns);

    for (unsigned i = 0; i < _NumberOfRows * _NumberOfColumns; i++) {
	_RawArray[i] = Matrix.RawArray[i];
    }
}

template<class T> inline int TKumaMatrixBase<T>::NumberOfRows(void) const
{
    return _NumberOfRows;
}

template<class T> inline int TKumaMatrixBase<T>::NumberOfColumns(void) const
{
    return _NumberOfColumns;
}

template<class T> inline void TKumaMatrixBase<T>::Resize(int NumberOfRows, int NumberOfColumns)
{
    _NumberOfRows = NumberOfRows;
    _NumberOfColumns = NumberOfColumns;

    if (_Capacity < _NumberOfRows * _NumberOfColumns) {
	delete[] _RawArray;
	_Capacity = _NumberOfRows * _NumberOfColumns;
	_RawArray = new T[_Capacity];
    }

    delete[] _RawMatrix;
    _RawMatrix = 0;
}

template<class T> inline void TKumaMatrixBase<T>::Fill(const T& Value) 
{
    for (unsigned i = 0; i < _NumberOfRows * _NumberOfColumns; i++) {
	_RawArray[i] = Value;
    }
}

template<class T> inline T* TKumaMatrixBase<T>::RawArray(void)
{
    return _RawArray;
}

template<class T> inline const T* TKumaMatrixBase<T>::RawArray(void) const
{
    return _RawArray;
}

template<class T> inline T** TKumaMatrixBase<T>::RawMatrix(void)
{
    if (_RawMatrix == 0) {
	_RawMatrix = new T* [_NumberOfRows];
	for (int Row = 0; Row < _NumberOfRows; Row++) {
	    _RawMatrix[Row] = _RawArray + Row * _NumberOfColumns;
	}
    }

    return _RawMatrix;
}

template<class T> inline const T** TKumaMatrixBase<T>::RawMatrix(void) const
{
    if (_RawMatrix == 0) {
	_RawMatrix = new T* [_NumberOfRows];
	for (int Row = 0; Row < _NumberOfRows; Row++) {
	    _RawMatrix[Row] = _RawArray + Row * _NumberOfColumns;
	}
    }

    return _RawMatrix;
}

template<class T> inline T* TKumaMatrixBase<T>::operator[] (int RowIndex)
{
    return _RawArray + RowIndex * _NumberOfColumns;
}

template<class T> inline const T* TKumaMatrixBase<T>::operator[] (int RowIndex) const
{
    return _RawArray + RowIndex * _NumberOfColumns;
}

template<class T> inline TKumaMatrixBase<T>::operator T*()
{
    return _RawArray;
}

template<class T> inline TKumaMatrixBase<T>::operator const T*() const
{
    return _RawArray;
}

template<class T> inline TKumaMatrixBase<T>::operator T**()
{
    return RawMatrix();
}

template<class T> inline TKumaMatrixBase<T>::operator const T**() const
{
    return RawMatrix();
}

}

#endif
