/* KumaSpecialFunction.cc */
/* Created by Enomoto Sanshiro on 22 November 2003. */
/* Last updated by Enomoto Sanshiro on 9 December 2009. */


#include <algorithm>
#include <cmath>
#include "KumaSpecialFunction.hh"

using namespace std;
using namespace kuma;


static double IncompleteGammaBySeriesExpansion(double a, double x);
static double IncompleteGammaByContinuedFraction(double a, double x);


double kuma::IncompleteGamma(double a, double x)
{
    if ((a <= 0) || (x < 0)) {
	throw TKumaException("IncompleteGamma()", "invalid parameter");
    }

    if (x == 0) {
	return 0;
    }
    else if (x < (a + 1)) {
	return IncompleteGammaBySeriesExpansion(a, x);
    }
    else {
	return IncompleteGammaByContinuedFraction(a, x);
    }
}


static double IncompleteGammaBySeriesExpansion(double a, double x)
{
    static const double EPS = 3.0e-7;
    int MaxNumberOfTerms = max(128, 2 * (int) sqrt(a));

    double Term = 1.0/a;
    double Sum = Term;
    for (int i = 1; i <= MaxNumberOfTerms; i++) {
	Term *= x / (a + i);
	Sum += Term;

	if (fabs(Term) < fabs(Sum) * EPS) {
	    return Sum * exp(-x + a * log(x) - LogGamma(a));
	}
    }

    throw TKumaException(
	"IncompleteGammaBySeriesExpansion()", 
	"series expansion not converged"
    );
}


static double IncompleteGammaByContinuedFraction(double a, double x)
{
    static const double EPS = 3.0e-7;
    int MaxDepth = max(128, 2 * (int) sqrt(a));

    double Cf, CfPrev = 0.0;

    double A0 = 0, B0 = 1;
    double A1 = 1, B1 = 0;

    double ai = 1;
    double bi = x;
    A1 = bi * A0 + ai * A1;
    B1 = bi * B0 + ai * B1;

    for (int i = 1; i <= MaxDepth; i++) {
	ai = i - a;
	bi = 1;
	A0 = bi * A1 + ai * A0;
	B0 = bi * B1 + ai * B0;

	ai = i;
	bi = x;
	A1 = bi * A0 + ai * A1;
	B1 = bi * B0 + ai * B1;

	if (B1 == 0) {
	    continue;
	}
	A0 /= B1;
	B0 /= B1;
	A1 /= B1;
	B1 = 1;

	CfPrev = Cf;
	Cf = A1;
	if (fabs(Cf - CfPrev) < EPS * fabs(Cf)) {
	    return 1.0 - exp(-x + a * log(x) - LogGamma(a)) * Cf;
	}
    }
    
    throw TKumaException(
	"IncompleteGammaByContinuedFraction()", 
	"continued fraction not converged"
    );
}


double kuma::IncompleteBeta(double a, double b, double x)
{
    if (x < 0) {
	throw TKumaException("IncompleteBeta()", "invalid parameter");
    }

    if (x == 0) {
	return 0;
    }
    if (x > (a + 1.0) / (a + b + 2.0)) {
	return 1.0 - IncompleteBeta(b, a, 1-x);
    }

    // Continued Fraction //
    double Cf;
    {
	static const double EPS = 3.0e-7;
	int MaxDepth = max(128, 5 * (int) sqrt(max(a, b)));

	double CfPrev = 0;

	double A0 = 0, B0 = 1;
	double A1 = 1, B1 = 0;
	
	double ai = 1;
	double bi = 1;
	A1 = bi * A0 + ai * A1;
	B1 = bi * B0 + ai * B1;

	ai = -(a+b)*x/(a+1);
	A0 = bi * A1 + ai * A0;
	B0 = bi * B1 + ai * B0;

	int m;
	for (m = 1; m < MaxDepth; m++) {
	    ai = m * (b - m) * x / ((a - 1 + 2*m) * (a + 2*m));
	    A1 = bi * A0 + ai * A1;
	    B1 = bi * B0 + ai * B1;

	    ai = -(a + m) * (a + b + m) * x / ((a + 2*m) * (a + 2*m + 1));
	    A0 = bi * A1 + ai * A0;
	    B0 = bi * B1 + ai * B0;

	    if (B0 == 0) {
		continue;
	    }
	    A1 /= B0;
	    B1 /= B0;
	    A0 /= B0;
	    B0 = 1;

	    CfPrev = Cf;
	    Cf = A0;
	    if (fabs(Cf - CfPrev) < EPS * fabs(Cf)) {
		break;
	    }
	}
	if (m == MaxDepth) {
	    throw TKumaException(
		"IncompleteBeta()",
		"continued fraction not converged"
	    );
	}
    }

    double ReciprocalLogBeta = LogGamma(a+b) - LogGamma(a) - LogGamma(b);
    
    return Cf * exp(ReciprocalLogBeta + a * log(x) + b * log(1-x)) / a;
}


double kuma::ErrorFunction(double x)
{
    if (x >= 0) {
	return IncompleteGamma(0.5, sqr(x));
    }
    else {
	return -IncompleteGamma(0.5, sqr(x));
    }
}


double kuma::Legendre(int l, double x)
{
    if (l == 0) {
	return 1;
    }
    if (l == 1) {
	return x;
    }

    double P_n2 = 1, P_n1 = x, P_n;
    for (int n = 2; n <= l; n++) {
	P_n = (x * (2*n-1) * P_n1 - (n-1) * P_n2) / n;
	P_n2 = P_n1;
	P_n1 = P_n;
    }

    return P_n;
}


double kuma::AssociatedLegendre(int l, int m, double x)
{
#if 1
    if ((m < 0) || (m > l) || (x < -1) || (x > 1)) {
	throw TKumaException("AssociatedLegendre()", "invalid parameter");
    }
#endif

    double P_m_m = 1.0;
    if (m > 0) {
	double SqarerootOfOneMinusSquaredX = sqrt((1.0 - x) * (1.0 + x));
	double Factor = 1.0;
	for (int i = 0; i < m; i++) {
	    P_m_m *= -Factor * SqarerootOfOneMinusSquaredX;
	    Factor += 2.0;
	}
    }

    if (l == m) {
	return P_m_m;
    }

    double P_m1_m = x * (2*m + 1) * P_m_m;

    if (l == (m + 1)) {
	return P_m1_m;
    }

    double P_k_m;
    for (int k = m+2; k <= l; k++) {
	P_k_m = (x * (2*k-1) * P_m1_m - (k+m-1) * P_m_m) / (k-m);
	P_m_m = P_m1_m;
	P_m1_m = P_k_m;
    }

    return P_k_m;
}


complex<double> kuma::SphericalHarmonics(int l, int m, double theta, double phi)
{
    if (m < 0) {
	double Sign = (m % 2 == 0) ? 1 : -1;
	return Sign * conj(SphericalHarmonics(l, -m, theta, phi));
    }

    double SquaredFactor = (2*l + 1) / (4 * M_PI);
    for (int k = (l-m) + 1; k <= (l+m); k++) {
	SquaredFactor /= k;
    }
    
    double Norm = sqrt(SquaredFactor) * AssociatedLegendre(l, m, cos(theta));

    return Norm * exp(complex<double>(0, m*phi));
}
