/* KumaSpecialFunction.hh */
/* Created by Enomoto Sanshiro on 22 November 2003. */
/* Last updated by Enomoto Sanshiro on 18 July 2004. */


#ifndef __KumaSpecialFunction_hh__
#define __KumaSpecialFunction_hh__


#include <cmath>
#include <complex>
#include "KumaDefs.hh"

namespace kuma {


template<class T> T LogGamma(const T& x);
template<class T> T Beta(const T& z, const T& w);
double IncompleteGamma(double a, double x);
double IncompleteBeta(double a, double b, double x);

double ErrorFunction(double x);

double Legendre(int l, double x);
double AssociatedLegendre(int l, int m, double x);
std::complex<double> SphericalHarmonics(int l, int m, double theta, double phi);


template<class T> T LogGamma(const T& x) 
{
    static const double Pi = 3.141592653589793238;
    static const double LogPi = std::log(Pi);
    static const double SqrtOf2Pi = std::sqrt(2*Pi);

    if (std::real(std::complex<double>(x)) < 0) {
        return (
	    LogPi + std::log(1.0-x) - 
	    LogGamma(2.0-x) - std::log(std::sin(Pi*(1.0-x)))
	);
    }

    static const double CoefficientList[] = {
        1.000000000190015,
        76.18009172947146, -86.50532032941677, 24.01409824083091, 
        -1.231739572450155, 0.1208650973866179e-2, -0.5395239384953e-5
    };

    T Series = CoefficientList[0];
    for (unsigned i = 1; i < sizeof(CoefficientList)/sizeof(double); i++) {
        Series += CoefficientList[i] / (x + (double) i);
    }

    return (x+0.5) * log(x+5.5) - (x+5.5) + std::log(SqrtOf2Pi * Series/x);
}


template<class T> T Beta(const T& z, const T& w)
{
    return std::exp(LogGamma(z) + LogGamma(w) - LogGamma(z + w));
}


}

#endif
