/* MushCheckSum.cc */
/* Created by Enomoto Sanshiro on 15 February 2002. */
/* Last updated by Enomoto Sanshiro on 15 February 2002. */


#include <iostream>
#include <iomanip>
#include "MushConfig.hh"
#include "MushCheckSum.hh"


// The generating polynomial for POSIX.2 CRC32 is
//
//           32   26   23   22   16   12   11   10   8   7   5   4   2   1
//   P(x) = x  + x  + x  + x  + x  + x  + x  + x  + x + x + x + x + x + x + 1
//
// The following is a bit array of this polynomial (except the 32th bit):
static mush::UInt32 Crc32GeneratingPolynominal = 0x04c11db7;

mush::UInt32 TMushCrc32CheckSum::_CrcTable[256];
bool TMushCrc32CheckSum::_IsCrcTableComputed = false;


TMushCrc32CheckSum::TMushCrc32CheckSum(void)
{
    if (! _IsCrcTableComputed) {
        mush::UInt32 r[8];
	r[0] = Crc32GeneratingPolynominal;
	for (int Bit = 1; Bit < 8; Bit++) {
	    if (r[Bit - 1] & 0x80000000) {
		r[Bit] = (r[Bit - 1] << 1) ^ r[0];
	    }
	    else {
		r[Bit] = r[Bit - 1] << 1;
	    }
	}
	
	_CrcTable[0] = 0;
	for (int i = 1; i < 256; i++) {
	    mush::UInt32 rem = 0;
	    for (int Bit = 0; Bit < 8; Bit++) {
		if (i & (0x0001 << Bit)) {
		    rem = rem ^ r[Bit];
		}
	    }
	    
	    _CrcTable[i] = rem & 0xffffffff;
	}

#if 0
	for (int i = 0; i < 256; i++) {
	    cout << dec << i << " ";
	    cout << hex << setw(8) << setfill('0') << _CrcTable[i] << endl;
	}
#endif

	_IsCrcTableComputed = true;
    }

    _DataLength = 0;
    _CrcValue = 0xffffffff;
}

TMushCrc32CheckSum::~TMushCrc32CheckSum()
{
}

void TMushCrc32CheckSum::ReadData(const char* Data, unsigned long Size)
{
    int TableIndex;

    _CrcValue = ~_CrcValue & 0xffffffff;
    for (unsigned long i = 0; i < Size; i++) {
	TableIndex = (_CrcValue >> 24) ^ (Data[i] & 0xff);
	_CrcValue = (_CrcValue << 8) ^ _CrcTable[TableIndex];
    }
    _CrcValue = ~_CrcValue & 0xffffffff;

    _DataLength += Size;
}

unsigned long TMushCrc32CheckSum::DataLength(void)
{
    return _DataLength;
}

unsigned TMushCrc32CheckSum::ResultLength(void)
{
    return 4;  // 32bit
}

unsigned char TMushCrc32CheckSum::ByteValueAt(unsigned Index)
{
    if ((Index >= 0) && (Index < 4)) {
	return (_CrcValue >> (8 * Index)) & 0x00ff;
    }
    else {
	return 0;
    }
}
