/* MushDataCompressor.cc */
/* Created by Enomoto Sanshiro on 29 January 2002. */
/* Last updated by Enomoto Sanshiro on 29 January 2002. */


#include "MushDefs.hh"
#include "MushDataCompressor.hh"

using namespace std;


TMushDataCompressor::TMushDataCompressor(void)
{
}

TMushDataCompressor::~TMushDataCompressor()
{
}



#if USE_ZLIB

TMushZlibDataCompressor::TMushZlibDataCompressor(int CompressionLevel)
{
    _CompressionLevel = CompressionLevel;

    _DeflateZ = 0;
    _InflateZ = 0;

    _IsLastData = false;
    _IsLastDataProcessed = false;
}

TMushZlibDataCompressor::~TMushZlibDataCompressor()
{
    if (_DeflateZ) {
	deflateEnd(_DeflateZ);
	delete _DeflateZ;
    }
    if (_InflateZ) {
	inflateEnd(_InflateZ);
	delete _InflateZ;
    }
}

void TMushZlibDataCompressor::SetCompressInput(void* InputData, size_t InputDataSize, bool IsLast)
{
    if (_DeflateZ == 0) {
	_DeflateZ = new z_stream;

	_DeflateZ->zalloc = Z_NULL;
	_DeflateZ->zfree = Z_NULL;
	_DeflateZ->opaque = Z_NULL;

	_DeflateZ->next_in = Z_NULL;
	_DeflateZ->avail_in = 0;

	if (deflateInit(_DeflateZ, _CompressionLevel) != Z_OK) {
	    throw TSystemCallException(
		"TMushZlibDataCompressor::SetCompressInput()",
		_DeflateZ->msg
	    );
	}
    }

    if (_DeflateZ->avail_in > 0) {
	throw TSystemCallException(
	    "TMushZlibDataCompressor::SetCompressInput()",
	    "unprocessed data is remaining in the zlib buffer"
	);
    }
    if (_IsLastDataProcessed) {
	throw TSystemCallException(
	    "TMushZlibDataCompressor::SetCompressInput()",
	    "extra data after 'last data'"
	);
    }

    _DeflateZ->next_in = (Bytef*) InputData;
    _DeflateZ->avail_in = InputDataSize;

    _IsLastData = IsLast;
}

void TMushZlibDataCompressor::SetDecompressInput(void* InputData, size_t InputDataSize, bool IsLast)
{
    if (_InflateZ == 0) {
	_InflateZ = new z_stream;

	_InflateZ->zalloc = Z_NULL;
	_InflateZ->zfree = Z_NULL;
	_InflateZ->opaque = Z_NULL;

	_InflateZ->next_in = Z_NULL;
	_InflateZ->avail_in = 0;

	if (inflateInit(_InflateZ) != Z_OK) {
	    throw TSystemCallException(
		"TMushZlibDataCompressor::SetDecompressInput()",
		_InflateZ->msg
	    );
	}
    }

    if (_InflateZ->avail_in > 0) {
	throw TSystemCallException(
	    "TMushZlibDataCompressor::SetDecompressInput()",
	    "unprocessed data is remaining in the zlib buffer"
	);
    }
    if (_IsLastDataProcessed) {
	throw TSystemCallException(
	    "TMushZlibDataCompressor::SetDecompressInput()",
	    "extra data after 'last data'"
	);
    }

    _InflateZ->next_in = (Bytef*) InputData;
    _InflateZ->avail_in = InputDataSize;

    _IsLastData = IsLast;
}

int TMushZlibDataCompressor::Compress(void* Buffer, size_t BufferSize)
{
    if (
	((_DeflateZ->avail_in == 0) && (! _IsLastData)) || 
	_IsLastDataProcessed
    ){
	return 0;
    }

    _DeflateZ->next_out = (Bytef*) Buffer;
    _DeflateZ->avail_out = BufferSize;

    int Flag;
    if (! _IsLastData) {
	Flag = Z_NO_FLUSH;
    }
    else {
	Flag = Z_FINISH;
    }

    int Status = deflate(_DeflateZ, Flag);
    if (_IsLastData && (Status == Z_STREAM_END)) {
	_IsLastDataProcessed = true;
    }
    else if (Status != Z_OK) {
	throw TSystemCallException(
	    "TMushZlibDataCompressor::Compress()",
	    "deflate(): " + string(_DeflateZ->msg)
	);
    }

    return BufferSize - _DeflateZ->avail_out;
}

int TMushZlibDataCompressor::Decompress(void* Buffer, size_t BufferSize)
{
    if ((_InflateZ->avail_in == 0) || _IsLastDataProcessed) {
	return 0;
    }

    _InflateZ->next_out = (Bytef*) Buffer;
    _InflateZ->avail_out = BufferSize;

    int Status = inflate(_InflateZ, Z_NO_FLUSH);
    if (Status == Z_STREAM_END) {
	_IsLastDataProcessed = true;
    }
    else if (Status != Z_OK) {
	throw TSystemCallException(
	    "TMushZlibDataCompressor::Decompress()",
	    "inflate(): " + string(_InflateZ->msg)
	);
    }

    return BufferSize - _InflateZ->avail_out;
}


#else

TMushZlibDataCompressor::TMushZlibDataCompressor(int CompressionLevel) 
{
    throw TSystemCallException(
	"TMushZlibDataCompressor::TMushZlibDataCompressor()", 
	"function not available ('zlib' must be installed)"
    );
}

TMushZlibDataCompressor::~TMushZlibDataCompressor() 
{
}

void TMushZlibDataCompressor::SetCompressInput(void* InputData, size_t InputDataSize, bool IsLast) 
{
}

void TMushZlibDataCompressor::SetDecompressInput(void* InputData, size_t InputDataSize, bool IsLast) 
{
}

int TMushZlibDataCompressor::Compress(void* Buffer, size_t BufferSize) 
{ 
    return 0; 
}

int TMushZlibDataCompressor::Decompress(void* Buffer, size_t BufferSize) 
{ 
    return 0; 
}

#endif
