/* MushFile.cc */
/* Created by Enomoto Sanshiro on 3 March 1998. */
/* Last updated by Enomoto Sanshiro on 30 April 2002. */


#include <string>
#include <cerrno>
#include <fcntl.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/stat.h>
#include "MushTimer.hh"
#include "MushFile.hh"


using namespace std;
using namespace mush;


TMushFile::TMushFile(const string& FileName)
{
    _Name = FileName;
    _FileDescriptor = -1;
    _MyFileDescriptor = -1;
}

TMushFile::TMushFile(int FileDescriptor)
{
    _Name = "";
    _FileDescriptor = FileDescriptor;
    _MyFileDescriptor = -1;
}

TMushFile::~TMushFile()
{
    if (_FileDescriptor >= 0) {
	TMushFile::Close();
    }
}

void TMushFile::OpenReadMode(void) throw(TSystemCallException)
{
    Open(O_RDONLY, 0);
}

void TMushFile::OpenWriteMode(unsigned AccessMode, bool IsExclusive) throw(TSystemCallException)
{
    if (IsExclusive) {
	Open(O_WRONLY | O_CREAT | O_TRUNC | O_EXCL, AccessMode);
    }
    else {
	Open(O_WRONLY | O_CREAT | O_TRUNC, AccessMode);
    }
}

void TMushFile::OpenAppendMode(unsigned AccessMode) throw(TSystemCallException)
{
    Open(O_RDWR | O_APPEND | O_CREAT, AccessMode);
}

void TMushFile::OpenReadWriteMode(unsigned AccessMode) throw(TSystemCallException)
{
    Open(O_RDWR | O_CREAT, AccessMode);
}

void TMushFile::Open(int OpenFlag, unsigned AccessMode) throw(TSystemCallException)
{
    if (_FileDescriptor >= 0) {
	return;
    }

    _MyFileDescriptor = open(_Name.c_str(), OpenFlag, AccessMode);
    if (_MyFileDescriptor < 0) {
	throw TSystemCallException("TMushFile::Open(): open(2): " + _Name);
    }

    _FileDescriptor = _MyFileDescriptor;
}

void TMushFile::Close(void)
{
    if (_MyFileDescriptor < 0) {
	return;
    }

    close(_MyFileDescriptor);
    _MyFileDescriptor = -1;
}

int TMushFile::Read(void *Buffer, size_t Length) throw(TSystemCallException)
{
    int ReadLength = read(_FileDescriptor, Buffer, Length);

    if (ReadLength < 0) {
	if (errno != EINTR) {
	    throw TSystemCallException("TMushFile::Read(): read(2)");
	}
	else {
	    ReadLength = -1;
	}
    }

    return ReadLength;
}

int TMushFile::Write(void *Buffer, size_t Length) throw(TSystemCallException)
{
    int WrittenLength = write(_FileDescriptor, Buffer, Length);

    if (WrittenLength < 0) {
	if (errno != EINTR) {
	    throw TSystemCallException("TMushFile::Write(): write(2)");
	}
	else {
	    WrittenLength = -1;
	}
    }

    return WrittenLength;
}

Int64 TMushFile::Position(void) throw(TSystemCallException)
{
    return lseek(_FileDescriptor, 0, SEEK_CUR);
}

void TMushFile::SeekTo(Int64 Offset) throw(TSystemCallException)
{
    if (lseek(_FileDescriptor, Offset, SEEK_SET) < 0) {
	throw TSystemCallException("TMushFile::Seek(): lseek(2)");	
    }
}

void TMushFile::SeekBy(Int64 Offset) throw(TSystemCallException)
{
    if (lseek(_FileDescriptor, Offset, SEEK_CUR) < 0) {
	throw TSystemCallException("TMushFile::Seek(): lseek(2)");	
    }
}

int TMushFile::ReadBlock(void *Buffer, size_t Length, bool IsPatient) throw(TSystemCallException)
{
    char* DataPtr = (char *) Buffer;
    char* EndPtr = DataPtr + Length;
    int RemainingLength, ThisLength;

    while ((RemainingLength = EndPtr - DataPtr) > 0) {
	ThisLength = Read(DataPtr, RemainingLength);
	if (ThisLength > 0) {
	    DataPtr += ThisLength;
	}
	else if (ThisLength == 0) {
	    // end of file //
	    if (IsPatient) {
		TMushRealTimeTimer(0, 10000).Suspend(); // 10ms sleep
		continue;
	    }
	    else {
		break;
	    }
	}
	else {
	    // interrupted etc. //
	    continue;
	}
    }

    return Length - RemainingLength;
}

int TMushFile::WriteBlock(void *Buffer, size_t Length) throw(TSystemCallException)
{
    char* DataPtr = (char *) Buffer;
    char* EndPtr = DataPtr + Length;
    int RemainingLength, ThisLength;

    while ((RemainingLength = EndPtr - DataPtr) > 0) {
	ThisLength = Write(DataPtr, RemainingLength);
	if (ThisLength > 0) {
	    DataPtr += ThisLength;
	}
	else if (ThisLength == 0) {
	    throw TSystemCallException(
		"TMushFile::WriteBlock()", "unable to write file"
	    );
	}
	else {
	    // interrupted etc. //
	    continue;
	}
    }

    return Length - RemainingLength;
}

void TMushFile::Unlink(void) throw(TSystemCallException)
{
    if (unlink(_Name.c_str()) < 0) {
	throw TSystemCallException("TMushFile::Unlink(): unlink(2)");
    }
}

string TMushFile::Name(void) const
{
    return _Name;
}

long TMushFile::Size(void) const throw(TSystemCallException)
{
    struct stat FileStat;
    if (stat(_Name.c_str(), &FileStat) < 0) {
	throw TSystemCallException("TMushFile::Size(): stat(2)");
    }

    return FileStat.st_size;
}

bool TMushFile::IsOpened(void) const
{
    return _FileDescriptor >= 0;
}

int TMushFile::FileDescriptor(void) const
{
    return _FileDescriptor;
}



TMushLockFile::TMushLockFile(string Name)
: _File(Name)
{
    _IsLocked = false;
}

TMushLockFile::~TMushLockFile()
{
    if (_IsLocked) {
	Unlock();
    }
}

string TMushLockFile::Name(void) const
{
    return _File.Name();
}

bool TMushLockFile::Lock(unsigned AccessMode)
{
    if (_IsLocked) {
	return false;
    }

    unsigned OldUMask = umask(0);

    _IsLocked = true;
    try {
	bool IsExclusive;
	_File.OpenWriteMode(AccessMode, IsExclusive = true);
    }
    catch (TSystemCallException &e) {
	_IsLocked = false;
    }

    umask(OldUMask);

    return _IsLocked;
}

void TMushLockFile::Unlock(void)
{
    if (_IsLocked) {
	_File.Close();
	_File.Unlink();
    }
}
