/* MushMisc.cc */
/* Created by Enomoto Sanshiro on 28 May 1998. */
/* Last updated by Enomoto Sanshiro on 24 September 1998. */


#include <iostream>
#include <sstream>
#include <string>
#include <map>
#include <cstdlib>
#include <cstring>
#include <unistd.h>
#include <sys/types.h>
#include <sys/utsname.h>
#include <sys/time.h>
#include <time.h>
#include <pwd.h>
#include <grp.h>
#include "MushMisc.hh"


using namespace std;


TMushDateTime::TMushDateTime(TTimeZone TimeZone)
{
    _TimeZone = TimeZone;
    Update();
}

TMushDateTime::TMushDateTime(long TimeValue, TTimeZone TimeZone)
{
    _TimeZone = TimeZone;
    Set(TimeValue);
}

TMushDateTime::TMushDateTime(int Year, int Month, int Day, int Hour, int Min, int Sec, TTimeZone TimeZone)
{
    _TimeZone = TimeZone;
    Set(Year, Month, Day, Hour, Min, Sec);
}

TMushDateTime& TMushDateTime::Update(void)
{
    gettimeofday(&_TimeValue, 0);
    if (_TimeZone == TimeZone_Local) {
	_TimeStruct = *(localtime((time_t *) &_TimeValue.tv_sec));
    }
    else {
	_TimeStruct = *(gmtime((time_t *) &_TimeValue.tv_sec));
    }

    return *this;
}

TMushDateTime& TMushDateTime::Set(long TimeValue)
{    
    _TimeValue.tv_sec = TimeValue;
    _TimeValue.tv_usec = 0;

    if (_TimeZone == TimeZone_Local) {
	_TimeStruct = *(localtime((time_t *) &_TimeValue.tv_sec));
    }
    else {
	_TimeStruct = *(gmtime((time_t *) &_TimeValue.tv_sec));
    }

    return *this;
}

TMushDateTime& TMushDateTime::Set(int Year, int Month, int Day, int Hour, int Min, int Sec)
{
    _TimeStruct.tm_year = Year-1900; // Year: A.C.
    _TimeStruct.tm_mon = Month - 1;  // Month: from 1 to 12
    _TimeStruct.tm_mday = Day;       // Day: from 1 to 31
    _TimeStruct.tm_hour = Hour;      // Hour: from 0 to 23
    _TimeStruct.tm_min = Min;        // Min: from 0 to 59
    _TimeStruct.tm_sec = Sec;        // Sec: from 0 to 59
    _TimeStruct.tm_isdst = 0;

    _TimeValue.tv_sec = mktime(&_TimeStruct);
    _TimeValue.tv_usec = 0;

    if (_TimeZone == TimeZone_Local) {
	_TimeStruct = *(localtime((time_t *) &_TimeValue.tv_sec));
    }
    else {
	_TimeStruct = *(gmtime((time_t *) &_TimeValue.tv_sec));
    }

    return *this;
}

long TMushDateTime::AsLong(void)
{
    return _TimeValue.tv_sec;
}

long TMushDateTime::PassedTime(void)
{
    static struct timeval CurrentTimeValue;
    gettimeofday(&CurrentTimeValue, 0);

    return CurrentTimeValue.tv_sec - _TimeValue.tv_sec;
}

long TMushDateTime::PassedTimeUSec(void)
{
    static struct timeval CurrentTimeValue;
    gettimeofday(&CurrentTimeValue, 0);

    long PassedSec = CurrentTimeValue.tv_sec - _TimeValue.tv_sec;
    long PassedUSec = CurrentTimeValue.tv_usec - _TimeValue.tv_usec;

    return (1000000 * PassedSec + PassedUSec);
}

string TMushDateTime::AsString(string TimeFormat)
{
    if (TimeFormat.empty()) {
	TimeFormat = "%d %b %Y %H:%M:%S %Z";
    }

    char Buffer[256];
    strftime(Buffer, sizeof(Buffer), TimeFormat.c_str(), &_TimeStruct);

    return Buffer;
}

long TMushDateTime::USec(void)
{
    return _TimeValue.tv_usec;
}

int TMushDateTime::Sec(void)
{
    return _TimeStruct.tm_sec;
}

int TMushDateTime::Min(void)
{
    return _TimeStruct.tm_min;
}

int TMushDateTime::Hour(void)
{
    return _TimeStruct.tm_hour;
}

int TMushDateTime::Day(void)
{
    return _TimeStruct.tm_mday;
}

int TMushDateTime::Month(void)
{
    return 1 + _TimeStruct.tm_mon;   // 1: Jan, 2: Feb, ...
}

int TMushDateTime::Year(void)
{
    return 1900 + _TimeStruct.tm_year;
}

int TMushDateTime::DayOfWeek(void)
{
    return _TimeStruct.tm_wday;   // 0: Sun, 1: Mon, ...
}

int TMushDateTime::SecInDay(void)
{
    return (_TimeStruct.tm_hour * 60 + _TimeStruct.tm_min) * 60 + _TimeStruct.tm_sec;
}

int TMushDateTime::DayInYear(void)
{
    return _TimeStruct.tm_yday;  // 0: 1st Jan, 1: 2nd Jan, ...
}



TMushUser::TMushUser(void)
{
    _UserId = getuid();
}

TMushUser::TMushUser(long UserId) throw(TSystemCallException)
{
    struct passwd *PassWordEntry = getpwuid(UserId);
    if (PassWordEntry == 0) { 
	throw TSystemCallException(
            "TMushUser::TMushUser(): getpwuid(3C)",
            "No such user."
        );
    }

    _UserId = PassWordEntry->pw_uid;
}

TMushUser::TMushUser(const string& LoginName) throw(TSystemCallException)
{
    struct passwd *PassWordEntry = getpwnam(LoginName.c_str());
    if (PassWordEntry == 0) { 
	throw TSystemCallException(
            "TMushUser::TMushUser(): getpwnam(3C)",
            string("No such user: ") + LoginName
        );
    }

    _UserId = PassWordEntry->pw_uid;
}

long TMushUser::Id(void)
{
    return _UserId;
}

string TMushUser::Name(void)
{
    return getpwuid(_UserId)->pw_name;
}

long TMushUser::GroupId(void)
{
    return getpwuid(_UserId)->pw_gid;
}

string TMushUser::GroupName(void)
{
    return getgrgid(getpwuid(_UserId)->pw_gid)->gr_name;
}

string TMushUser::HomeDirectory(void)
{
    return getpwuid(_UserId)->pw_dir;
}

string TMushUser::Shell(void)
{
    return getpwuid(_UserId)->pw_shell;
}
    


TMushGroup::TMushGroup(void)
{
    _GroupId = getgid();
}

TMushGroup::TMushGroup(long GroupId) throw(TSystemCallException)
{
    _GroupId = GroupId;
}

TMushGroup::TMushGroup(const string& GroupName) throw(TSystemCallException)
{
    struct group* GroupEntry = getgrnam(GroupName.c_str());
    if (GroupEntry == 0) {
	throw TSystemCallException(
            "TMushGroup::TMushGroup(): getgrnam(3C)",
            "No such group: " + GroupName
        );
    }

    _GroupId = GroupEntry->gr_gid;
}

long TMushGroup::Id(void)
{
    return _GroupId;
}

string TMushGroup::Name(void)
{
    return getgrgid(_GroupId)->gr_name;
}



static struct utsname UtsName;
string TMushLocalHost::SystemName(void)
{
    uname(&UtsName);
    return UtsName.sysname;
}

string TMushLocalHost::SystemRelease(void)
{
    uname(&UtsName);
    return UtsName.release;
}

string TMushLocalHost::SystemVersion(void)
{
    uname(&UtsName);
    return UtsName.version;
}

string TMushLocalHost::HostName(void)
{
    uname(&UtsName);
    return UtsName.nodename;
}

string TMushLocalHost::Machine(void)
{
    uname(&UtsName);
    return UtsName.machine;
}


bool TMushCpu::IsLittleEndian(void)
{
    static const int IntValue = 1;
    return ((char*) &IntValue)[0] == 1;
}



TMushEnvironmentVariable::TMushEnvironmentVariable(const string& Name)
{
    _Name = Name;
    _Value = Get(Name);
}

TMushEnvironmentVariable::~TMushEnvironmentVariable()
{
}

string TMushEnvironmentVariable::AsString(void) throw(TSystemCallException)
{
    if (_Value.empty()) {
	throw TSystemCallException(
            "TMushEnvironmentVariable::AsString()",
            string("undefined variable: ") + _Name
        );
    }

    return _Value;
}

long TMushEnvironmentVariable::AsLong(void) throw(TSystemCallException)
{
    if (_Value.empty()) {
	throw TSystemCallException(
            "TMushEnvironmentVariable::AsLong()",
            string("undefined variable: ") + _Name
        );
    }

    istringstream ValueStream(_Value);
    long LongValue;
    if (! (ValueStream >> LongValue)) {
	throw TSystemCallException(
            "TMushEnvironmentVariable::AsLong()",
            string("variable(") + _Name + string(") is not long: ") + _Value
        );
    }

    return LongValue;
}

double TMushEnvironmentVariable::AsDouble(void) throw(TSystemCallException)
{
    if (_Value.empty()) {
	throw TSystemCallException(
            "TMushEnvironmentVariable::AsDouble()",
            string("undefined variable: ") + _Name
        );
    }

    istringstream ValueStream(_Value);
    double DoubleValue;
    if (! (ValueStream >> DoubleValue)) {
	throw TSystemCallException(
            "TMushEnvironmentVariable::AsDouble()",
            string("variable(") + _Name + string(") is not double: ") + _Value
        );
    }

    return DoubleValue;
}

TMushEnvironmentVariable::operator const char* ()
{
    return _Value.c_str();
}

TMushEnvironmentVariable::operator const string& ()
{
    return _Value;
}

string TMushEnvironmentVariable::Get(const string& Name)
{
    const char* Value = getenv(Name.c_str());
    if (Value == 0) {
	return "";
    }
    else {
	return Value;
    }
}

void TMushEnvironmentVariable::Set(const string& Name, const string& Value)
{
    string Command = (Name + '=' + Value);
    putenv(strdup((char*) Command.c_str()));
}

bool TMushEnvironmentVariable::IsDefined(const string& Name)
{
    return (getenv(Name.c_str()) != 0);
}

void TMushEnvironmentVariable::LoadTable(map<string, string>& Table)
{
    extern char** environ;

    for (char** Entry = environ; *Entry != NULL; Entry++) {
	string EntryString = *Entry, Name, Value;
	size_t NameLength = EntryString.find_first_of('=');
	Name = EntryString.substr(0, NameLength);
	if (NameLength != string::npos) {
	    Value = EntryString.substr(NameLength + 1, EntryString.size());
	}

	Table[Name] = Value;
    }
}

