/* MushSocket.cc */
/* Created by Enomoto Sanshiro on 21 January 1998. */
/* Last updated by Enomoto Sanshiro on 19 November 1999. */


#include <iostream>
#include <string>
#include <cerrno>
#include <unistd.h>
#include <sys/types.h>
#include <sys/utsname.h>

#include <sys/socket.h>
#include <sys/un.h>

#include "MushSocket.hh"


using namespace std;


TMushSocket::TMushSocket(void)
{
    _SocketStatus = ssInitial;
}

TMushSocket::~TMushSocket() 
{
}

int TMushSocket::Send(void *Buffer, size_t Length) throw(TSystemCallException)
{
    int SentLength = send(_ConnectSocket, (char *) Buffer, Length, 0);

    if (SentLength < 0) {
	if (errno != EINTR) {
	    throw TSystemCallException(
		"TMushNetworkSocket::Send(): send(3N)"
	    );
	}
	else {
	    SentLength = -1;
	}
    }

    return SentLength;
}

int TMushSocket::Receive(void *Buffer, size_t Length) throw(TSystemCallException)
{
    int ReceivedLength = recv(_ConnectSocket, (char *) Buffer, Length, 0);

    if (ReceivedLength < 0) {
	if (errno != EINTR) {
	    throw TSystemCallException(
                "TMushNetworkSocket::Receive(): recv(3N)"
            );
	}
	else {
	    ReceivedLength = -1;
	}
    }

    return ReceivedLength;
}

TMushSocket::TSocketStatus TMushSocket::Status(void) const
{
    return _SocketStatus;
}

int TMushSocket::FileDescriptor(void) const
{
    return _ConnectSocket;
}


TMushServerSocket::TMushServerSocket(void)
{
}

TMushServerSocket::~TMushServerSocket()
{
    Close();
}

void TMushServerSocket::EnableLocalAddressReuse(void) throw(TSystemCallException)
{
    // allow the local address to be reused when the server is restarted
    // before the TIME_WAIT state finishs. For details, refer to:
    //
    //   Vic Metcalfe, Andrew Gierth et.al.,
    //   Programming UNIX Sockets in C - Frequently Asked Questions
    //   Section 2.7: Please explain the TIME_WAIT state. 

    if (_SocketStatus == ssInitial) {
	Create();
    }

    if (_SocketStatus != ssCreated) {
	throw TSystemCallException(
	    "TMushServerSocket::EnableLocalAddressReuse(): ",
	    "bad socket state to change socket options"
	);
    }

    int Option = 1;
    int Status = setsockopt(
	_ListenSocket, SOL_SOCKET, SO_REUSEADDR, &Option, sizeof(Option)
    );

    if (Status < 0) {
	throw TSystemCallException(
	    "TMushServerSocket::EnableLocalAddressReuse(): "
	    "setsockopt(3N)"
	);
    }
}

void TMushServerSocket::Open(void) throw(TSystemCallException)
{
    if (_SocketStatus == ssInitial) {
	Create();
    }
    if (_SocketStatus == ssCreated) {
	Bind();
    }
    if (_SocketStatus == ssBound) {
	Listen();
    }
    if (_SocketStatus == ssListening) {
	Accept();
    }

    if (_SocketStatus == ssOpened) {
	;
    }
    else {
	throw TSystemCallException(
            "TMushServerSocket::Open()",
            "bad socket status to open"
        );
    }
}

void TMushServerSocket::Close(void)
{
    static const int ShutdownMethod = 2;  // 2: shutdown send and receive

    switch (_SocketStatus) {
      case ssOpened:
	shutdown(_ConnectSocket, ShutdownMethod);
	close(_ConnectSocket);
	_SocketStatus = ssListening;
      case ssListening:
	_SocketStatus = ssBound;
      case ssBound:
	_SocketStatus = ssCreated;
      case ssCreated:
	//close(_ListenSocket);  // close()ed just after accept()ed
	_SocketStatus = ssInitial;
      case ssInitial:
	break;
      default:
	;
    }
}

void TMushServerSocket::Listen(int MaxNumberOfConnections) throw(TSystemCallException)
{
    if (_SocketStatus == ssInitial) {
	Create();
    }
    if (_SocketStatus == ssCreated) {
	Bind();
    }

    if (_SocketStatus == ssBound) {
	if (listen(_ListenSocket, MaxNumberOfConnections) < 0) {
	    throw TSystemCallException(
                "TMushServerSocket::Listen(): listen(3N)"
            );
	}
	_SocketStatus = ssListening;
    }   

    if (_SocketStatus != ssListening) {
	throw TSystemCallException(
            "TMushServerSocket::Listen()",
            "bad socket status to listen"
        );
    }
}


TMushClientSocket::TMushClientSocket(void)
{
}

TMushClientSocket::~TMushClientSocket()
{
    Close();
}

void TMushClientSocket::Open(void) throw(TSystemCallException)
{
    if (_SocketStatus == ssInitial) {
	Create();
    }
    if (_SocketStatus == ssCreated) {
	Connect();
    }

    if (_SocketStatus == ssOpened) {
	;
    }    
    else {
	throw TSystemCallException(
            "TMushClientSocket::Open()",
            "bad socket status to open"
        );
    }
}

void TMushClientSocket::Close(void)
{
    static const int ShutdownMethod = 2;  // 2: shutdown send and receive

    switch (_SocketStatus) {
      case ssOpened:
	shutdown(_ConnectSocket, ShutdownMethod);
	close(_ConnectSocket);
	_SocketStatus = ssCreated;
      case ssListening:
	;
      case ssBound:
	;
      case ssCreated:
	_SocketStatus = ssInitial;
      case ssInitial:
	break;
      default:
	;
    }
}

