/* MushSocketSelector.cc */
/* UNIX System Call Interface (select(3N) or poll(3N)) */
/* Created by Enomoto Sanshiro on 4 December 1999. */
/* Last updated by Enomoto Sanshiro on 15 March 2000. */


#include <iostream>
#include <sstream>
#include <string>
#include <errno.h>
#include "MushSocketSelector.hh"

using namespace std;


static const int InitialFdSetCapacity = 32;


TMushSocketSelector::TMushSocketSelector(void)
{
    _PollFdSetCapacity = InitialFdSetCapacity;
    _PollFdSet = new pollfd[_PollFdSetCapacity];

    _NumberOfSockets = 0;
    _NextPolledFdIndex = 0;
}

TMushSocketSelector::~TMushSocketSelector()
{
    delete[] _PollFdSet;
}

void TMushSocketSelector::ResizeFdTable(int NewCapacity)
{
    _PollFdSetCapacity = NewCapacity;

    pollfd* OldPollFdSet = _PollFdSet;
    _PollFdSet = new pollfd[_PollFdSetCapacity];

    for (int i = 0; i < _NumberOfSockets; i++) {
	_PollFdSet[i] = OldPollFdSet[i];
    }

    delete[] OldPollFdSet;
}

void TMushSocketSelector::AddSocket(TMushSocket* Socket)
{
    if (_NumberOfSockets >= _PollFdSetCapacity) {
	ResizeFdTable(_PollFdSetCapacity * 2);
    }

    int FileDescriptor = Socket->FileDescriptor();
    _SocketTable[FileDescriptor] = Socket;

    _PollFdSet[_NumberOfSockets].fd = FileDescriptor;
    _PollFdSet[_NumberOfSockets].events = POLLIN;

    _NumberOfSockets++;
}

void TMushSocketSelector::RemoveSocket(TMushSocket* Socket)
{
    pollfd* OldPollFdSet = _PollFdSet;
    _PollFdSet = new pollfd[_PollFdSetCapacity];

    int OldNumberOfSockets = _NumberOfSockets;
    _NumberOfSockets = 0;
    for (int i = 0; i < OldNumberOfSockets; i++) {
	if (OldPollFdSet[i].fd != Socket->FileDescriptor()) {
	    _PollFdSet[_NumberOfSockets].fd = OldPollFdSet[i].fd;
	    _PollFdSet[_NumberOfSockets].events = OldPollFdSet[i].events;
	    _NumberOfSockets++;
	}
    }

    delete[] OldPollFdSet;
    _NextPolledFdIndex = 0;
}

pair<TMushSocket*, TMushSocketSelector::TSocketEvent> TMushSocketSelector::WaitEvent(int TimeOut_ms) throw(TSystemCallException)
{
    if (_NumberOfSockets == 0) {
	return make_pair((TMushSocket*) 0, SocketEvent_TimedOut);
    }

    int NumberOfEvents = poll(_PollFdSet, _NumberOfSockets, TimeOut_ms);

    if (NumberOfEvents == 0) {
	return make_pair((TMushSocket*) 0, SocketEvent_TimedOut);
    }
    else if (NumberOfEvents < 0) {
	if (errno == EINTR) {
	    return make_pair((TMushSocket*) 0, SocketEvent_Interrupted);
	}
	else {
	    throw TSystemCallException(
		"TMushSocketSelector::WaitEvent(): poll(3N)"
	    );
	}
    }

    TSocketEvent SocketEvent = SocketEvent_Others;
    int Event;
    int FdIndex, FdCount;
    for (FdCount = 0; FdCount < _NumberOfSockets; FdCount++) {

	FdIndex = _NextPolledFdIndex;
	_NextPolledFdIndex++;
	if (_NextPolledFdIndex >= _NumberOfSockets) {
	     _NextPolledFdIndex = 0;
	}
	    
	Event = _PollFdSet[FdIndex].revents;
	if (Event == 0) {
	    continue;
	}

	if (Event & POLLIN) {
	    SocketEvent = SocketEvent_DataArrived;
	    break;
	}
	else if (Event & POLLHUP) {
	    SocketEvent = SocketEvent_HungUp;
	    break;
	}
	else {
            ostringstream ErrorMessageStream;
	    if (Event & POLLNVAL) {
	        ErrorMessageStream << "Bad file descriptor: ";
		ErrorMessageStream << _PollFdSet[FdIndex].fd;
	    }
	    else {
	        ErrorMessageStream << "Error on data stream or deivce";
	    }

	    throw TSystemCallException(
		"TMushSocketSelector::WaitEvent(): poll(3N)",
		ErrorMessageStream.str()
	    );
	}
    }	 

    if (FdCount == _NumberOfSockets) {
	throw TSystemCallException(
	    "TMushSocketSelector::WaitEvent(): poll(3N)",
	    "poll() returned positive value, but no fd was set. Why?"
	);
    }

    int FileDescriptor = _PollFdSet[FdIndex].fd;
    TMushSocket* Socket = _SocketTable[FileDescriptor];

    return make_pair(Socket, SocketEvent);
}
