/* MushXml.cc */
/* Created by Enomoto Sanshiro on 30 July 2003. */
/* Last updated by Enomoto Sanshiro on 27 February 2008. */


#include <string>
#include <map>
#include <cstdarg>
#include <cstring>
#include "MushConfig.hh"
#include "MushXml.hh"

using namespace std;


#if USE_LIBXML

#if LIBXML_MAJOR_VERSION==1
  #include <parser.h>
  #include <parserInternals.h>
  #define xmlChar CHAR
#else
  #include <libxml/parser.h>
  #include <libxml/parserInternals.h>
#endif

static xmlSAXHandler sax_handler_list;
static bool is_sax_handler_list_construcuted = false;
static void sax_start_document(void* ctxt);
static void sax_end_document(void* ctxt);
static void sax_start_element(void* ctxt, const xmlChar* name, const xmlChar** attrs);
static void sax_end_element(void* ctxt, const xmlChar* name);
static void sax_characters(void* ctxt, const xmlChar* ch, int len);
static void sax_ignorable_whitespace(void* ctxt, const xmlChar* ch, int len);
static void sax_comment(void* ctxt, const xmlChar* ch);
static void sax_warning(void* ctxt, const char* msg, ...);
static void sax_error(void* ctxt, const char* msg, ...);
static void sax_fatal_error(void* ctxt, const char* msg, ...);
static void sax_set_document_locator(void* ctxt, xmlSAXLocator* loc);


static map<void*, TMushSaxDocumentHandler*> context_handler_table;
static map<void*, TMushSaxLocator*> context_locator_table;
static map<TMushSaxLocator*, pair<void*, xmlSAXLocator*> > locator_table;


TMushSaxLocator::TMushSaxLocator(void)
{
}

TMushSaxLocator::~TMushSaxLocator()
{
}

int TMushSaxLocator::getColumnNumber(void)
{
    void* ctxt = locator_table[this].first;
    xmlSAXLocator* locator = locator_table[this].second;

    return locator->getColumnNumber(ctxt);
}

int TMushSaxLocator::getLineNumber(void)
{
    void* ctxt = locator_table[this].first;
    xmlSAXLocator* locator = locator_table[this].second;

    return locator->getLineNumber(ctxt);
}

const char* TMushSaxLocator::getPublicId(void)
{
    void* ctxt = locator_table[this].first;
    xmlSAXLocator* locator = locator_table[this].second;

    return (const char*) locator->getPublicId(ctxt);
}

const char* TMushSaxLocator::getSystemId(void)
{
    void* ctxt = locator_table[this].first;
    xmlSAXLocator* locator = locator_table[this].second;

    return (const char*) locator->getSystemId(ctxt);
}



TMushSaxParser::TMushSaxParser(void)
{
    if (! is_sax_handler_list_construcuted) {
	memset(&sax_handler_list, sizeof(sax_handler_list), 0);

	sax_handler_list.startDocument = sax_start_document;
	sax_handler_list.endDocument = sax_end_document;
	sax_handler_list.startElement = sax_start_element;
	sax_handler_list.endElement = sax_end_element;
	sax_handler_list.characters = sax_characters;
	sax_handler_list.ignorableWhitespace = sax_ignorable_whitespace;
	sax_handler_list.comment = sax_comment;
	sax_handler_list.warning = sax_warning;
	sax_handler_list.error = sax_error;
	sax_handler_list.fatalError = sax_fatal_error;
	sax_handler_list.setDocumentLocator = sax_set_document_locator;

	is_sax_handler_list_construcuted = true;
    }

    _document_handler = 0;
}

TMushSaxParser::~TMushSaxParser()
{
}

void TMushSaxParser::setDocumentHandler(TMushSaxDocumentHandler* document_handler)
{
    _document_handler = document_handler;
}

void TMushSaxParser::parse(const std::string& file_name) throw(TMushSaxException)
{
    if (_document_handler == 0) {
	return;
    }

    xmlParserCtxt* context = xmlCreateFileParserCtxt(file_name.c_str());
    if (! context) {
	throw TMushSaxException(
	    "TMushSaxParser::parse()", "unable to open file: " + file_name
	);
    }
    context->sax = &sax_handler_list;

    context_handler_table[context] = _document_handler;

    xmlParseDocument(context);
    if (! context->wellFormed) {
	throw TMushSaxException(
	    "TMushSaxParser::parse()", "badly formed XML file: " + file_name
	);
    }

    context->sax = NULL;
    xmlFreeParserCtxt(context);

    TMushSaxLocator* Locator = context_locator_table[context];
    delete Locator;

    context_handler_table.erase(context_handler_table.find(context));
    context_locator_table.erase(context_locator_table.find(context));
    locator_table.erase(locator_table.find(Locator));
}

static void sax_start_document(void* ctxt)
{
    context_handler_table[ctxt]->startDocument();
}

static void sax_end_document(void* ctxt)
{
    context_handler_table[ctxt]->endDocument();
}

static void sax_start_element(void* ctxt, const xmlChar* name, const xmlChar** attrs)
{
    TMushSaxAttributeList attribute_list;

    if (attrs) {
	const char* attribute;
	string attribute_name, attribute_value;
	for (int i = 0; (attribute = (const char*) attrs[i]) != NULL; i++) {
	    if (i % 2 == 0) {
		attribute_name = attribute;
	    }
	    else {
		attribute_value = attribute;
		attribute_list.add(attribute_name, attribute_value);
	    }
	}
    }

    context_handler_table[ctxt]->startElement((const char*) name, attribute_list);
}

static void sax_end_element(void* ctxt, const xmlChar* name)
{
    context_handler_table[ctxt]->endElement((const char*) name);
}

static void sax_characters(void* ctxt, const xmlChar* ch, int len)
{
    string text;
    for (int i = 0; i < len; i++) {
	text += ch[i];
    }

    context_handler_table[ctxt]->characters(text);
}

static void sax_ignorable_whitespace(void* ctxt, const xmlChar* ch, int len)
{
    string text;
    for (int i = 0; i < len; i++) {
	text += ch[i];
    }

    context_handler_table[ctxt]->ignorableWhitespace(text);
}

static void sax_comment(void* ctxt, const xmlChar* ch)
{
    context_handler_table[ctxt]->comment((const char*) ch);
}

static void sax_warning(void* ctxt, const char* msg, ...)
{
    va_list args;
    char Buffer[256];

    va_start(args, msg);
    vsnprintf(Buffer, sizeof(Buffer), msg, args);
    va_end(args);

    context_handler_table[ctxt]->warning((const char*) Buffer);
}

static void sax_error(void* ctxt, const char* msg, ...)
{
    va_list args;
    char Buffer[256];

    va_start(args, msg);
    vsnprintf(Buffer, sizeof(Buffer), msg, args);
    va_end(args);

    context_handler_table[ctxt]->error((const char*) Buffer);
}

static void sax_fatal_error(void* ctxt, const char* msg, ...)
{
    va_list args;
    char Buffer[256];

    va_start(args, msg);
    vsnprintf(Buffer, sizeof(Buffer), msg, args);
    va_end(args);

    context_handler_table[ctxt]->fatalError((const char*) Buffer);
}

static void sax_set_document_locator(void* ctxt, xmlSAXLocator* loc)
{
    TMushSaxLocator* Locator = new TMushSaxLocator();
    context_locator_table[ctxt] = Locator;
    context_handler_table[ctxt]->setDocumentLocator(Locator);

    locator_table[Locator] = make_pair(ctxt, loc);
}

#else


TMushSaxLocator::TMushSaxLocator(void)
{
    throw TSystemCallException(
	"TMushSaxLocator", 
	"function not available ('libxml' must be installed)"
    );
}

TMushSaxLocator::~TMushSaxLocator()
{
}

int TMushSaxLocator::getColumnNumber(void)
{
    return 0;
}

int TMushSaxLocator::getLineNumber(void)
{
    return 0;
}

const char* TMushSaxLocator::getPublicId(void)
{
    return 0;
}

const char* TMushSaxLocator::getSystemId(void)
{
    return 0;
}


TMushSaxParser::TMushSaxParser(void)
{
    _document_handler = 0;
}

TMushSaxParser::~TMushSaxParser()
{
}

void TMushSaxParser::setDocumentHandler(TMushSaxDocumentHandler* document_handler)
{
    _document_handler = document_handler;
}

void TMushSaxParser::parse(const std::string& file_name) throw(TMushSaxException)
{
    throw TSystemCallException(
	"TMushSaxParser::parse()", 
	"function not available ('libxml' must be installed)"
    );
}

#endif



unsigned TMushSaxAttributeList::getLength(void) const
{
    return _name_list.size();
}

const std::string& TMushSaxAttributeList::getName(unsigned index) const
{
    return _name_list[index];
}

const std::string& TMushSaxAttributeList::getValue(unsigned index) const
{
    return _value_list[index];
}

const std::string& TMushSaxAttributeList::getValue(const std::string& name) const
{
    return _name_value_table[name];
}

void TMushSaxAttributeList::add(const std::string& name, const std::string& value)
{
    _name_list.push_back(name);
    _value_list.push_back(value);
    _name_value_table[name] = value;
}



TMushSaxException::TMushSaxException(const std::string& location, const std::string& message)
: TSystemCallException(location, message)
{
}
