/* ParaMathLibrary.cc */
/* Created by Enomoto Sanshiro on 22 August 1999. */
/* Last updated by Enomoto Sanshiro on 12 April 2001. */


#include <string>
#include <complex>
#include <vector>
#include <map>
#include <cmath>
#include <cstdlib>
#include "ParaObject.hh"
#include "ParaMathLibrary.hh"

using namespace std;


template<class T> inline static T sqr(const T& x) { return x*x; }


TParaMathObject::TParaMathObject(void)
: TParaObjectPrototype("Math")
{
}

TParaMathObject::~TParaMathObject()
{
}

TParaObjectPrototype* TParaMathObject::Clone(void)
{
    return new TParaMathObject();
}

int TParaMathObject::MethodIdOf(const std::string& MethodName)
{
    if (MethodName == "sin") {
        return MethodId_Sin;
    }
    else if (MethodName == "cos") {
        return MethodId_Cos;
    }
    else if (MethodName == "tan") {
        return MethodId_Tan;
    }
    else if (MethodName == "asin") {
        return MethodId_Asin;
    }
    else if (MethodName == "acos") {
        return MethodId_Acos;
    }
    else if (MethodName == "atan") {
        return MethodId_Atan;
    }
    else if (MethodName == "atan2") {
        return MethodId_Atan2;
    }
    else if (MethodName == "exp") {
        return MethodId_Exp;
    }
    else if (MethodName == "log") {
        return MethodId_Log;
    }
    else if (MethodName == "log10") {
        return MethodId_Log10;
    }
    else if (MethodName == "sqrt") {
        return MethodId_Sqrt;
    }
    else if (MethodName == "abs") {
        return MethodId_Abs;
    }
    else if (MethodName == "arg") {
        return MethodId_Arg;
    }
    else if (MethodName == "real") {
        return MethodId_Real;
    }
    else if (MethodName == "imag") {
        return MethodId_Imag;
    }
    else if (MethodName == "round") {
        return MethodId_Round;
    }
    else if (MethodName == "trunc") {
        return MethodId_Trunc;
    }
    else if (MethodName == "ceil") {
        return MethodId_Ceil;
    }
    else if (MethodName == "floor") {
        return MethodId_Floor;
    }
    else if (MethodName == "srand") {
        return MethodId_Srand;
    }
    else if (MethodName == "rand") {
        return MethodId_Rand;
    }
    else if (MethodName == "sinc") {
        return MethodId_Sinc;
    }

    return TParaObjectPrototype::MethodIdOf(MethodName);
}

int TParaMathObject::InvokeMethod(int MethodId, std::vector<TParaValue*>& ArgumentList, TParaValue& ReturnValue) throw(TScriptException)
{
    if ((ArgumentList.size() == 1) && ArgumentList[0]->IsList()) {
	ReturnValue = TParaValue(TParaListValue());
	vector<TParaValue>& InputList = ArgumentList[0]->AsValueList();
	vector<TParaValue>& ResultList = ReturnValue.AsValueList();

	TParaValue Result;
	vector<TParaValue*> InputArgumentList;
	InputArgumentList.push_back(&InputList[0]);

	for (unsigned i = 0; i < InputList.size(); i++) {
	    InputArgumentList[0] = &InputList[i];
	    if (this->InvokeMethod(MethodId, InputArgumentList, Result)) {
		ResultList.push_back(Result);
	    }
	    else {
		break;
	    }
	}
	return ResultList.size();
    }


    int Result = 0;

    switch (MethodId) {
      case MethodId_Sin:
        Result = Sin(ArgumentList, ReturnValue);
        break;
      case MethodId_Cos:
        Result = Cos(ArgumentList, ReturnValue);
        break;
      case MethodId_Tan:
        Result = Tan(ArgumentList, ReturnValue);
        break;
      case MethodId_Asin:
        Result = Asin(ArgumentList, ReturnValue);
        break;  
      case MethodId_Acos:
        Result = Acos(ArgumentList, ReturnValue);
        break;
      case MethodId_Atan:
        Result = Atan(ArgumentList, ReturnValue);
        break;
      case MethodId_Atan2:
        Result = Atan2(ArgumentList, ReturnValue);
        break;
      case MethodId_Exp:
        Result = Exp(ArgumentList, ReturnValue);
        break;
      case MethodId_Log:
        Result = Log(ArgumentList, ReturnValue);
        break;
      case MethodId_Log10:
        Result = Log10(ArgumentList, ReturnValue);
        break;
      case MethodId_Sqrt:
        Result = Sqrt(ArgumentList, ReturnValue);
        break;
      case MethodId_Abs:
        Result = Abs(ArgumentList, ReturnValue);
        break;
      case MethodId_Arg:
        Result = Arg(ArgumentList, ReturnValue);
        break;
      case MethodId_Real:
        Result = Real(ArgumentList, ReturnValue);
        break;
      case MethodId_Imag:
        Result = Imag(ArgumentList, ReturnValue);
        break;
      case MethodId_Round:
        Result = Round(ArgumentList, ReturnValue);
        break;
      case MethodId_Trunc:
        Result = Trunc(ArgumentList, ReturnValue);
        break;
      case MethodId_Ceil:
        Result = Ceil(ArgumentList, ReturnValue);
        break;
      case MethodId_Floor:
        Result = Floor(ArgumentList, ReturnValue);
        break;
      case MethodId_Srand:
        Result = Srand(ArgumentList, ReturnValue);
        break;
      case MethodId_Rand:
        Result = Rand(ArgumentList, ReturnValue);
        break;
      case MethodId_Sinc:
        Result = Sinc(ArgumentList, ReturnValue);
        break;
      default:
	Result = 0;
    }

    return Result;
}

int TParaMathObject::Sin(std::vector<TParaValue*>& ArgumentList, TParaValue& ReturnValue) throw(TScriptException)
{
    if (ArgumentList.size() != 1) {
        throw TScriptException("sin(): invalid number of argument[s]");
    }
    
    if (ArgumentList[0]->IsComplex()) {
	ReturnValue = TParaValue(sin(ArgumentList[0]->AsComplex()));
    }
    else {
	ReturnValue = TParaValue(sin(ArgumentList[0]->AsDouble()));
    }

    return 1;
}

int TParaMathObject::Cos(std::vector<TParaValue*>& ArgumentList, TParaValue& ReturnValue) throw(TScriptException)
{
    if (ArgumentList.size() != 1) {
        throw TScriptException("cos(): invalid number of argument[s]");
    }
    
    if (ArgumentList[0]->IsComplex()) {
	ReturnValue = TParaValue(cos(ArgumentList[0]->AsComplex()));
    }
    else {
	ReturnValue = TParaValue(cos(ArgumentList[0]->AsDouble()));
    }

    return 1;
}

int TParaMathObject::Tan(std::vector<TParaValue*>& ArgumentList, TParaValue& ReturnValue) throw(TScriptException)
{
    if (ArgumentList.size() != 1) {
        throw TScriptException("tan(): invalid number of argument[s]");
    }

    if (ArgumentList[0]->IsComplex()) {
	if (abs(cos(ArgumentList[0]->AsComplex())) == 0) {
	    throw TScriptException("tan(): invalid argument");
	}
	ReturnValue = TParaValue(tan(ArgumentList[0]->AsComplex()));
    }
    else {
	if (cos(ArgumentList[0]->AsDouble()) == 0) {
	    throw TScriptException("tan(): invalid argument");
	}
	ReturnValue = TParaValue(tan(ArgumentList[0]->AsDouble()));
    }

    return 1;
}

int TParaMathObject::Asin(std::vector<TParaValue*>& ArgumentList, TParaValue& ReturnValue) throw(TScriptException)
{
    if (ArgumentList.size() != 1) {
        throw TScriptException("asin(): invalid number of argument[s]");
    }
    
    // no asin(complex) available in the standard library //

    double x = ArgumentList[0]->AsDouble();
    if ((x < -1.0) || (x > 1.0)) {
	throw TScriptException("asin(): invalid argument");
    }
    ReturnValue = TParaValue(asin(x));

    return 1;
}

int TParaMathObject::Acos(std::vector<TParaValue*>& ArgumentList, TParaValue& ReturnValue) throw(TScriptException)
{
    if (ArgumentList.size() != 1) {
        throw TScriptException("acos(): invalid number of argument[s]");
    }
    
    // no acos(complex) available in the standard library //

    double x = ArgumentList[0]->AsDouble();
    if ((x < -1.0) || (x > 1.0)) {
	throw TScriptException("acos(): invalid argument");
    }
    ReturnValue = TParaValue(acos(x));

    return 1;
}

int TParaMathObject::Atan(std::vector<TParaValue*>& ArgumentList, TParaValue& ReturnValue) throw(TScriptException)
{
    if (ArgumentList.size() != 1) {
        throw TScriptException("atan(): invalid number of argument[s]");
    }
    
    // no atan(complex) available in the standard library //

    ReturnValue = TParaValue(atan(ArgumentList[0]->AsDouble()));

    return 1;
}

int TParaMathObject::Atan2(std::vector<TParaValue*>& ArgumentList, TParaValue& ReturnValue) throw(TScriptException)
{
    if (ArgumentList.size() != 2) {
        throw TScriptException(
	    "atan2(double y, double x): invalid number of argument[s]"
	);
    }
    
    // no atan2(complex) available in the standard library //

    if ((! ArgumentList[0]->IsList()) && (! ArgumentList[1]->IsList())) {
	ReturnValue = TParaValue(atan2(
            ArgumentList[0]->AsDouble(), ArgumentList[1]->AsDouble()
	));

	return 1;
    }

    vector<TParaValue>& YList = ArgumentList[0]->AsValueList();
    vector<TParaValue>& XList = ArgumentList[1]->AsValueList();
    if (YList.size() != XList.size()) {
	throw TScriptException(
	    "atan2(): inconsistent length of argument lists"
	);
    }
    
    ReturnValue = TParaValue(TParaListValue());
    vector<TParaValue>& ResultList = ReturnValue.AsValueList();
    for (unsigned i = 0; i < XList.size(); i++) {
	ResultList.push_back(TParaValue(atan2(
	    ArgumentList[0]->AsDouble(), ArgumentList[1]->AsDouble()
	)));
    }

    return ResultList.size();
}

int TParaMathObject::Exp(std::vector<TParaValue*>& ArgumentList, TParaValue& ReturnValue) throw(TScriptException)
{
    if (ArgumentList.size() != 1) {
        throw TScriptException("exp(): invalid number of argument[s]");
    }
    
    if (ArgumentList[0]->IsComplex()) {
	ReturnValue = TParaValue(exp(ArgumentList[0]->AsComplex()));
    }
    else {
	ReturnValue = TParaValue(exp(ArgumentList[0]->AsDouble()));
    }

    return 1;
}

int TParaMathObject::Log(std::vector<TParaValue*>& ArgumentList, TParaValue& ReturnValue) throw(TScriptException)
{
    if (ArgumentList.size() != 1) {
        throw TScriptException("log(): invalid number of argument[s]");
    }

    if (ArgumentList[0]->IsComplex()) {
	ReturnValue = TParaValue(log(ArgumentList[0]->AsComplex()));
    }
    else {
	double x = ArgumentList[0]->AsDouble();
	if (x <= 0) {
	    throw TScriptException("log(): invalid argument");
	}
	ReturnValue = TParaValue(log(x));
    }

    return 1;
}

int TParaMathObject::Log10(std::vector<TParaValue*>& ArgumentList, TParaValue& ReturnValue) throw(TScriptException)
{
    if (ArgumentList.size() != 1) {
        throw TScriptException("log10(): invalid number of argument[s]");
    }

    if (ArgumentList[0]->IsComplex()) {
	ReturnValue = TParaValue(log10(ArgumentList[0]->AsComplex()));
    }
    else {
	double x = ArgumentList[0]->AsDouble();
	if (x <= 0) {
	    throw TScriptException("log10(): invalid argument");
	}
	ReturnValue = TParaValue(log10(x));
    }

    return 1;
}

int TParaMathObject::Sqrt(std::vector<TParaValue*>& ArgumentList, TParaValue& ReturnValue) throw(TScriptException)
{
    if (ArgumentList.size() != 1) {
        throw TScriptException("sqrt(): invalid number of argument[s]");
    }
    
    if (ArgumentList[0]->IsComplex()) {
	ReturnValue = TParaValue(sqrt(ArgumentList[0]->AsComplex()));
    }
    else {
	double x = ArgumentList[0]->AsDouble();
	if (x < 0) {
	    throw TScriptException("sqrt(): invalid argument");
	}
	ReturnValue = TParaValue((double) sqrt(x));
    }

    return 1;
}

int TParaMathObject::Abs(std::vector<TParaValue*>& ArgumentList, TParaValue& ReturnValue) throw(TScriptException)
{
    if (ArgumentList.size() != 1) {
        throw TScriptException("abs(): invalid number of argument[s]");
    }
    
    if (ArgumentList[0]->IsComplex()) {
	ReturnValue = TParaValue(abs(ArgumentList[0]->AsComplex()));
    }
    else if (ArgumentList[0]->IsLong()) {
	ReturnValue = TParaValue((long) abs(ArgumentList[0]->AsLong()));
    }
    else {
	ReturnValue = TParaValue(fabs(ArgumentList[0]->AsDouble()));
    }

    return 1;
}

int TParaMathObject::Arg(std::vector<TParaValue*>& ArgumentList, TParaValue& ReturnValue) throw(TScriptException)
{
    if (ArgumentList.size() != 1) {
        throw TScriptException("arg(): invalid number of argument[s]");
    }
    
    if (ArgumentList[0]->IsComplex()) {
	ReturnValue = TParaValue((double) arg(ArgumentList[0]->AsComplex()));
    }
    else {
	ReturnValue = TParaValue(
	    (double) ((ArgumentList[0]->AsDouble() < 0) ? M_PI : 0)
	);
    }

    return 1;
}

int TParaMathObject::Real(std::vector<TParaValue*>& ArgumentList, TParaValue& ReturnValue) throw(TScriptException)
{
    if (ArgumentList.size() != 1) {
        throw TScriptException("real(): invalid number of argument[s]");
    }
    
    if (ArgumentList[0]->IsComplex()) {
	ReturnValue = TParaValue(ArgumentList[0]->AsComplex().real());
    }
    else {
	ReturnValue = *ArgumentList[0];
    }

    return 1;
}

int TParaMathObject::Imag(std::vector<TParaValue*>& ArgumentList, TParaValue& ReturnValue) throw(TScriptException)
{
    if (ArgumentList.size() != 1) {
        throw TScriptException("imag(): invalid number of argument[s]");
    }
    
    if (ArgumentList[0]->IsComplex()) {
	ReturnValue = TParaValue(ArgumentList[0]->AsComplex().imag());
    }
    else if (ArgumentList[0]->IsLong()) {
	ReturnValue = TParaValue((long) 0);
    }
    else {
	ReturnValue = TParaValue((double) 0);
    }

    return 1;
}

int TParaMathObject::Round(std::vector<TParaValue*>& ArgumentList, TParaValue& ReturnValue) throw(TScriptException)
{
    if (ArgumentList.size() != 1) {
        throw TScriptException("round(): invalid number of argument[s]");
    }
    
    if (! ArgumentList[0]->IsNumeric()) {
        throw TScriptException("round(): invalid argument");
    }

    ReturnValue = TParaValue(round(ArgumentList[0]->AsDouble()));

    return 1;
}

int TParaMathObject::Trunc(std::vector<TParaValue*>& ArgumentList, TParaValue& ReturnValue) throw(TScriptException)
{
    if (ArgumentList.size() != 1) {
        throw TScriptException("trunc(): invalid number of argument[s]");
    }
    
    if (! ArgumentList[0]->IsNumeric()) {
        throw TScriptException("trunc(): invalid argument");
    }

    ReturnValue = TParaValue(trunc(ArgumentList[0]->AsDouble()));

    return 1;
}

int TParaMathObject::Ceil(std::vector<TParaValue*>& ArgumentList, TParaValue& ReturnValue) throw(TScriptException)
{
    if (ArgumentList.size() != 1) {
        throw TScriptException("ceil(): invalid number of argument[s]");
    }
    
    if (! ArgumentList[0]->IsNumeric()) {
        throw TScriptException("ceil(): invalid argument");
    }

    ReturnValue = TParaValue(ceil(ArgumentList[0]->AsDouble()));

    return 1;
}

int TParaMathObject::Floor(std::vector<TParaValue*>& ArgumentList, TParaValue& ReturnValue) throw(TScriptException)
{
    if (ArgumentList.size() != 1) {
        throw TScriptException("floor(): invalid number of argument[s]");
    }
    
    if (! ArgumentList[0]->IsNumeric()) {
        throw TScriptException("floor(): invalid argument");
    }

    ReturnValue = TParaValue(floor(ArgumentList[0]->AsDouble()));

    return 1;
}

int TParaMathObject::Srand(std::vector<TParaValue*>& ArgumentList, TParaValue& ReturnValue) throw(TScriptException)
{
    if (ArgumentList.size() != 1) {
        throw TScriptException("srand(int): invalid number of argument[s]");
    }

    if (! ArgumentList[0]->IsNumeric()) {
        throw TScriptException("srand(): invalid argument");
    }

    srand48(ArgumentList[0]->AsLong());

    return 1;
}

int TParaMathObject::Rand(std::vector<TParaValue*>& ArgumentList, TParaValue& ReturnValue) throw(TScriptException)
{
    ReturnValue = TParaValue((double) drand48());

    return 1;
}

int TParaMathObject::Sinc(std::vector<TParaValue*>& ArgumentList, TParaValue& ReturnValue) throw(TScriptException)
{
    if (ArgumentList.size() != 1) {
        throw TScriptException("sinc(): invalid number of argument[s]");
    }
    
    if (ArgumentList[0]->IsComplex()) {
	complex<double> x = M_PI * ArgumentList[0]->AsComplex();
	complex<double> Sin = sin(x);
	complex<double> Sinc = abs(Sin - x) > 0 ? (Sin/x) : 1;
	ReturnValue = TParaValue(Sinc);
    }
    else {
	double x = M_PI * ArgumentList[0]->AsDouble();
	double Sin = sin(x);
	double Sinc = fabs(Sin - x) > 0 ? (Sin/x) : 1;
	ReturnValue = TParaValue(Sinc);
    }

    return 1;
}



TParaListMathObject::TParaListMathObject(void)
: TParaObjectPrototype("ListMath")
{
}

TParaListMathObject::~TParaListMathObject()
{
}

TParaObjectPrototype* TParaListMathObject::Clone(void)
{
    return new TParaListMathObject();
}

int TParaListMathObject::MethodIdOf(const std::string& MethodName)
{
    if (MethodName == "length") {
        return MethodId_Length;
    }
    else if (MethodName == "min") {
        return MethodId_Min;
    }
    else if (MethodName == "max") {
        return MethodId_Max;
    }
    else if (MethodName == "sum") {
        return MethodId_Sum;
    }
    else if ((MethodName == "mean") || (MethodName == "average")) {
        return MethodId_Mean;
    }
    else if ((MethodName == "deviation") || (MethodName == "rms")) {
        return MethodId_Deviation;
    }
    else if (MethodName == "delta") {
        return MethodId_Delta;
    }
    else if (MethodName == "sigma") {
        return MethodId_Sigma;
    }
    else if (MethodName == "zeros") {
        return MethodId_Zeros;
    }
    else if (MethodName == "ones") {
        return MethodId_Ones;
    }
    else if (MethodName == "find") {
        return MethodId_Find;
    }
    else if (MethodName == "count") {
        return MethodId_Count;
    }
    else if (MethodName == "divide") {
        return MethodId_Divide;
    }

    return TParaObjectPrototype::MethodIdOf(MethodName);
}

int TParaListMathObject::InvokeMethod(int MethodId, std::vector<TParaValue*>& ArgumentList, TParaValue& ReturnValue) throw(TScriptException)
{
    int Result = 0;

    switch (MethodId) {
      case MethodId_Length:
        Result = Length(ArgumentList, ReturnValue);
        break;
      case MethodId_Min:
        Result = Min(ArgumentList, ReturnValue);
        break;
      case MethodId_Max:
        Result = Max(ArgumentList, ReturnValue);
        break;
      case MethodId_Sum:
        Result = Sum(ArgumentList, ReturnValue);
        break;
      case MethodId_Mean:
        Result = Mean(ArgumentList, ReturnValue);
        break;
      case MethodId_Deviation:
        Result = Deviation(ArgumentList, ReturnValue);
        break;
      case MethodId_Delta:
        Result = Delta(ArgumentList, ReturnValue);
        break;
      case MethodId_Sigma:
        Result = Sigma(ArgumentList, ReturnValue);
        break;
      case MethodId_Zeros:
        Result = Zeros(ArgumentList, ReturnValue);
        break;
      case MethodId_Ones:
        Result = Ones(ArgumentList, ReturnValue);
        break;
      case MethodId_Find:
        Result = Find(ArgumentList, ReturnValue);
        break;
      case MethodId_Count:
        Result = Count(ArgumentList, ReturnValue);
        break;
      case MethodId_Divide:
        Result = Divide(ArgumentList, ReturnValue);
        break;
      default:
	Result = 0;
    }

    return Result;
}

int TParaListMathObject::Length(std::vector<TParaValue*>& ArgumentList, TParaValue& ReturnValue) throw(TScriptException)
{
    if ((ArgumentList.size() != 1) || (! ArgumentList[0]->IsList())) {
        throw TScriptException("length(): invalid argument[s]");
    }
    
    const TParaListValue& ListValue = ArgumentList[0]->AsConstList();
    const vector<TParaValue>& ValueList = ListValue.ConstValueList();
    unsigned ListLength = ValueList.size();

    ReturnValue = TParaValue((long) ListLength);

    return 1;
}

int TParaListMathObject::Min(std::vector<TParaValue*>& ArgumentList, TParaValue& ReturnValue) throw(TScriptException)
{
    if ((ArgumentList.size() != 1) || (! ArgumentList[0]->IsList())) {
        throw TScriptException("min(): invalid argument[s]");
    }
    
    TParaListValue& ListValue = ArgumentList[0]->AsList();
    vector<TParaValue>& ValueList = ListValue.ValueList();
    unsigned ListLength = ValueList.size();

    if (ListLength == 0) {
        throw TScriptException("min(): empty list");
    }

    //...BUG: check not only the first element ...//
    if (ValueList[0].IsList()) {
	ReturnValue = TParaValue(TParaListValue());
	vector<TParaValue>& ResultList = ReturnValue.AsValueList();
	for (unsigned i = 0; i < ListLength; i++) {
	    TParaValue SubReturnValue;
	    vector<TParaValue*> SubArgumentList;
	    SubArgumentList.push_back(&ValueList[i]);
	    this->Min(SubArgumentList, SubReturnValue);
	    ResultList.push_back(SubReturnValue);
	}
    }
    else {
	double Value = ValueList[0].AsDouble();
	for (unsigned i = 1; i < ListLength; i++) {
	    Value = min(Value, ValueList[i].AsDouble());
	}
	ReturnValue = TParaValue(Value);
    }

    return 1;
}

int TParaListMathObject::Max(std::vector<TParaValue*>& ArgumentList, TParaValue& ReturnValue) throw(TScriptException)
{
    if ((ArgumentList.size() != 1) || (! ArgumentList[0]->IsList())) {
        throw TScriptException("max(): invalid argument[s]");
    }
    
    TParaListValue& ListValue = ArgumentList[0]->AsList();
    vector<TParaValue>& ValueList = ListValue.ValueList();
    unsigned ListLength = ValueList.size();

    if (ListLength == 0) {
        throw TScriptException("max(): empty list");
    }

    //...BUG: check not only the first element ...//
    if (ValueList[0].IsList()) {
	ReturnValue = TParaValue(TParaListValue());
	vector<TParaValue>& ResultList = ReturnValue.AsValueList();
	for (unsigned i = 0; i < ListLength; i++) {
	    TParaValue SubReturnValue;
	    vector<TParaValue*> SubArgumentList;
	    SubArgumentList.push_back(&ValueList[i]);
	    this->Max(SubArgumentList, SubReturnValue);
	    ResultList.push_back(SubReturnValue);
	}
    }
    else {
	double Value = ValueList[0].AsDouble();
	for (unsigned i = 1; i < ListLength; i++) {
	    Value = max(Value, ValueList[i].AsDouble());
	}
	ReturnValue = TParaValue(Value);
    }

    return 1;
}

int TParaListMathObject::Sum(std::vector<TParaValue*>& ArgumentList, TParaValue& ReturnValue) throw(TScriptException)
{
    if ((ArgumentList.size() != 1) || (! ArgumentList[0]->IsList())) {
        throw TScriptException("sum(): invalid argument[s]");
    }
    
    TParaListValue& ListValue = ArgumentList[0]->AsList();
    vector<TParaValue>& ValueList = ListValue.ValueList();
    unsigned ListLength = ValueList.size();

    if (ListLength == 0) {
        throw TScriptException("sum(): empty list");
    }

    //...BUG: check not only the first element ...//
    if (ValueList[0].IsList()) {
	ReturnValue = TParaValue(TParaListValue());
	vector<TParaValue>& ResultList = ReturnValue.AsValueList();
	for (unsigned i = 0; i < ListLength; i++) {
	    TParaValue SubReturnValue;
	    vector<TParaValue*> SubArgumentList;
	    SubArgumentList.push_back(&ValueList[i]);
	    this->Sum(SubArgumentList, SubReturnValue);
	    ResultList.push_back(SubReturnValue);
	}
    }
    else {
	double Value = 0;
	for (unsigned i = 0; i < ListLength; i++) {
	    Value += ValueList[i].AsDouble();
	}
	ReturnValue = TParaValue(Value);
    }

    return 1;
}

int TParaListMathObject::Mean(std::vector<TParaValue*>& ArgumentList, TParaValue& ReturnValue) throw(TScriptException)
{
    if ((ArgumentList.size() != 1) || (! ArgumentList[0]->IsList())) {
        throw TScriptException("mean(): invalid argument[s]");
    }
    
    TParaListValue& ListValue = ArgumentList[0]->AsList();
    vector<TParaValue>& ValueList = ListValue.ValueList();
    unsigned ListLength = ValueList.size();

    if (ListLength == 0) {
        throw TScriptException("mean(): empty list");
    }

    //...BUG: check not only the first element ...//
    if (ValueList[0].IsList()) {
	ReturnValue = TParaValue(TParaListValue());
	vector<TParaValue>& ResultList = ReturnValue.AsValueList();
	for (unsigned i = 0; i < ListLength; i++) {
	    TParaValue SubReturnValue;
	    vector<TParaValue*> SubArgumentList;
	    SubArgumentList.push_back(&ValueList[i]);
	    this->Mean(SubArgumentList, SubReturnValue);
	    ResultList.push_back(SubReturnValue);
	}
    }
    else {
	double Value = 0;
	for (unsigned i = 0; i < ValueList.size(); i++) {
	    Value += ValueList[i].AsDouble();
	}
	Value /= ValueList.size();
	ReturnValue = TParaValue(Value);
    }

    return 1;
}

int TParaListMathObject::Deviation(std::vector<TParaValue*>& ArgumentList, TParaValue& ReturnValue) throw(TScriptException)
{
    if ((ArgumentList.size() != 1) || (! ArgumentList[0]->IsList())) {
        throw TScriptException("deviation(): invalid argument[s]");
    }
    
    TParaListValue& ListValue = ArgumentList[0]->AsList();
    vector<TParaValue>& ValueList = ListValue.ValueList();
    unsigned ListLength = ValueList.size();

    if (ListLength == 0) {
        throw TScriptException("deviation(): empty list");
    }

    //...BUG: check not only the first element ...//
    if (ValueList[0].IsList()) {
	ReturnValue = TParaValue(TParaListValue());
	vector<TParaValue>& ResultList = ReturnValue.AsValueList();
	for (unsigned i = 0; i < ListLength; i++) {
	    TParaValue SubReturnValue;
	    vector<TParaValue*> SubArgumentList;
	    SubArgumentList.push_back(&ValueList[i]);
	    this->Deviation(SubArgumentList, SubReturnValue);
	    ResultList.push_back(SubReturnValue);
	}
    }
    else {
	double Sum = 0, SumOfSquared = 0;
	for (unsigned i = 0; i < ValueList.size(); i++) {
	    Sum += ValueList[i].AsDouble();
	    SumOfSquared += sqr(ValueList[i].AsDouble());
	}
	double SquaredMean = sqr(Sum / ListLength);
	double MeanOfSquared = SumOfSquared / ListLength;
	if (MeanOfSquared > SquaredMean) {
	    double Value = sqrt(MeanOfSquared - SquaredMean);
	    ReturnValue = TParaValue(Value);
	}
	else {
	    ReturnValue = TParaValue((double) 0);
	}
    }

    return 1;
}

int TParaListMathObject::Delta(std::vector<TParaValue*>& ArgumentList, TParaValue& ReturnValue) throw(TScriptException)
{
    if ((ArgumentList.size() != 1) || (! ArgumentList[0]->IsList())) {
        throw TScriptException("delta(): invalid argument[s]");
    }
    
    ReturnValue = TParaValue(TParaListValue());
    vector<TParaValue>& InputList = ArgumentList[0]->AsValueList();
    vector<TParaValue>& ResultList = ReturnValue.AsValueList();
    unsigned ListLength = InputList.size();

    if (ListLength == 0) {
        throw TScriptException("delta(): empty list");
    }

    bool IsLong = InputList[0].IsLong();
    for (unsigned i = 1; i < ListLength; i++) {
	if (IsLong && ! InputList[i].IsLong()) {
	    IsLong = false;
	}
	if (IsLong) {
	    ResultList.push_back(TParaValue(
	        InputList[i].AsLong() - InputList[i-1].AsLong()
	    ));
	}
	else {
	    ResultList.push_back(TParaValue(
	        InputList[i].AsDouble() - InputList[i-1].AsDouble()
	    ));
	}
    }

    return 1;
}

int TParaListMathObject::Sigma(std::vector<TParaValue*>& ArgumentList, TParaValue& ReturnValue) throw(TScriptException)
{
    if ((ArgumentList.size() != 1) || (! ArgumentList[0]->IsList())) {
        throw TScriptException("sigma(): invalid argument[s]");
    }
    
    ReturnValue = TParaValue(TParaListValue());
    vector<TParaValue>& InputList = ArgumentList[0]->AsValueList();
    vector<TParaValue>& ResultList = ReturnValue.AsValueList();

    ResultList.push_back(TParaValue((long) 0));

    bool IsLong = true;
    long LongSum = 0;
    double DoubleSum = 0;
    for (unsigned i = 0; i < InputList.size(); i++) {
	if (IsLong && ! InputList[i].IsLong()) {
	    DoubleSum = LongSum;
	    IsLong = false;
	}
	if (IsLong) {
	    LongSum += InputList[i].AsLong();
	    ResultList.push_back(TParaValue(LongSum));
	}
	else {
	    DoubleSum += InputList[i].AsDouble();
	    ResultList.push_back(TParaValue(DoubleSum));
	}
    }

    return 1;
}

int TParaListMathObject::Zeros(std::vector<TParaValue*>& ArgumentList, TParaValue& ReturnValue) throw(TScriptException)
{
    if ((ArgumentList.size() != 1) || (! ArgumentList[0]->IsLong())) {
        throw TScriptException("zeros(): invalid argument[s]");
    }
    
    int Size = ArgumentList[0]->AsLong();
    ReturnValue = TParaValue(TParaListValue());
    vector<TParaValue>& ResultList = ReturnValue.AsValueList();

    for (int i = 0; i < Size; i++) {
	ResultList.push_back(TParaValue((long) 0));
    }

    return 1;
}

int TParaListMathObject::Ones(std::vector<TParaValue*>& ArgumentList, TParaValue& ReturnValue) throw(TScriptException)
{
    if ((ArgumentList.size() != 1) || (! ArgumentList[0]->IsLong())) {
        throw TScriptException("ones(): invalid argument[s]");
    }
    
    int Size = ArgumentList[0]->AsLong();
    ReturnValue = TParaValue(TParaListValue());
    vector<TParaValue>& ResultList = ReturnValue.AsValueList();

    for (int i = 0; i < Size; i++) {
	ResultList.push_back(TParaValue((long) 1));
    }

    return 1;
}

int TParaListMathObject::Find(std::vector<TParaValue*>& ArgumentList, TParaValue& ReturnValue) throw(TScriptException)
{
    if ((ArgumentList.size() != 1) || (! ArgumentList[0]->IsList())) {
        throw TScriptException("find(): invalid argument[s]");
    }
    
    ReturnValue = TParaValue(TParaListValue());
    vector<TParaValue>& ResultList = ReturnValue.AsValueList();

    vector<TParaValue>& InputList = ArgumentList[0]->AsValueList();
    for (unsigned i = 0; i < InputList.size(); i++) {
	if (InputList[i].AsBool()) {
	    ResultList.push_back(TParaValue((long) i));
	}
    }

    return 1;
}

int TParaListMathObject::Count(std::vector<TParaValue*>& ArgumentList, TParaValue& ReturnValue) throw(TScriptException)
{
    if ((ArgumentList.size() != 1) || (! ArgumentList[0]->IsList())) {
        throw TScriptException("count(): invalid argument[s]");
    }
    
    long Result = 0;

    vector<TParaValue>& InputList = ArgumentList[0]->AsValueList();
    for (unsigned i = 0; i < InputList.size(); i++) {
	if (InputList[i].AsBool()) {
	    Result++;
	}
    }

    ReturnValue = TParaValue(Result);

    return 1;
}

int TParaListMathObject::Divide(std::vector<TParaValue*>& ArgumentList, TParaValue& ReturnValue) throw(TScriptException)
{
    if (
	(ArgumentList.size() != 2) || 
	(! ArgumentList[0]->IsList()) || (! ArgumentList[1]->IsLong())
    ){
        throw TScriptException(
	    "divide(list src, int length): invalid argument[s]"
	);
    }
    
    vector<TParaValue>& InputList = ArgumentList[0]->AsValueList();

    int SectionLength = ArgumentList[1]->AsLong();
    if (SectionLength < 1) {
        throw TScriptException(
	    "divide(list src, int length): invalid length"
	);
    }

    int NumberOfSections = InputList.size() / SectionLength;
    if ((unsigned) (SectionLength * NumberOfSections) < InputList.size()) {
	NumberOfSections += 1;
    }

    ReturnValue = TParaValue(TParaListValue());
    vector<TParaValue>& ResultList = ReturnValue.AsValueList();

    unsigned Index = 0;
    for (int i = 0; i < NumberOfSections; i++) {
	TParaValue SectionListValue = TParaValue(TParaListValue());
	vector<TParaValue>& SectionList = SectionListValue.AsValueList();
	for (int j = 0; j < SectionLength; j++) {
	    if (Index >= InputList.size()) {
		break;
	    }
	    SectionList.push_back(InputList[Index++]);
	}
	ResultList.push_back(SectionListValue);
    }

    return 1;
}
