/* RoomVmeAccess.hh */
/* Created by Enomoto Sanshiro on 6 October 1997. */
/* Last updated by Enomoto Sanshiro on 9 August 2002. */


#ifndef __RoomVmeAccess_hh__
#define __RoomVmeAccess_hh__

#include <string>
#include <vector>
#include <sys/types.h>
#include "RoomDefs.hh"
#include "RoomDeviceBase.hh"
#include "RoomModule.hh"



enum TRoomVmeTransferMode {
    VmeTransferMode_A16D16, 
    VmeTransferMode_A16D32, 
    VmeTransferMode_A24D16, 
    VmeTransferMode_A24D32, 
    VmeTransferMode_A32D16, 
    VmeTransferMode_A32D32,
    VmeTransferMode_Unknown
};


class TRoomVmeAccessProperty {
  public:
    off_t VmeAddress;
    TRoomVmeTransferMode TransferMode;
    int DeviceDescriptor;
    caddr_t MappedAddress;
    size_t MapSize;
    off_t MapOffset;
    int DmaDeviceDescriptor;
    int InterruptNumber;
    int InterruptVector;
    int InterruptSignalId;
    bool IsInterruptAutodisableEnabled;
    void* DeviceSpecificData;
};


class TRoomVmeModule;

class TRoomVmeController: public TRoomController {
  public:
    TRoomVmeController(void);
    virtual ~TRoomVmeController();
    virtual TRoomVmeController* CloneController(void) = 0;
  public:
    virtual TRoomVmeAccessProperty* Open(TRoomVmeTransferMode TransferMode) throw(THardwareException);
    virtual void Close(TRoomVmeAccessProperty* Property);
    virtual long Read(TRoomVmeAccessProperty* Property, void* Buffer, int Size) throw(THardwareException);
    virtual long Write(TRoomVmeAccessProperty* Property, const void* Buffer, int Size) throw(THardwareException);
    virtual off_t Seek(TRoomVmeAccessProperty* Property, off_t OffsetAddress) throw(THardwareException);
    virtual caddr_t Map(TRoomVmeAccessProperty* Property, off_t VmeAddress, size_t MapSize, off_t MapOffset) throw(THardwareException);
    virtual void Unmap(TRoomVmeAccessProperty* Property) throw(THardwareException);
    virtual void RegisterInterruptNotification(TRoomVmeAccessProperty* Property, int SignalId = 0) throw(THardwareException) = 0;
    virtual void UnregisterInterruptNotification(TRoomVmeAccessProperty* Property) throw(THardwareException) = 0;
    virtual void EnableInterruptNotification(TRoomVmeAccessProperty* Property) throw(THardwareException) = 0;
    virtual void DisableInterruptNotification(TRoomVmeAccessProperty* Property) throw(THardwareException) = 0;
    virtual bool WaitForInterruptNotification(TRoomVmeAccessProperty* Property, unsigned TimeOut_sec) throw(THardwareException) = 0;
    virtual void SetInterruptNotificationAutodisable(TRoomVmeAccessProperty* Property) throw(THardwareException) = 0;
    virtual long DmaRead(TRoomVmeAccessProperty* Property, void* Buffer, int Size) throw(THardwareException);
    virtual long DmaWrite(TRoomVmeAccessProperty* Property, const void* Buffer, int Size) throw(THardwareException);
    virtual off_t DmaSeek(TRoomVmeAccessProperty* Property, off_t OffsetAddress) throw(THardwareException);
  public:
    virtual bool IsSignalOnInterruptAvailable(void);
  protected:
    virtual const char* DeviceFileName(TRoomVmeTransferMode TransferMode) = 0;
    virtual const char* DmaDeviceFileName(TRoomVmeTransferMode TransferMode) = 0;
    virtual void DmaOpen(TRoomVmeAccessProperty* Property) throw(THardwareException);
    virtual void DmaClose(TRoomVmeAccessProperty* Property);
};


class TRoomNullVmeController: public TRoomVmeController {
  public:
    TRoomNullVmeController(void);
    virtual ~TRoomNullVmeController();
    virtual TRoomVmeController* CloneController(void);
  public:
    virtual caddr_t Map(TRoomVmeAccessProperty* Property, off_t VmeAddress, size_t MapSize, off_t MapOffset) throw(THardwareException);
    virtual void Unmap(TRoomVmeAccessProperty* Property) throw(THardwareException);
    virtual void RegisterInterruptNotification(TRoomVmeAccessProperty* Property, int SignalId = 0) throw(THardwareException);
    virtual void UnregisterInterruptNotification(TRoomVmeAccessProperty* Property) throw(THardwareException);
    virtual void EnableInterruptNotification(TRoomVmeAccessProperty* Property) throw(THardwareException);
    virtual void DisableInterruptNotification(TRoomVmeAccessProperty* Property) throw(THardwareException);
    virtual bool WaitForInterruptNotification(TRoomVmeAccessProperty* Property, unsigned TimeOut_sec) throw(THardwareException);
    virtual void SetInterruptNotificationAutodisable(TRoomVmeAccessProperty* Property) throw(THardwareException);
  protected:
    virtual const char* DeviceFileName(TRoomVmeTransferMode TransferMode);
    virtual const char* DmaDeviceFileName(TRoomVmeTransferMode TransferMode);
};


class TRoomCheckerVmeController: public TRoomVmeController {
  public:
    TRoomCheckerVmeController(void);
    virtual ~TRoomCheckerVmeController();
    virtual TRoomVmeController* CloneController(void);
  public:
    virtual caddr_t Map(TRoomVmeAccessProperty* Property, off_t VmeAddress, size_t MapSize, off_t MapOffset) throw(THardwareException);
    virtual void Unmap(TRoomVmeAccessProperty* Property) throw(THardwareException);
    virtual void RegisterInterruptNotification(TRoomVmeAccessProperty* Property, int SignalId = 0) throw(THardwareException);
    virtual void UnregisterInterruptNotification(TRoomVmeAccessProperty* Property) throw(THardwareException);
    virtual void EnableInterruptNotification(TRoomVmeAccessProperty* Property) throw(THardwareException);
    virtual void DisableInterruptNotification(TRoomVmeAccessProperty* Property) throw(THardwareException);
    virtual bool WaitForInterruptNotification(TRoomVmeAccessProperty* Property, unsigned TimeOut_sec) throw(THardwareException);
    virtual void SetInterruptNotificationAutodisable(TRoomVmeAccessProperty* Property) throw(THardwareException);
  protected:
    virtual const char* DeviceFileName(TRoomVmeTransferMode TransferMode);
    virtual const char* DmaDeviceFileName(TRoomVmeTransferMode TransferMode);
};


class TRoomVmeCrate: public TRoomCrate {
  public:
    TRoomVmeCrate(TRoomVmeController *VmeController, int NumberOfSlots = 21);
    TRoomVmeCrate(int NumberOfSlots = 21);
    virtual ~TRoomVmeCrate();
    virtual void InstallController(TRoomVmeController *VmeController);
    virtual void Install(TRoomVmeModule *VmeModule, off_t VmeAddress) throw(THardwareException);
    virtual void Install(TRoomVmeModule *VmeModule, off_t VmeAddress, int InterruptNumber, int InterruptVector) throw(THardwareException);
    virtual void Uninstall(TRoomVmeModule *VmeModule);
    virtual TRoomVmeController* VmeController(void) const;
  protected:
    TRoomVmeController *_VmeController;
    std::vector<TRoomVmeModule*> _ModuleList;
    int _NumberOfSlots;
};


class TRoomVmeModule: public TRoomModule {
  public:
    TRoomVmeModule(TRoomVmeTransferMode TransferMode, size_t MapSize = 0, off_t MapOffset = 0);
    TRoomVmeModule(const std::string& ModuleType, const std::string& ModelName, TRoomVmeTransferMode TransferMode, size_t MapSize = 0, off_t MapOffset = 0);
    virtual ~TRoomVmeModule();
    virtual TRoomVmeModule* Clone(void) = 0;
    virtual void Destroy(void);
  public:
    // VME access functions //
    TRoomVmeTransferMode TransferMode(void) const;
    volatile caddr_t MappedAddress(void) const;
    size_t MapSize(void) const;
    off_t MapOffset(void) const;
    inline volatile Word* WordPointer(off_t OffsetAddress = 0);
    inline volatile Word& WordAt(off_t OffsetAddress);
    inline volatile DoubleWord* DoubleWordPointer(off_t OffsetAddress = 0);
    inline volatile DoubleWord& DoubleWordAt(off_t OffsetAddress);
    long PioRead(off_t Address, void* Buffer, int Size) throw(THardwareException);
    long PioWrite(off_t Address, const void* Buffer, int Size) throw(THardwareException);
    long DmaRead(off_t Address, void* Buffer, int Size) throw(THardwareException);
    long DmaWrite(off_t Address, const void* Buffer, int Size) throw(THardwareException);
  public:
    // Common implementations of the Module interface //
    virtual int BlockRead(int Address, void* Data, int MaxSize) throw(THardwareException);
    virtual int BlockWrite(int Address, const void* Data, int Size) throw(THardwareException);
    virtual int ReadRegister(int Address, int& Data) throw(THardwareException);
    virtual int WriteRegister(int Address, int Data) throw(THardwareException);
    virtual int EnableInterrupt(int SignalId = 0) throw(THardwareException);
    virtual int DisableInterrupt(void) throw(THardwareException);
    virtual int ClearInterrupt(void) throw(THardwareException);
    virtual bool WaitData(unsigned TimeOut_sec) throw(THardwareException);
  public:
    // Common implementations of the ServiceRequester interface //
    virtual bool IsSignalOnServiceRequestAvailable(void);
    virtual void EnableSignalOnServiceRequest(int SignalId) throw(THardwareException);
    virtual void DisableSignalOnServiceRequest(void) throw(THardwareException);
  public:
    // Override the following methods if necessary //
    virtual bool IsInterruptAutodisableEnabled(void);
  protected:
    TRoomVmeAccessProperty* VmeAccessProperty(void) const;
  private:
    friend class TRoomVmeCrate;
    void Attach(TRoomVmeCrate* Crate, TRoomVmeAccessProperty* Property, caddr_t MappedAddress);
    void Detach(void);
  private:
    TRoomVmeCrate* _VmeCrate;
    TRoomVmeController* _VmeController;
    TRoomVmeController* _CheckerVmeController;
    TRoomVmeAccessProperty* _VmeAccessProperty;
    TRoomVmeTransferMode _TransferMode;
    volatile caddr_t _MappedAddress;
    size_t _MapSize;
    off_t _MapOffset;
    bool _IsInterruptRegistered;
    bool _IsInterruptEnabled;
  public:
    //... for backward compatibility... use [Double]WordAt() methods instead.
    inline void PutWord(off_t OffsetAddress, Word Value);
    inline volatile Word GetWord(off_t OffsetAddress);
    inline void PutDoubleWord(off_t OffsetAddress, DoubleWord Value);
    inline volatile DoubleWord GetDoubleWord(off_t OffsetAddress);
};



inline volatile Word* TRoomVmeModule::WordPointer(off_t OffsetAddress) 
{
    return (volatile Word*) (_MappedAddress + OffsetAddress);
}

inline volatile Word& TRoomVmeModule::WordAt(off_t OffsetAddress) 
{
    return *((volatile Word*) (_MappedAddress + OffsetAddress));
}

inline volatile DoubleWord* TRoomVmeModule::DoubleWordPointer(off_t OffsetAddress) 
{
    return (volatile DoubleWord *) (_MappedAddress + OffsetAddress);
}

inline volatile DoubleWord& TRoomVmeModule::DoubleWordAt(off_t OffsetAddress) 
{
    return *((volatile DoubleWord *) (_MappedAddress + OffsetAddress));
}


//... for backward compatibility... use [Double]WordAt() methods instead.

inline void TRoomVmeModule::PutWord(off_t OffsetAddress, Word Value) 
{
    *((volatile Word*) (_MappedAddress + OffsetAddress)) = Value;
}

inline volatile Word TRoomVmeModule::GetWord(off_t OffsetAddress) 
{
    return *((volatile Word*) (_MappedAddress + OffsetAddress));
}

inline void TRoomVmeModule::PutDoubleWord(off_t OffsetAddress, DoubleWord Value) 
{
    *((volatile DoubleWord *) (_MappedAddress + OffsetAddress)) = Value;
}

inline volatile DoubleWord TRoomVmeModule::GetDoubleWord(off_t OffsetAddress) 
{
    return *((volatile DoubleWord *) (_MappedAddress + OffsetAddress));
}


#endif
