/* bridge-Toyo_CCNET.cc */
/* Created by Enomoto Sanshiro on 22 July 2009. */
/* Last updated by Enomoto Sanshiro on 15 January 2010. */


#include <iostream>
#include <cstring>
#include <arpa/inet.h>
#include "RoomDeviceFactory.hh"
#include "RoomNetworkBridge.hh"
#include "bridge-Toyo_CCNET.hh"

using namespace std;


#define DEBUG(x) (x)


static TRoomNetworkCamacBridgeCreator Creator1(
    "Toyo_CCNET_Remote", new TNetworkCamacBridge_Toyo_CCNET()
);
static TRoomNetworkCamacBridgeCreator Creator2(
    "KEK_CCNET_Remote", new TNetworkCamacBridge_Toyo_CCNET()
);

static TRoomNetworkModuleCreator Creator3(
    "Toyo_CCNET_Remote", new TNetworkCamacBridge_Toyo_CCNET()
);
static TRoomNetworkModuleCreator Creator4(
    "KEK_CCNET_Remote", new TNetworkCamacBridge_Toyo_CCNET()
);


// CCNET Function Implementation //

TMiscDevice_Toyo_CCNET::TMiscDevice_Toyo_CCNET(TRoomNetworkConnection* NetworkConnection)
{
    _Connection = NetworkConnection;

    _ReceiveBuffer = 0;
    _SendBuffer = 0;
}

TMiscDevice_Toyo_CCNET::~TMiscDevice_Toyo_CCNET()
{
    delete[] _ReceiveBuffer;
    delete[] _SendBuffer;
}

void TMiscDevice_Toyo_CCNET::CCNET_Open(void) throw(THardwareException)
{
    if (_SendBuffer == 0) {
	_SendBuffer = new int[_PacketSize];
    }
    if (_ReceiveBuffer == 0) {
	_ReceiveBuffer = new int[_PacketSize];
    }

    _SendBuffer[Offset_TotalSize] = htonl(_PacketHeaderSize);
    _SendBuffer[Offset_Command] = htonl(Command_Open);

    _Connection->SendBlock(_SendBuffer, _PacketHeaderSize);
    _Connection->ReceiveBlock(_ReceiveBuffer, _PacketHeaderSize);
}

void TMiscDevice_Toyo_CCNET::CCNET_Close(void) throw(THardwareException)
{
    _SendBuffer[Offset_TotalSize] = htonl(_PacketHeaderSize);
    _SendBuffer[Offset_Command] = htonl(Command_Close);

    _Connection->SendBlock(_SendBuffer, _PacketHeaderSize);
    _Connection->ReceiveBlock(_ReceiveBuffer, _PacketHeaderSize);
}

void TMiscDevice_Toyo_CCNET::CCNET_Reset(void) throw(THardwareException)
{
    _SendBuffer[Offset_TotalSize] = htonl(_PacketHeaderSize);
    _SendBuffer[Offset_Command] = htonl(Command_Reset);

    _Connection->SendBlock(_SendBuffer, _PacketHeaderSize);
    _Connection->ReceiveBlock(_ReceiveBuffer, _PacketHeaderSize);
}

void TMiscDevice_Toyo_CCNET::CCNET_ClearFifo(void) throw(THardwareException)
{
    _SendBuffer[Offset_TotalSize] = htonl(_PacketHeaderSize);
    _SendBuffer[Offset_Command] = htonl(Command_ClearFifo);
    
    _Connection->SendBlock(_SendBuffer, _PacketHeaderSize);
    _Connection->ReceiveBlock(_ReceiveBuffer, _PacketHeaderSize);
}

void TMiscDevice_Toyo_CCNET::CCNET_EnableLam(int LamPattern) throw(THardwareException)
{
    _SendBuffer[Offset_TotalSize] = htonl(_PacketHeaderSize + _WordSize);
    _SendBuffer[Offset_Command] = htonl(Command_EnableLam);
    _SendBuffer[Offset_DataStart] = htonl(LamPattern);

    _Connection->SendBlock(_SendBuffer, _PacketHeaderSize + _WordSize);
    _Connection->ReceiveBlock(_ReceiveBuffer, _PacketHeaderSize);
}

void TMiscDevice_Toyo_CCNET::CCNET_DisableLam(void) throw(THardwareException)
{
    _SendBuffer[Offset_TotalSize] = htonl(_PacketHeaderSize);
    _SendBuffer[Offset_Command] = htonl(Command_DisableLam);

    _Connection->SendBlock(_SendBuffer, _PacketHeaderSize);
    _Connection->ReceiveBlock(_ReceiveBuffer, _PacketHeaderSize);
}

int TMiscDevice_Toyo_CCNET::CCNET_WaitLam(int Timeout) throw(THardwareException)
{
    _SendBuffer[Offset_TotalSize] = htonl(_PacketHeaderSize + _WordSize);
    _SendBuffer[Offset_Command] = htonl(Command_WaitLam);
    _SendBuffer[Offset_DataStart] = htonl(Timeout);

    _Connection->SendBlock(_SendBuffer, _PacketHeaderSize + _WordSize);
    _Connection->ReceiveBlock(_ReceiveBuffer, _PacketHeaderSize + _WordSize);

    int LamPattern = ntohl(_ReceiveBuffer[Offset_DataStart]);

    return LamPattern;
}

void TMiscDevice_Toyo_CCNET::CCNET_EnableTrigger(void) throw(THardwareException)
{
    _SendBuffer[Offset_TotalSize] = htonl(_PacketHeaderSize);
    _SendBuffer[Offset_Command] = htonl(Command_EnableTrigger);

    _Connection->SendBlock(_SendBuffer, _PacketHeaderSize + _WordSize);
    _Connection->ReceiveBlock(_ReceiveBuffer, _PacketHeaderSize);
}

void TMiscDevice_Toyo_CCNET::CCNET_DisableTrigger(void) throw(THardwareException)
{
    _SendBuffer[Offset_TotalSize] = htonl(_PacketHeaderSize);
    _SendBuffer[Offset_Command] = htonl(Command_DisableTrigger);

    _Connection->SendBlock(_SendBuffer, _PacketHeaderSize);
    _Connection->ReceiveBlock(_ReceiveBuffer, _PacketHeaderSize);
}

int TMiscDevice_Toyo_CCNET::CCNET_WaitTrigger(int Timeout) throw(THardwareException)
{
    _SendBuffer[Offset_TotalSize] = htonl(_PacketHeaderSize + _WordSize);
    _SendBuffer[Offset_Command] = htonl(Command_WaitLam);
    _SendBuffer[Offset_DataStart] = htonl(Timeout);

    _Connection->SendBlock(_SendBuffer, _PacketHeaderSize + _WordSize);
    _Connection->ReceiveBlock(_ReceiveBuffer, _PacketHeaderSize + _WordSize);

    int EventCount = ntohl(_ReceiveBuffer[Offset_DataStart]);

    return EventCount;
}

void TMiscDevice_Toyo_CCNET::CCNET_Put(int Data, int Command) throw(THardwareException)
{
    _SendBuffer[Offset_TotalSize] = htonl(_PacketHeaderSize + 2*_WordSize);
    _SendBuffer[Offset_Command] = htonl(Command_Put);
    _SendBuffer[Offset_DataStart] = htonl(Data);
    _SendBuffer[Offset_DataStart+1] = htonl(Command);
    
    _Connection->SendBlock(_SendBuffer, _PacketHeaderSize + 2*_WordSize);
    _Connection->ReceiveBlock(_ReceiveBuffer, _PacketHeaderSize);
}

void TMiscDevice_Toyo_CCNET::CCNET_Get(int& Data, int& Reply) throw(THardwareException)
{
    _SendBuffer[Offset_TotalSize] = htonl(_PacketHeaderSize);
    _SendBuffer[Offset_Command] = htonl(Command_Get);
    
    _Connection->SendBlock(_SendBuffer, _PacketHeaderSize);
    _Connection->ReceiveBlock(_ReceiveBuffer, _PacketHeaderSize + 2*_WordSize);

    Data = ntohl(_ReceiveBuffer[Offset_DataStart]);
    Reply = ntohl(_ReceiveBuffer[Offset_DataStart + 1]);
}

int TMiscDevice_Toyo_CCNET::CCNET_ExecuteFrame(int Command, const int* CommandBuffer, int* ReplyBuffer) throw(THardwareException)
{
    int CommandSize = 8 + CommandBuffer[1] * 8;

    static int CommandSendBuffer[_MaxBufferSize/4];
    for (int i = 0; i < CommandSize/4; i++) {
	CommandSendBuffer[i] = htonl(CommandBuffer[i]);
    }
    _SendBuffer[Offset_TotalSize] = htonl(_PacketHeaderSize + CommandSize);
    _SendBuffer[Offset_Command] = Command;

    _Connection->SendBlock(_SendBuffer, _PacketHeaderSize);
    _Connection->SendBlock(CommandSendBuffer, CommandSize);

    _Connection->ReceiveBlock(_ReceiveBuffer, _PacketHeaderSize);

    _Connection->ReceiveBlock(ReplyBuffer, _PacketHeaderSize);
    int ReplyLength = ntohl(ReplyBuffer[1]);
    if (ReplyLength > _MaxFrameLength) {
	throw THardwareException(
	    "CCNET_ExecuteFrame()", "Reply packet too long"
	);
    }
    
    _Connection->ReceiveBlock(ReplyBuffer+2, ReplyLength * 8);
    for (int i = 0; i < ReplyLength*2 + 2; i++) {
	ReplyBuffer[i] = ntohl(ReplyBuffer[i]);
    }

    // Reply Foramt //
    // ReplyBuffer[0]: Capacity:32
    // ReplyBuffer[1]: Length:32
    // Reply[2+2*i]: 1st Reply Word of i-th Frame
    // Reply[2+2*i+1]: 2nd Reply Word of i-th Frame

    return ReplyLength;
}

int TMiscDevice_Toyo_CCNET::CCNET_ExecutePio(const int* CommandBuffer, int* ReplyBuffer) throw(THardwareException)
{
    return CCNET_ExecuteFrame(Command_ExecutePio, CommandBuffer, ReplyBuffer);
}

int TMiscDevice_Toyo_CCNET::CCNET_ExecuteDma(const int* CommandBuffer, int* ReplyBuffer) throw(THardwareException)
{
    return CCNET_ExecuteFrame(Command_ExecuteDma, CommandBuffer, ReplyBuffer);
}

int TMiscDevice_Toyo_CCNET::CCNET_ExecuteDmaSequence(const int* CommandBuffer, int* ReplyBuffer) throw(THardwareException)
{
    return CCNET_ExecuteFrame(
	Command_ExecuteDmaSequence, CommandBuffer, ReplyBuffer
    );
}

int TMiscDevice_Toyo_CCNET::CCNET_Execute(const int* CommandBuffer, int* ReplyBuffer) throw(THardwareException)
{
    return CCNET_ExecuteFrame(Command_Execute, CommandBuffer, ReplyBuffer);
}

static inline const int NAF(int N, int A, int F)
{
    return ((((N << 8) + A) << 8) + F);
}

void TMiscDevice_Toyo_CCNET::CCNET_FillCamacCommandTo(int Command[2], int N, int A, int F, int Data, int FrameFlag)
{
    if (FrameFlag == 0) {
	FrameFlag = FrameFlag_Packet;
    }
    FrameFlag |= FrameFlag_Camac;

    Command[0] = Data & 0x00ffffff;
    Command[1] = (FrameFlag << 24) | NAF(N, A, F);
}

void TMiscDevice_Toyo_CCNET::CCNET_FillDaqCommandTo(int Command[2], int Function, int FrameFlag)
{
    if (FrameFlag == 0) {
	FrameFlag = FrameFlag_Packet;
    }
    FrameFlag |= FrameFlag_Daq;

    Command[0] = 0;
    Command[1] = (FrameFlag << 24) | Function;
}

void TMiscDevice_Toyo_CCNET::CCNET_ExtractCamacReplyFrom(int Reply[2], int& N, int& A, int& F, int& Data, int&Q, int&X)
{
    N = (Reply[0] >> 16) & 0x1f;
    A = (Reply[0] >> 8) & 0x0f;
    F = (Reply[0] >> 0) & 0x1f;
    
    Data = Reply[0] & 0x00ffffff;

    Q = (Reply[1] & 0x01000000) ? 1 : 0;
    X = (Reply[1] & 0x02000000) ? 1 : 0;
}

void TMiscDevice_Toyo_CCNET::CCNET_ExtractDaqReplyFrom(int Reply[2], int& Data, int& Status)
{
    Data = Reply[0];
    Status = Reply[1] & 0x00ff;
}

bool TMiscDevice_Toyo_CCNET::CCNET_IsCamacReply(int Reply[2])
{
    return (Reply[0] & FrameFlag_Daq) == 0;
}

void TMiscDevice_Toyo_CCNET::CCNET_ExecuteSingleCamac(int N, int A, int F, int& Data, int& Q, int& X) throw(THardwareException)
{
    int Command[2], Reply[2];

    CCNET_FillCamacCommandTo(Command, N, A, F, Data);
    CCNET_Put(Command[0], Command[1]);

    CCNET_Get(Reply[0], Reply[1]);
    CCNET_ExtractCamacReplyFrom(Reply, N, A, F, Data, Q, X);
}

void TMiscDevice_Toyo_CCNET::CCNET_ExecuteSingleDaq(int Function, int& Data, int& Status) throw(THardwareException)
{
    int Command[2], Reply[2];

    CCNET_FillDaqCommandTo(Command, Function, Data);
    CCNET_Put(Command[0], Command[1]);

    CCNET_Get(Reply[0], Reply[1]);
    CCNET_ExtractDaqReplyFrom(Reply, Data, Status);
}

int* TMiscDevice_Toyo_CCNET::CCNET_CreateBuffer(void)
{
    // 64bit wide frame + two 4-byte header //
    return new int[2*_MaxFrameLength + 2];
}

void TMiscDevice_Toyo_CCNET::CCNET_DestroyBuffer(int* Buffer)
{
    delete[] Buffer;
}

void TMiscDevice_Toyo_CCNET::CCNET_InitializeCommandBuffer(int* CommandBuffer) throw(THardwareException)
{
    int Size = 8 * (_MaxFrameLength + 1);
    memset(CommandBuffer, 0, Size);

    CommandBuffer[0] = _MaxFrameLength;  
}

void TMiscDevice_Toyo_CCNET::CCNET_CompleteCommandBuffer(int* CommandBuffer) throw(THardwareException)
{
    int& Length = CommandBuffer[1];
    CommandBuffer[0] = Length;

    for (int Index = 0; Index < Length; Index++) {
	int* Frame = CommandBuffer + 2 * (Index + 1);
	int Flag;
	if (Length == 1) {
	    Flag = FrameFlag_Packet;
	}
	else if (Index == 0) {
	    Flag = FrameFlag_Start;
	}
	else if (Index == Length-1) {
	    Flag = FrameFlag_End;
	}
	else {
	    Flag = FrameFlag_Normal;
	}
	Frame[1] |= (Flag << 24);
    }
}

void TMiscDevice_Toyo_CCNET::CCNET_LoadCamacCommand(int* CommandBuffer, int N, int A, int F, int Data) throw(THardwareException)
{
    int& Capacity = CommandBuffer[0];
    int& Index = CommandBuffer[1];
    if (Index >= Capacity) {
	throw THardwareException(
	    "CCNET_LoadCamacCommand()", "command buffer overflow"
	);
    }

    int* Frame = CommandBuffer + 2 * (Index + 1);
    CCNET_FillCamacCommandTo(Frame, N, A, F, Data);
    Index++;
}

void TMiscDevice_Toyo_CCNET::CCNET_LoadDaqCommand(int* CommandBuffer, int Command, int Data) throw(THardwareException)
{
    int& Capacity = CommandBuffer[0];
    int& Index = CommandBuffer[1];
    if (Index >= Capacity) {
	throw THardwareException(
	    "CCNET_LoadDaqCommand()", "command buffer overflow"
	);
    }

    int* Frame = CommandBuffer + 2 * (Index + 1);
    CCNET_FillDaqCommandTo(Frame, Command, Data);
    Index++;
}


// Network-Camac Bridge //

TNetworkCamacBridge_Toyo_CCNET::TNetworkCamacBridge_Toyo_CCNET(void)
: TRoomNetworkCamacBridge("Toyo_CCNET_Remote"), TMiscDevice_Toyo_CCNET(this)
{
    _IsOpened = false;
    _IsInterruptNotificationEnabled = false;

    _CommandBuffer = 0;
    _ReplyBuffer = 0;

    _ReplySize = 0;
}

TNetworkCamacBridge_Toyo_CCNET::~TNetworkCamacBridge_Toyo_CCNET()
{
}

TRoomNetworkCamacBridge* TNetworkCamacBridge_Toyo_CCNET::CloneBridge(void)
{
    return new TNetworkCamacBridge_Toyo_CCNET();
}


// CAMAC Controller Functions //

void TNetworkCamacBridge_Toyo_CCNET::Open(void) throw(THardwareException)
{
    if (_IsOpened) {
	return;
    }

    CCNET_Open();
    _IsOpened = true;

    int N, A, F, Data, Q, X;
    CCNET_ExecuteSingleCamac(N=25, A=0, F=17, Data, Q, X);  // Z
    CCNET_ExecuteSingleCamac(N=25, A=0, F=16, Data, Q, X);  // C
    CCNET_ExecuteSingleCamac(N=25, A=0, F=24, Data, Q, X);  // RI

    _IsInterruptNotificationEnabled = false;
}

void TNetworkCamacBridge_Toyo_CCNET::Close(void)
{
    if (_IsOpened) {
	CCNET_Close();
    }
}

int TNetworkCamacBridge_Toyo_CCNET::Transact(int StationNumber, int Function, int Address, int &Data, int &Q, int &X) throw(THardwareException)
{
    CCNET_ExecuteSingleCamac(StationNumber, Address, Function, Data, Q, X);

    return 1;
}

int TNetworkCamacBridge_Toyo_CCNET::Initialize(void) throw(THardwareException)
{
    int N, A, F, Data, Q, X;
    CCNET_ExecuteSingleCamac(N=25, A=0, F=17, Data, Q, X);

    return 1;
}

int TNetworkCamacBridge_Toyo_CCNET::Clear(void) throw(THardwareException)
{
    int N, A, F, Data, Q, X;
    CCNET_ExecuteSingleCamac(N=25, A=0, F=16, Data, Q, X);

    return 1;
}

int TNetworkCamacBridge_Toyo_CCNET::SetInhibit(void) throw(THardwareException)
{
    int N, A, F, Data, Q, X;
    CCNET_ExecuteSingleCamac(N=25, A=0, F=26, Data, Q, X);

    return 1;
}

int TNetworkCamacBridge_Toyo_CCNET::ReleaseInhibit(void) throw(THardwareException)
{
    int N, A, F, Data, Q, X;
    CCNET_ExecuteSingleCamac(N=25, A=0, F=24, Data, Q, X);

    return 1;
}

int TNetworkCamacBridge_Toyo_CCNET::EnableInterruptNotification(void) throw(THardwareException)
{
    if (! _IsInterruptNotificationEnabled) {
	CCNET_EnableLam();

	//...TODO: check this ...//
	int N, A, F, Data, Q, X;
	CCNET_ExecuteSingleCamac(N=25, A=1, F=16, Data=0x00ffffff, Q, X);
	CCNET_ExecuteSingleCamac(N=25, A=1, F=26, Data, Q, X);

	_IsInterruptNotificationEnabled = true;
    }

    return 1;
}

int TNetworkCamacBridge_Toyo_CCNET::DisableInterruptNotification(void) throw(THardwareException)
{
    if (_IsInterruptNotificationEnabled) {

	//...TODO: check this ...//
	int N, A, F, Data, Q, X;
	CCNET_ExecuteSingleCamac(N=25, A=1, F=24, Data, Q, X);
	CCNET_ExecuteSingleCamac(N=25, A=1, F=16, Data=0, Q, X);

	CCNET_DisableLam();

	_IsInterruptNotificationEnabled = false;
    }

    return 1;
}

int TNetworkCamacBridge_Toyo_CCNET::ReadLam(int LamMask) throw(THardwareException)
{
    //... TODO: this is very inefficient ...//

    static const int fnTestLam = 8;

    int LamPattern = 0;
    int StationBit, Data, Q, X;
    for (int Station = 1; Station <= 24; Station++) {
	StationBit = Bit(Station);
	if (StationBit & LamMask) {
	    Transact(Station, fnTestLam, 0, Data, Q, X);
	    if (Q) {
		LamPattern |= StationBit;
	    }
	}
    }

    return LamPattern;
}

int TNetworkCamacBridge_Toyo_CCNET::WaitLam(int TimeOut_sec) throw(THardwareException)
{
    if (! _IsInterruptNotificationEnabled) {
	EnableInterruptNotification();
    }

    // depends on CCNET kernel setting //
    static const int Hz = 100;

    return CCNET_WaitLam(TimeOut_sec * Hz);
}


// Network Module //

int TNetworkCamacBridge_Toyo_CCNET::AddressBitLength(void)
{
    return 8;
}

int TNetworkCamacBridge_Toyo_CCNET::DataBitLength(void)
{
    return 24;
}

int TNetworkCamacBridge_Toyo_CCNET::Initialize(int InitialState) throw(THardwareException)
{
    if (_IsOpened) {
	return 1;
    }

    CCNET_Open();
    _IsOpened = true;

    int N, A, F, Data, Q, X;
    CCNET_ExecuteSingleCamac(N=25, A=0, F=17, Data, Q, X);  // Z
    CCNET_ExecuteSingleCamac(N=25, A=0, F=16, Data, Q, X);  // C
    CCNET_ExecuteSingleCamac(N=25, A=0, F=24, Data, Q, X);  // RI

    _IsInterruptNotificationEnabled = false;
    _ReplySize = 0;

    return 1;
}

int TNetworkCamacBridge_Toyo_CCNET::Finalize(int FinalState) throw(THardwareException)
{
    if (_IsOpened) {
	CCNET_Close();
    }

    if (_CommandBuffer) {
	CCNET_DestroyBuffer(_CommandBuffer);
	CCNET_DestroyBuffer(_ReplyBuffer);
    }

    return 1;
}

int TNetworkCamacBridge_Toyo_CCNET::Clear(int Address) throw(THardwareException)
{
    int N, A, F, Data, Q, X;
    CCNET_ExecuteSingleCamac(N=25, A=0, F=16, Data, Q, X);

    _ReplySize = 0;

    return 0;
}

int TNetworkCamacBridge_Toyo_CCNET::BlockRead(int Address, void *Data, int MaxSize) throw(THardwareException)
{
    if (MaxSize < _ReplySize) {
	throw THardwareException("CCNET::BlockRead()", "buffer too small");
    }

    memcpy(Data, _ReplyBuffer, _ReplySize);

    return _ReplySize;
}

int TNetworkCamacBridge_Toyo_CCNET::NextDataBlockSize(int Address) throw(THardwareException)
{
    return _ReplySize;
}

void TNetworkCamacBridge_Toyo_CCNET::ClearServiceRequest(void) throw(THardwareException)
{
    //... TODO: check this ...//
    DisableInterruptNotification();
    EnableInterruptNotification();
}

bool TNetworkCamacBridge_Toyo_CCNET::WaitForServiceRequest(int TimeOut_sec) throw(THardwareException)
{
    return (WaitLam(TimeOut_sec) > 0);
}

bool TNetworkCamacBridge_Toyo_CCNET::IsRequestingService(void) throw(THardwareException)
{
    //... TODO: this is very inefficient ...//
    return (ReadLam(0x00ffffff) > 0);
}

int TNetworkCamacBridge_Toyo_CCNET::MiscControlIdOf(const std::string& CommandName) throw (THardwareException)
{
    int ControlId = -1;
    if (CommandName == "executeSingleCamac") {
	ControlId = ControlId_ExecuteSingleCamac;
    }
    else if (CommandName == "executeSingleDaq") {
	ControlId = ControlId_ExecuteSingleDaq;
    }
    else if (CommandName == "initializeCommandBuffer") {
	ControlId = ControlId_InitializeCommandBuffer;
    }
    else if (CommandName == "loadCamacCommand") {
	ControlId = ControlId_LoadCamacCommand;
    }
    else if (CommandName == "loadDaqCommand") {
	ControlId = ControlId_LoadDaqCommand;
    }
    else if (CommandName == "execute") {
	ControlId = ControlId_Execute;
    }
    else if (CommandName == "executePio") {
	ControlId = ControlId_ExecutePio;
    }
    else if (CommandName == "executeDma") {
	ControlId = ControlId_ExecuteDma;
    }
    else if (CommandName == "executeDmaSequence") {
	ControlId = ControlId_ExecuteDmaSequence;
    }
    else {
	throw THardwareException(
	    "CCNET", "unknown command: " + CommandName
	);
    }

    return ControlId;
}

int TNetworkCamacBridge_Toyo_CCNET::MiscControl(int ControlId, int* ArgumentList, int NumberOfArguments) throw (THardwareException)
{
    if (ControlId == ControlId_ExecuteSingleCamac) {
	if (NumberOfArguments < 3) {
	    throw THardwareException(
		"CCNET::executeSingleCamac(int station, int function, int address, [int data, int q, int x])",
		"too few argument[s]"
	    );
	}
	int Station = ArgumentList[0];
	int Function = ArgumentList[1];
	int Address = ArgumentList[2];

	int Data = 0, Q , X;
	if (NumberOfArguments > 3) {
	    Data = ArgumentList[3];
	}

	CCNET_ExecuteSingleCamac(Station, Address, Function, Data, Q, X);

	if (NumberOfArguments > 3) {
	    ArgumentList[3] = Data;
	}
	if (NumberOfArguments > 4) {
	    ArgumentList[4] = Q;
	}
	if (NumberOfArguments > 5) {
	    ArgumentList[5] = X;
	}
    }

    else if (ControlId == ControlId_ExecuteSingleDaq) {
	if (NumberOfArguments < 1) {
	    throw THardwareException(
		"CCNET::executeSingleDaq(int function, [int data, [int status]])",
		"too few argument[s]"
	    );
	}
	int Function = ArgumentList[0];
	int Data = 0;
	if (NumberOfArguments > 1) {
	    Data = ArgumentList[1];
	}
	int Status = 0;

	CCNET_ExecuteSingleDaq(Function, Data, Status);

	if (NumberOfArguments > 1) {
	    ArgumentList[1] = Data;
	}
	if (NumberOfArguments > 2) {
	    ArgumentList[2] = Status;
	}
    }

    else if (ControlId == ControlId_InitializeCommandBuffer) {
	if (_CommandBuffer == 0) {
	    _CommandBuffer = CCNET_CreateBuffer();
	    _ReplyBuffer = CCNET_CreateBuffer();
	}
	CCNET_InitializeCommandBuffer(_CommandBuffer);
    }

    else if (ControlId == ControlId_LoadCamacCommand) {
	if (NumberOfArguments < 3) {
	    throw THardwareException(
		"CCNET::loadCamacCommand(int station, int function, int address, [int data])",
		"too few argument[s]"
	    );
	}
	int Station = ArgumentList[0];
	int Function = ArgumentList[1];
	int Address = ArgumentList[2];
	int Data = 0;
	if (NumberOfArguments > 3) {
	    Data = ArgumentList[3];
	}

	CCNET_LoadCamacCommand(
	    _CommandBuffer, Station, Address, Function, Data
	);
    }

    else if (ControlId == ControlId_LoadDaqCommand) {
	if (NumberOfArguments < 1) {
	    throw THardwareException(
		"CCNET::loadDaqCommand(int function, [int data])",
		"too few argument[s]"
	    );
	}
	int Function = ArgumentList[0];
	int Data = 0;
	if (NumberOfArguments > 1) {
	    Data = ArgumentList[1];
	}

	CCNET_LoadDaqCommand(_CommandBuffer, Function, Data);
    }

    else if (ControlId == ControlId_Execute) {
	CCNET_CompleteCommandBuffer(_CommandBuffer);
	int ReplyLength = CCNET_Execute(_CommandBuffer, _ReplyBuffer);
	_ReplySize = 2 * ReplyLength + 2;
    }
    else if (ControlId == ControlId_ExecutePio) {
	CCNET_CompleteCommandBuffer(_CommandBuffer);
	int ReplyLength = CCNET_ExecutePio(_CommandBuffer, _ReplyBuffer);
	_ReplySize = 2 * ReplyLength + 2;
    }
    else if (ControlId == ControlId_ExecuteDma) {
	CCNET_CompleteCommandBuffer(_CommandBuffer);
	int ReplyLength = CCNET_ExecuteDma(_CommandBuffer, _ReplyBuffer);
	_ReplySize = 2 * ReplyLength + 2;
    }
    else if (ControlId == ControlId_ExecuteDmaSequence) {
	CCNET_CompleteCommandBuffer(_CommandBuffer);
	int ReplyLength = CCNET_ExecuteDmaSequence(
	    _CommandBuffer, _ReplyBuffer
	);
	_ReplySize = 2 * ReplyLength + 2;
    }

    else {
	return 0;
    }

    return 1;
}
