/* controller-Kinoko_Vmedrv.cc */
/* Created by Enomoto Sanshiro on 11 December 1999. */
/* Last updated by Enomoto Sanshiro on 8 July 2001. */


#include <cerrno>
#include <fcntl.h>
#include <unistd.h>
#include <sys/mman.h>
#include <sys/ioctl.h>
#include "vmedrv.h"
#include "RoomDeviceFactory.hh"
#include "controller-Kinoko_Vmedrv.hh"

using namespace std;


static TRoomVmeControllerCreator Creator1(
    "Kinoko_Vmedrv", new TVmeController_Kinoko_Vmedrv()
);
static TRoomVmeControllerCreator Creator2(
    "SBS_617", new TVmeController_Kinoko_Vmedrv()
);
static TRoomVmeControllerCreator Creator3(
    "SBS_618", new TVmeController_Kinoko_Vmedrv()
);
static TRoomVmeControllerCreator Creator4(
    "SBS_620", new TVmeController_Kinoko_Vmedrv()
);


//... for backward compatibiliy ...
static TRoomVmeControllerCreator Creator5(
    "Bit3_617", new TVmeController_Kinoko_Vmedrv()
);
static TRoomVmeControllerCreator Creator6(
    "Bit3_618", new TVmeController_Kinoko_Vmedrv()
);
static TRoomVmeControllerCreator Creator7(
    "Bit3_620", new TVmeController_Kinoko_Vmedrv()
);



static const char *VmedrvDeviceFileName[] = {
    /* this must be consistent with TRoomVmeTarnsferMode */
    /* (declared in RoomVmeAccess.hh) */
    "/dev/vmedrv16d16", "/dev/vmedrv16d32",
    "/dev/vmedrv24d16", "/dev/vmedrv24d32", 
    "/dev/vmedrv32d16", "/dev/vmedrv32d32"
};

static const char *VmedrvDmaDeviceFileName[] = {
    /* this must be consistent with TRoomVmeTarnsferMode */
    /* (declared in RoomVmeAccess.hh) */
    "/dev/vmedrv16d16", "/dev/vmedrv16d32",
    "/dev/vmedrv24d16dma", "/dev/vmedrv24d32dma", 
    "/dev/vmedrv32d16dma", "/dev/vmedrv32d32dma"
};

static const unsigned VmedrvPageSize = 0x1000;
static const unsigned VmedrvPageMask = ~(VmedrvPageSize - 1);

class TVmedrvSpecificData {
  public:
    TVmedrvSpecificData(void);
  public:
    caddr_t AlignedMappedAddress;
    size_t AlignedMapSize;
};

TVmedrvSpecificData::TVmedrvSpecificData(void)
{
    AlignedMappedAddress = 0;
    AlignedMapSize = 0;
}



TVmeController_Kinoko_Vmedrv::TVmeController_Kinoko_Vmedrv(void)
{
    _InterruptEnableCount = 0;
}

TVmeController_Kinoko_Vmedrv::~TVmeController_Kinoko_Vmedrv()
{
}

TRoomVmeController* TVmeController_Kinoko_Vmedrv::CloneController(void)
{
    return new TVmeController_Kinoko_Vmedrv();
}

const char* TVmeController_Kinoko_Vmedrv::DeviceFileName(TRoomVmeTransferMode TransferMode)
{
    return VmedrvDeviceFileName[TransferMode];
}

const char* TVmeController_Kinoko_Vmedrv::DmaDeviceFileName(TRoomVmeTransferMode TransferMode)
{
    return VmedrvDmaDeviceFileName[TransferMode];
}

TRoomVmeAccessProperty* TVmeController_Kinoko_Vmedrv::Open(TRoomVmeTransferMode TransferMode) throw(THardwareException)
{
    TRoomVmeAccessProperty* Property = TRoomVmeController::Open(TransferMode);
    Property->DeviceSpecificData = new TVmedrvSpecificData();

    return Property;
}

void TVmeController_Kinoko_Vmedrv::Close(TRoomVmeAccessProperty* Property)
{
    delete (TVmedrvSpecificData*) Property->DeviceSpecificData;
    TRoomVmeController::Close(Property);
}

caddr_t TVmeController_Kinoko_Vmedrv::Map(TRoomVmeAccessProperty* Property, off_t VmeAddress, size_t MapSize, off_t MapOffset) throw(THardwareException)
{
    TVmedrvSpecificData* SpecificData;
    SpecificData = (TVmedrvSpecificData*) Property->DeviceSpecificData;

    caddr_t MappedAddress = 0;
    caddr_t AlignedMappedAddress = 0;

    off_t AlignedVmeAddress = (VmeAddress + MapOffset) & VmedrvPageMask;
    off_t AlignmentOffset = (VmeAddress + MapOffset) & ~VmedrvPageMask;
    size_t AlignedMapSize = AlignmentOffset + MapSize;
    if (AlignedMapSize < VmedrvPageSize) {
	AlignedMapSize = VmedrvPageSize;
    }

    if (MapSize > 0) {
	caddr_t MapAddress = (caddr_t) 0;  // assigned by kernel
	int MapProtection = PROT_READ | PROT_WRITE;
	int MapFlag = MAP_SHARED;
	AlignedMappedAddress = (caddr_t) mmap(
	    MapAddress,
	    AlignedMapSize,
	    MapProtection,
	    MapFlag,
	    Property->DeviceDescriptor,
	    AlignedVmeAddress
	);
	if (MappedAddress == (caddr_t) MAP_FAILED) {
	    throw THardwareException(
		"TVmeController_Kinoko_Vmedrv::Map(): mmap(2)",
		strerror(errno)
	    );
	}

	MappedAddress = AlignedMappedAddress + AlignmentOffset;
    }

    Property->VmeAddress = VmeAddress;
    Property->MapSize = MapSize;
    Property->MapOffset = MapOffset;
    Property->MappedAddress = MappedAddress;

    SpecificData->AlignedMappedAddress = AlignedMappedAddress;
    SpecificData->AlignedMapSize = AlignedMapSize;

    return MappedAddress;
}

void TVmeController_Kinoko_Vmedrv::Unmap(TRoomVmeAccessProperty* Property) throw(THardwareException)
{
    TVmedrvSpecificData* SpecificData;
    SpecificData = (TVmedrvSpecificData*) Property->DeviceSpecificData;

    if (Property->MappedAddress > 0) {
	int Result = munmap(
	    SpecificData->AlignedMappedAddress, SpecificData->AlignedMapSize
	);
	if (Result < 0) {
	    throw THardwareException(
		"TVmeController_Kinoko_Vmedrv::Map(): munmap(2)",
		strerror(errno)
	    );	     
	}

	Property->MappedAddress = 0;
    }
}

void TVmeController_Kinoko_Vmedrv::RegisterInterruptNotification(TRoomVmeAccessProperty* Property, int SignalId) throw(THardwareException)
{
    Property->InterruptSignalId = SignalId;

    struct vmedrv_interrupt_property_t InterruptProperty;
    InterruptProperty.irq = Property->InterruptNumber;
    InterruptProperty.vector = Property->InterruptVector;
    InterruptProperty.signal_id = SignalId;

    int Status = ioctl(
	Property->DeviceDescriptor, 
	VMEDRV_IOC_REGISTER_INTERRUPT, 
	&InterruptProperty
    );
    if (Status < 0) {
	throw THardwareException(
            "TVmeController_Kinoko_Vmedrv::RegisterInterruptNotification(): "
	    "ioctl(2)",
	    strerror(errno)
        );
    }
}

void TVmeController_Kinoko_Vmedrv::UnregisterInterruptNotification(TRoomVmeAccessProperty* Property) throw(THardwareException)
{
    struct vmedrv_interrupt_property_t InterruptProperty;
    InterruptProperty.irq = Property->InterruptNumber;
    InterruptProperty.vector = Property->InterruptVector;
    InterruptProperty.signal_id = Property->InterruptSignalId;

    int Status = ioctl(
	Property->DeviceDescriptor, 
	VMEDRV_IOC_UNREGISTER_INTERRUPT, 
	&InterruptProperty
    );
    if (Status < 0) {
	throw THardwareException(
            "TVmeController_Kinoko_Vmedrv::UnregisterInterruptNotification(): "
	    "ioctl(2)",
	    strerror(errno)
        );
    }
}

void TVmeController_Kinoko_Vmedrv::EnableInterruptNotification(TRoomVmeAccessProperty* Property) throw(THardwareException)
{
    if (_InterruptEnableCount++ > 0) {
	return;
    }

    int Status = ioctl(
        Property->DeviceDescriptor, VMEDRV_IOC_ENABLE_INTERRUPT
    );
    if (Status < 0) {
	throw THardwareException(
            "TVmeController_Kinoko_Vmedrv::EnableInterruptNotification(): "
	    "ioctl(2)",
	    strerror(errno)
        );
    }
}

void TVmeController_Kinoko_Vmedrv::DisableInterruptNotification(TRoomVmeAccessProperty* Property) throw(THardwareException)
{
    if (--_InterruptEnableCount > 0) {
	return;
    }

    int Status = ioctl(
	Property->DeviceDescriptor, VMEDRV_IOC_DISABLE_INTERRUPT
    );
    if (Status < 0) {
	throw THardwareException(
            "TVmeController_Kinoko_Vmedrv::DisableInterruptNotification(): "
	    "ioctl(2)",
	    strerror(errno)
        );
    }
}

bool TVmeController_Kinoko_Vmedrv::WaitForInterruptNotification(TRoomVmeAccessProperty* Property, unsigned TimeOut) throw(THardwareException)
{
    struct vmedrv_interrupt_property_t InterruptProperty;
    InterruptProperty.irq = Property->InterruptNumber;
    InterruptProperty.vector = Property->InterruptVector;
    InterruptProperty.signal_id = 0;
    InterruptProperty.timeout = TimeOut;

    int Status = ioctl(
	Property->DeviceDescriptor, 
	VMEDRV_IOC_WAIT_FOR_INTERRUPT, 
	&InterruptProperty
    );
    if ((Status < 0) && (errno == ENODEV)) {
	// this error may be due to interrupt never registered -- try.
	Status= ioctl(
	    Property->DeviceDescriptor, 
	    VMEDRV_IOC_REGISTER_INTERRUPT, 
	    &InterruptProperty
	);
	if (Status >= 0) {
	    Status = ioctl(
		Property->DeviceDescriptor, 
		VMEDRV_IOC_WAIT_FOR_INTERRUPT, 
		&InterruptProperty
	    );
	}
    }

    if (Status < 0){
	if (errno == ETIMEDOUT) {
	    ;
	}
	else if (errno == EINTR) {
	    ;
	}
	else {
	    throw THardwareException(
		"TVmeController_Kinoko_Vmedrv::WaitForInterrupt(): "
		"ioctl(2)",
		strerror(errno)
	    );
	}

	return false;
    }
    
    return true;
}

void TVmeController_Kinoko_Vmedrv::SetInterruptNotificationAutodisable(TRoomVmeAccessProperty* Property) throw(THardwareException)
{
    struct vmedrv_interrupt_property_t InterruptProperty;
    InterruptProperty.irq = Property->InterruptNumber;
    InterruptProperty.vector = Property->InterruptVector;
    InterruptProperty.signal_id = Property->InterruptSignalId;

    int Status = ioctl(
	Property->DeviceDescriptor, 
	VMEDRV_IOC_SET_INTERRUPT_AUTODISABLE,
	&InterruptProperty
    );
    if (Status < 0) {
	throw THardwareException(
            "TVmeController_Kinoko_Vmedrv::SetInterruptNotificationAutodisable(): "
	    "ioctl(2)",
	    strerror(errno)
        );
    }
}

bool TVmeController_Kinoko_Vmedrv::IsSignalOnInterruptAvailable(void)
{
    return true;
}
