/* module-LeCroy_1976.cc */
/* Created by Enomoto Sanshiro on 5 December 2005 */
/* Last updated by Enomoto Sanshiro on 9 December 2005 */


#include <string>
#include "RoomDeviceFactory.hh"
#include "RoomModule.hh"
#include "RoomCamacAccess.hh"
#include "module-LeCroy_1976.hh"


static TRoomCamacModuleCreator Creator(
    "LeCroy_1976", new TCamacAdcTdcTester_LeCroy_1976()
);



TCamacAdcTdcTester_LeCroy_1976::TCamacAdcTdcTester_LeCroy_1976(void)
: TRoomCamacModule("CamacAdcTdcTester", "LeCroy_1976")
{
    _Mode = Mode_Charge;
    _ChargeFullScale = 0;
    _TimeFullScale = 0;

    _Amplitude = 0;

    _GateWidthFraction = 0;
    _PulseWidth = 0;
    _TailFallTime = 0;

    _ChargeScaling = 0;
    _TimeScaling = 0;
}

TCamacAdcTdcTester_LeCroy_1976::~TCamacAdcTdcTester_LeCroy_1976()
{
}

TRoomCamacModule* TCamacAdcTdcTester_LeCroy_1976::Clone(void)
{
    return new TCamacAdcTdcTester_LeCroy_1976();
}

int TCamacAdcTdcTester_LeCroy_1976::Initialize(int InitialState) throw(THardwareException)
{
    _Mode = Mode_Charge;

    _ChargeFullScale = 0;
    _ChargeFullScale_pC = 300;

    _TimeFullScale = 0;
    _TimeFullScale_nsec = 9600;

    _GateWidthFraction = 0;
    _GateWidth_nsec = 9610;

    _PulseWidth = 0;
    _PulseWidth_nsec = 20;

    _TailFallTime = 0;

    _Amplitude = 0;

    _ChargeScaling = 1;
    _TimeScaling = 1;

    return 0;
}

int TCamacAdcTdcTester_LeCroy_1976::Finalize(int FinalState) throw(THardwareException)
{
    return 0;
}

int TCamacAdcTdcTester_LeCroy_1976::Read(int Address, int& Data) throw(THardwareException)
{
    int Result = 1;

    if (Address == 0) {
	Data = _Amplitude;
    }
    else if (Address == 1) {
	if (_Mode == Mode_Charge) {
	    Data = (int) (_ChargeFullScale_pC * _ChargeScaling);
	}
	else {
	    Data = (int) (_TimeFullScale_nsec * _TimeScaling);
	}
    }
    else if (Address == 2) {
	Data = _GateWidth_nsec;
    }
    else if (Address == 3) {
	if (_ChargeFullScale_pC == 1200) {
	    Data = (_TailFallTime == 0) ? 1200 : 65;
	}
	else if (_ChargeFullScale_pC == 600) {
	    Data = (_TailFallTime == 0) ? 600 : 35;
	}
	else {
	    Data = (_TailFallTime == 0) ? 350 : 15;
	}
    }
    else if (Address == 4) {
	Data = _PulseWidth_nsec;
    }
    else {
	return 0;
    }

    return Result;
}

int TCamacAdcTdcTester_LeCroy_1976::Clear(int Address) throw (THardwareException)
{
    return Initialize();
}

int TCamacAdcTdcTester_LeCroy_1976::MiscControlIdOf(const std::string& CommandName) throw (THardwareException)
{
    int ControlId = -1;

    if (CommandName == "selectChargeMode") {
	ControlId = ControlId_SelectChargeMode;
    }
    else if (CommandName == "selectTimeMode") {
	ControlId = ControlId_SelectTimeMode;
    }
    else if (CommandName == "setChargeFullScale") {
	ControlId = ControlId_SetChargeFullScale;
    }
    else if (CommandName == "setTimeFullScale") {
	ControlId = ControlId_SetTimeFullScale;
    }
    else if (CommandName == "setAmplitude") {
	ControlId = ControlId_SetAmplitude;
    }
    else if (CommandName == "setGateWidth") {
	ControlId = ControlId_SetGateWidth;
    }
    else if (CommandName == "selectShortPulse") {
	ControlId = ControlId_SelectShortPulse;
    }
    else if (CommandName == "selectLongPulse") {
	ControlId = ControlId_SelectLongPulse;
    }
    else if (CommandName == "selectFastTail") {
	ControlId = ControlId_SelectFastTail;
    }
    else if (CommandName == "selectSlowTail") {
	ControlId = ControlId_SelectSlowTail;
    }
    else if (CommandName == "execute") {
	ControlId = ControlId_Execute;
    }
    else {
	throw THardwareException(
	    "TCamacAdcTdcTester_LeCroy_1976::MiscControlIdOf()",
	    "unknown command: " + CommandName
	);
    }

    return ControlId;
}

int TCamacAdcTdcTester_LeCroy_1976::MiscControl(int ControlId, int* ArgumentList, int NumberOfArguments) throw (THardwareException)
{
    int Result = 1;

    if (ControlId == ControlId_SelectChargeMode) {
	_Mode = Mode_Charge;
    }
    else if (ControlId == ControlId_SelectTimeMode) {
	_Mode = Mode_Time;
    }
    else if (ControlId == ControlId_SetChargeFullScale) {
	Result = SetChargeFullScale(ArgumentList, NumberOfArguments);
    }
    else if (ControlId == ControlId_SetTimeFullScale) {
	Result = SetTimeFullScale(ArgumentList, NumberOfArguments);
    }
    else if (ControlId == ControlId_SetAmplitude) {
	Result = SetAmplitude(ArgumentList, NumberOfArguments);
    }
    else if (ControlId == ControlId_SetGateWidth) {
	Result = SetGateWidth(ArgumentList, NumberOfArguments);
    }
    else if (ControlId == ControlId_SelectShortPulse) {
	_PulseWidth = 0;
	_PulseWidth_nsec = 20;
    }
    else if (ControlId == ControlId_SelectLongPulse) {
	_PulseWidth = 1;
	_PulseWidth_nsec = 100;
    }
    else if (ControlId == ControlId_SelectFastTail) {
	_TailFallTime = 1;
    }
    else if (ControlId == ControlId_SelectSlowTail) {
	_TailFallTime = 0;
    }
    else if (ControlId == ControlId_Execute) {
	Result = Execute(ArgumentList, NumberOfArguments);
    }
    else {
	Result = 0;
    }

    return Result;
}

int TCamacAdcTdcTester_LeCroy_1976::SetChargeFullScale(int* ArgumentList, int NumberOfArguments) throw (THardwareException)
{
    if (NumberOfArguments < 1) {
	throw THardwareException(
	    _ModelName + "::setChargeFullScale(int full_scale)", 
	    "too few argument[s]"
	);
    }

    int FullScale_pC = ArgumentList[0];
    if (FullScale_pC <= 0) {
	throw THardwareException(
	    _ModelName + "::setChargeFullScale(int full_scale)", 
	    "invalid full_scale value (negative)"
	);
    }
    else if (FullScale_pC <= 300) {
	_ChargeFullScale = 0x00;
	_ChargeFullScale_pC = 300;
    }
    else if (FullScale_pC <= 600) {
	_ChargeFullScale = 0x01;
	_ChargeFullScale_pC = 600;
    }
    else if (FullScale_pC <= 1200) {
	_ChargeFullScale = 0x03;
	_ChargeFullScale_pC = 1200;
    }
    else {
	throw THardwareException(
	    _ModelName + "::setChargeFullScale(int full_scale)", 
	    "full_scale too large"
	);
    }

    _ChargeScaling = (double) FullScale_pC / _ChargeFullScale_pC;

    return 1;
}

int TCamacAdcTdcTester_LeCroy_1976::SetTimeFullScale(int* ArgumentList, int NumberOfArguments) throw (THardwareException)
{
    if (NumberOfArguments < 1) {
	throw THardwareException(
	    _ModelName + "::setTimeFullScale(int full_scale)", 
	    "too few argument[s]"
	);
    }

    int FullScale_nsec = ArgumentList[0];
    if (FullScale_nsec <= 0) {
	throw THardwareException(
	    _ModelName + "::setTimeFullScale(int full_scale)", 
	    "invalid full_scale value (negative)"
	);
    }
    else if (FullScale_nsec <= 150) {
	_TimeFullScale = 0x3f;
	_TimeFullScale_nsec = 150;
    }
    else if (FullScale_nsec <= 300) {
	_TimeFullScale = 0x1f;
	_TimeFullScale_nsec = 300;
    }
    else if (FullScale_nsec <= 600) {
	_TimeFullScale = 0x0f;
	_TimeFullScale_nsec = 600;
    }
    else if (FullScale_nsec <= 1200) {
	_TimeFullScale = 0x07;
	_TimeFullScale_nsec = 1200;
    }
    else if (FullScale_nsec <= 2400) {
	_TimeFullScale = 0x03;
	_TimeFullScale_nsec = 2400;
    }
    else if (FullScale_nsec <= 4800) {
	_TimeFullScale = 0x01;
	_TimeFullScale_nsec = 4800;
    }
    else if (FullScale_nsec <= 9600) {
	_TimeFullScale = 0x00;
	_TimeFullScale_nsec = 9600;
    }
    else {
	throw THardwareException(
	    _ModelName + "::setTimeFullScale(int full_scale)", 
	    "full_scale too large"
	);
    }

    _TimeScaling = (double) FullScale_nsec / _TimeFullScale_nsec;

    return 1;
}

int TCamacAdcTdcTester_LeCroy_1976::SetAmplitude(int* ArgumentList, int NumberOfArguments) throw (THardwareException)
{
    if (NumberOfArguments < 1) {
	throw THardwareException(
	    _ModelName + "::setAmplitude(int amplitude)",
	    "too few argument[s]"
	);
    }

    _Amplitude = ArgumentList[0] & 0x0000ffff;

    return 1;
}

int TCamacAdcTdcTester_LeCroy_1976::SetGateWidth(int* ArgumentList, int NumberOfArguments) throw (THardwareException)
{
    if (NumberOfArguments < 1) {
	throw THardwareException(
	    _ModelName + "::setGateWidth(int width_nsec)",
	    "too few argument[s]"
	);
    }

    int GateWidth = ArgumentList[0] - 10;
    if (GateWidth > 9600) {
	throw THardwareException(
	    _ModelName + "::setGateWidth(int width_nsec)",
	    "too large gate width (must be less than 9600)"
	);
    }
    else if (GateWidth < 0) {
	GateWidth = 0;
    }

    int TimeFullScale;
    if (GateWidth > 4800) {
	TimeFullScale = 9600;
    }
    else if (GateWidth > 2400) {
	TimeFullScale = 4800;
    }
    else if (GateWidth > 1200) {
	TimeFullScale = 2400;
    }
    else if (GateWidth > 600) {
	TimeFullScale = 1200;
    }
    else if (GateWidth > 300) {
	TimeFullScale = 600;
    }
    else if (GateWidth > 150) {
	TimeFullScale = 300;
    }
    else {
	TimeFullScale = 150;
    }
    SetTimeFullScale(&TimeFullScale, 1);

    _GateWidthFraction = 255 - (255 * GateWidth) / TimeFullScale;
    _GateWidth_nsec = GateWidth + 10;

    return 1;
}

int TCamacAdcTdcTester_LeCroy_1976::Execute(int* ArgumentList, int NumberOfArguments) throw (THardwareException)
{
    int ScaledAmplitude;
    int ControlRegister;
    if (_Mode == Mode_Charge) {
	ControlRegister = Bit(12) | Bit(7);
	ScaledAmplitude = (int) (_Amplitude * _ChargeScaling);
    }
    else {
	ControlRegister = Bit(12) | Bit(8);
	double T = _TimeScaling * (1 - _Amplitude / 65535.0);
	ScaledAmplitude = (int) (65535.0 * (1 - T));
    }

    ControlRegister |= _TimeFullScale << 0;
    ControlRegister |= _PulseWidth << 6;
    ControlRegister |= _ChargeFullScale << 9;
    ControlRegister |= _TailFallTime << 11;

    int Address;
    Transact(fnWrite, Address = 1, ControlRegister);

    if (_Mode == Mode_Charge) {
	Transact(fnWrite, Address = 2, _GateWidthFraction);
    }

    Transact(fnWrite, Address = 0, ScaledAmplitude);

    return 1;
}


#if 0
datasource CamacAdcTester {
    int tester_station = 22;
    int adc_station = 17;
    int adc_channel = #0;

    ChannelList tester_channels;
    tester_channels.add(0, "amplitude");
    tester_channels.add(1, "full_scale");
    tester_channels.add(2, "gate_width");
    tester_channels.add(3, "tail_fall_time");

    CamacCrate crate;
    CamacController controller("Toyo-CC7x00");
    CamacModule tester("LeCroy-1976");
    CamacModule adc("Generic-Standard");

    crate.installController(controller);
    crate.installModule(tester, tester_station);
    crate.installModule(adc, adc_station);

    SoftwareModule timer("IntervalTimer");
    Register amplitude;

    on run_begin {
	tester.selectChargeMode();
	tester.setChargeFullScale(600);
	tester.setGateWidth(200);
	tester.selectFastTail();

	amplitude.load(0);
	timer.setInterval(0, 10000);
    }

    on trigger (timer) {
	amplitude.add(0x10);
	when (amplitude > 0xffff) {
	    //terminate();
	    amplitude.load(0x10);
	}
	tester.setAmplitude(amplitude);
	tester.execute();
    }

    on trigger (adc) {
	adc.read(adc_channel);
	adc.clear();

        // read test pulse setting
	tester.tagRead(tester_channels);  
    }
}


datasource CamacTdcTester {
    int tester_station = 22;
    int tdc_station = 15;
    int tdc_channel = #0;

    ChannelList tester_channels;
    tester_channels.add(0, "amplitude");
    tester_channels.add(1, "full_scale");

    CamacCrate crate;
    CamacController controller("Toyo-CC7x00");
    CamacModule tester("LeCroy-1976");
    CamacModule tdc("Generic-Standard");

    crate.installController(controller);
    crate.installModule(tester, tester_station);
    crate.installModule(tdc, tdc_station);

    SoftwareModule timer("IntervalTimer");
    Register amplitude;

    on run_begin {
	tester.selectTimeMode();
	tester.setTimeFullScale(4800);
	tester.selectShortPulse();

	amplitude.load(0);
	timer.setInterval(0, 10000);
    }

    on trigger (timer) {
	amplitude.add(0x10);
	when (amplitude > 0xffff) {
	    //terminate();
            amplitude.load(0x10);
	}

	tester.setAmplitude(amplitude);
	tester.execute();
    }

    on trigger (tdc) {
	tdc.read(tdc_channel);
	tdc.clear();

        // read test pulse setting
	tester.tagRead(tester_channels);  
    }
}

#endif
