/* module-RawFile.cc */
/* Created by Enomoto Sanshiro on 25 August 2009. */
/* Last updated by Enomoto Sanshiro on 25 August 2009. */


#include <string>
#include <sstream>
#include <algorithm>
#include <fcntl.h>
#include <unistd.h>
#include <errno.h>
#include <poll.h>

#include "RoomDefs.hh"
#include "RoomModule.hh"
#include "RoomSoftwareDevice.hh"
#include "RoomDeviceFactory.hh"
#include "module-RawFile.hh"

using namespace std;


static TRoomSoftwareModuleCreator Creator1(
    "RawFile", new TSoftwareModule_RawFile()
);


TSoftwareModule_RawFile::TSoftwareModule_RawFile(void)
: TRoomSoftwareModule("RawFileAccess", "Software_RawFile")
{
    _File = -1;
    _FileName = "/dev/zero";
    _BlockSize = 256;
}

TSoftwareModule_RawFile::~TSoftwareModule_RawFile()
{
}

TRoomSoftwareModule* TSoftwareModule_RawFile::Clone(void)
{
    return new TSoftwareModule_RawFile();
}

bool TSoftwareModule_RawFile::ProcessCommand(const string& Command, const vector<string>& ArgumentList) throw(THardwareException)
{
    if (Command == "setFile") {
	if (ArgumentList.size() < 1) {
	    throw THardwareException(
		"RawFile::setFile(string FileName)", "too few argument"
	    );
	}
	_FileName = ArgumentList[0];
    }
    else if (Command == "setBlockSize") {
	if (ArgumentList.size() < 1) {
	    throw THardwareException(
		"RawFile::setBlockSize(int Size)", "too few argument"
	    );
	}
	if (! (istringstream(ArgumentList[0]) >> _BlockSize)) {
	    throw THardwareException(
		"RawFile::setBlockSize(int Size)", "invalid argument"
	    );
	}
    }
    else {
	return TRoomSoftwareModule::ProcessCommand(Command, ArgumentList);
    }

    return true;
}

int TSoftwareModule_RawFile::Initialize(int InitialState) throw(THardwareException)
{
    _File = open(_FileName.c_str(), O_RDONLY, 0);
    if (_File < 0) {
	throw THardwareException(
	    "RawFile::Open()",
	    "open(2): " + string(strerror(errno)) + ": " + _FileName
	);
    }

    _IsEnd = false;

    return 1;
}

int TSoftwareModule_RawFile::Finalize(int FinalState) throw(THardwareException)
{
    if (_File >= 0) {
	close(_File);
    }

    return 1;
}

int TSoftwareModule_RawFile::Clear(int Address) throw(THardwareException)
{
    return 1;
}

bool TSoftwareModule_RawFile::HasData(int Address) throw(THardwareException)
{
    if (_IsEnd) {
	return false;
    }

    return WaitData(0);
}

bool TSoftwareModule_RawFile::WaitData(unsigned TimeOut_sec) throw(THardwareException)
{
    if (_IsEnd) {
	return false;
    }

    pollfd PollFd;
    PollFd.fd = _File;
    PollFd.events = POLLIN;

    int NumberOfEvents = poll(&PollFd, 1, TimeOut_sec);
    if (NumberOfEvents < 0) {
	if (errno == EINTR) {
	    return false;
	}
	else {
	    throw THardwareException(
		"RawFile::WaitData()",
		"poll(2): " + string(strerror(errno))
	    );
	}
    }

    if (NumberOfEvents == 0) {
	return false;
    }

    return (PollFd.revents & POLLIN) != 0;
}


int TSoftwareModule_RawFile::NextDataBlockSize(int Address) throw(THardwareException)
{
    return _BlockSize;
}

int TSoftwareModule_RawFile::BlockRead(int Address, void *Data, int MaxSize) throw(THardwareException)
{
    int ReadLength = read(_File, Data, min(MaxSize, _BlockSize));

    if (ReadLength == 0) {
	_IsEnd = true;
    }

    if (ReadLength < 0) {
	if (errno != EINTR) {
	    throw THardwareException(
		"RawFile::BlockRead()",
		"read(2): " + string(strerror(errno))
	    );
	}
	ReadLength = 0;
    }
    
    return ReadLength;
}
