/* module-SimulatedModules.cc */
/* Created by Enomoto Sanshiro on 23 April 1998. */
/* Last updated by Enomoto Sanshiro on 8 July 2001. */


#include <iostream>
#include <cmath>
#include <cstdlib>
#include <time.h>
#include "RoomDeviceFactory.hh"
#include "RoomSoftwareDevice.hh"
#include "module-SimulatedModules.hh"

using namespace std;


static TRoomSoftwareModuleCreator Creator1(
    "PoissonGenerator", new TPoissonGenerator()
);
static TRoomSoftwareModuleCreator Creator2(
    "GaussianGenerator", new TGaussianGenerator()
);
static TRoomSoftwareModuleCreator Creator3(
    "ExponentialGenerator", new TExponentialGenerator()
);
static TRoomSoftwareModuleCreator Creator4(
    "WaveGenerator", new TWaveGenerator()
);
static TRoomSoftwareModuleCreator Creator5(
    "PmtSimulator", new TPmtSimulator()
);


static void InitializeRandom(void)
{
    srand48(time(NULL));
}

static double Poisson(double Lambda)
{
    int Count = -1;
    while (Lambda > 0) {
	Count++;

	double Step = drand48();
	if (Step == 0) {
	    break;
	}

	Lambda += log(Step);
    }

    return Count;
}

static double Gaussian(double Mean, double Deviation)
{
    double N = sqrt(-2.0 * log(drand48())) * cos(2 * M_PI * drand48());
    return Mean + Deviation * N;
}

static double Exponential(double Lambda)
{
    double T;
    do {
	T = drand48();
    } while (T == 0);

    return -log(T) / Lambda;
}



TPoissonGenerator::TPoissonGenerator(int NumberOfChannels, int FullScale)
: TRoomSoftwareModule("SimPoissonGenerator", "Software_PoissonGenerator")
{
    _NumberOfChannels = NumberOfChannels;
    _FullScale = FullScale;
    _Mean = 12.0;

    InitializeRandom();
    Clear();
}

TPoissonGenerator::~TPoissonGenerator()
{
}
    
TRoomSoftwareModule* TPoissonGenerator::Clone(void)
{
    return new TPoissonGenerator(_NumberOfChannels, _FullScale);
}

int TPoissonGenerator::Clear(int Address) throw(THardwareException)
{
    return 1;
}

int TPoissonGenerator::Read(int Address, int &Data) throw(THardwareException)
{
    float Gain = _FullScale / (3.0 * _Mean);
    Data = (int) (Gain * (Poisson(_Mean) + drand48()));

    if (Data >= _FullScale) {
	Data = _FullScale - 1;
    }

    return 1;
}

int TPoissonGenerator::NumberOfChannels(void) throw(THardwareException)
{
    return _NumberOfChannels;
}



TGaussianGenerator::TGaussianGenerator(int NumberOfChannels, int FullScale)
: TRoomSoftwareModule("SimGaussianGenerator", "Software_GaussianGenerator")
{
    _NumberOfChannels = NumberOfChannels;
    _FullScale = FullScale;

    _Mean = _FullScale / 2;
    _Deviation = _FullScale / 8;

    InitializeRandom();
    Clear();
}

TGaussianGenerator::~TGaussianGenerator()
{
}
    
TRoomSoftwareModule* TGaussianGenerator::Clone(void)
{
    return new TGaussianGenerator(_NumberOfChannels, _FullScale);
}

int TGaussianGenerator::Clear(int Address) throw(THardwareException)
{
    return 1;
}

int TGaussianGenerator::Read(int Address, int &Data) throw(THardwareException)
{
    Data = (int) Gaussian(_Mean, _Deviation);

    if (Data >= _FullScale) {
	Data = _FullScale - 1;
    }
    if (Data < 0) {
	Data = 0;
    }

    return 1;
}

int TGaussianGenerator::NumberOfChannels(void) throw(THardwareException)
{
    return _NumberOfChannels;
}



TExponentialGenerator::TExponentialGenerator(int NumberOfChannels, int FullScale)
: TRoomSoftwareModule("SimExponentialGenerator", "Software_ExponentialGenerator")
{
    _NumberOfChannels = NumberOfChannels;
    _FullScale = FullScale;

    _Lambda = 1.0 / (_FullScale / 4);

    InitializeRandom();
    Clear();
}

TExponentialGenerator::~TExponentialGenerator()
{
}
    
TRoomSoftwareModule* TExponentialGenerator::Clone(void)
{
    return new TExponentialGenerator(_NumberOfChannels, _FullScale);
}

int TExponentialGenerator::Clear(int Address) throw(THardwareException)
{
    return 1;
}

int TExponentialGenerator::Read(int Address, int &Data) throw(THardwareException)
{
    Data = (int) Exponential(_Lambda);

    if (Data >= _FullScale) {
	Data = _FullScale - 1;
    }
    if (Data < 0) {
	Data = 0;
    }

    return 1;
}

int TExponentialGenerator::NumberOfChannels(void) throw(THardwareException)
{
    return _NumberOfChannels;
}



TWaveGenerator::TWaveGenerator(int NumberOfChannels, int NumberOfSamples, int FullScale)
: TRoomSoftwareModule("SimWaveGenerator", "Software_WaveGenerator")
{
    _NumberOfChannels = NumberOfChannels;
    _NumberOfSamples = NumberOfSamples;

    _FullScale = FullScale;
    _NoiseLevel = 0.2;

    _ReadoutCount = new int[NumberOfChannels];

    InitializeRandom();
    Clear();
}

TWaveGenerator::~TWaveGenerator()
{
    delete[] _ReadoutCount;
}
    
TRoomSoftwareModule* TWaveGenerator::Clone(void)
{
    return new TWaveGenerator(_NumberOfChannels, _NumberOfSamples, _FullScale);
}

int TWaveGenerator::Clear(int Address) throw(THardwareException)
{
    if (Address < 0) {
	for (int Address = 0; Address < _NumberOfChannels; Address++) {
	    _ReadoutCount[Address] = 0;
	}
    }
    else if (Address < _NumberOfChannels) {
	_ReadoutCount[Address] = 0;
    }
    else {
	return -1;
    }

    return 0;
}

int TWaveGenerator::Read(int Address, int &Data) throw(THardwareException)
{
    if (_ReadoutCount[Address] >= _NumberOfSamples) {
	return 0;
    }
    else {
	_ReadoutCount[Address]++;
    }
    
    double Phase = (double) _ReadoutCount[Address] / _NumberOfSamples;
    double Value = Gaussian(sin(2.0 * Phase * M_PI), _NoiseLevel);
    Data = (int) (_FullScale / 3.0 * Value + _FullScale / 2);

    if (Data >= _FullScale) {
	Data = _FullScale - 1;
    }
    else if (Data < 0) {
	Data = 0;
    }

    return 1;
}

int TWaveGenerator::NumberOfChannels(void) throw(THardwareException)
{
    return _NumberOfChannels;
}

int TWaveGenerator::ElementDepth(void)
{
    return _NumberOfSamples;
}

int TWaveGenerator::AddressBitLength(void)
{
    int Length = 0;
    while ((0x0001 << Length) < _NumberOfChannels) {
	Length++;
    }

    return Length;
}

int TWaveGenerator::NextNumberOfDataElements(int Address) throw(THardwareException)
{
    return _NumberOfSamples;
}

int TWaveGenerator::MiscControlIdOf(const string& Command) throw (THardwareException)
{
    int ControlId = -1;
    if (Command == "setSamplingDepth") {
        ControlId = ControlId_SetSamplingDepth;
    }
    else if (Command == "setFullScale") {
        ControlId = ControlId_SetFullScale;
    }
    else if (Command == "setNoiseLevel") {
        ControlId = ControlId_SetNoiseLevel;
    }
    else {
	throw THardwareException(
	    "TWaveGenerator::MiscControlIdOf()", "unknown command: " + Command
	);
    }

    return ControlId;
}

int TWaveGenerator::MiscControl(int ControlId, int* ArgumentList, int NumberOfArguments) throw (THardwareException)
{
    if (ControlId == ControlId_SetSamplingDepth) {
	if (NumberOfArguments < 1) {
	    throw THardwareException(
		"TWaveGenerator::MiscControl(): setSamplingDepth", 
		"too few argument[s]"
	    );
	}
	_NumberOfSamples = ArgumentList[0];
    }
    else if (ControlId == ControlId_SetFullScale) {
	if (NumberOfArguments < 1) {
	    throw THardwareException(
		"TWaveGenerator::MiscControl(): setFullScale", 
		"too few argument[s]"
	    );
	}
	_FullScale = ArgumentList[0];
    }
    else if (ControlId == ControlId_SetNoiseLevel) {
	if (NumberOfArguments < 1) {
	    throw THardwareException(
		"TWaveGenerator::MiscControl(): setNoiseLevel", 
		"too few argument[s]"
	    );
	}
	_NoiseLevel = ArgumentList[0];
    }
    else {
	return 0;
    }

    return 1;
}



TPmtSimulator::TPmtSimulator(int FullScale)
: TRoomSoftwareModule("SimPmtSimulator", "Software_PmtSimulator")
{
    _FullScale = FullScale;
}

TPmtSimulator::~TPmtSimulator()
{
}

TRoomSoftwareModule* TPmtSimulator::Clone(void)
{
    return new TPmtSimulator(_FullScale);
}

int TPmtSimulator::Clear(int Address) throw(THardwareException)
{
    return 0;
}

int TPmtSimulator::Read(int Address, int &Data) throw(THardwareException)
{
    static const int Pedestal = 120;
    static const double NoiseRatio = 0.4;
    static const double TwoHitRatio = 0.1;
    static const double Lambda = 12;

    double RawValue;
    if (drand48() < NoiseRatio) {
	RawValue = fabs(Gaussian(0, Lambda / 6));
    }
    else if (drand48() < TwoHitRatio) {
	RawValue = Poisson(2.0 * Lambda) + drand48();
    }
    else {
	RawValue = Poisson(Lambda) + drand48();
    }

    float Gain = _FullScale / (3.0 * Lambda);
    Data = (int) (RawValue * Gain);

    if (Data >= _FullScale) {
	Data = _FullScale - 1;
    }
    else if (Data < Pedestal) {
	return Read(Address, Data);
    }

    return 1;
}

int TPmtSimulator::NumberOfChannels(void) throw(THardwareException)
{
    return 32;
}

int TPmtSimulator::AddressBitLength(void) throw(THardwareException)
{
    return 5;
}

int TPmtSimulator::DataBitLength(void) throw(THardwareException)
{
    return 16;
}
