/* KinokoAnalysisScriptStatements.cc */
/* Created by Enomoto Sanshiro on 26 December 2000. */
/* Last updated by Enomoto Sanshiro on 23 September 2001. */


#include <string>
#include <vector>
#include "ParaParser.hh"
#include "KinokoAnalysisSequenceBuilder.hh"
#include "KinokoAnalysisAction.hh"
#include "KinokoAnalysisMessenger.hh"
#include "KinokoAnalysisScriptStatement.hh"

using namespace std;


TKinokoAnalysisScriptAnalysisStatement::TKinokoAnalysisScriptAnalysisStatement(TKinokoAnalysisSequenceBuilder* AnalysisSequenceBuilder)
{
    _AnalysisSequenceBuilder = AnalysisSequenceBuilder;
 
    _DataSourceNameExpression = 0;
    _Statement = 0;
}

TKinokoAnalysisScriptAnalysisStatement::~TKinokoAnalysisScriptAnalysisStatement()
{
    delete _Statement;
    delete _DataSourceNameExpression;
}

TParaStatement* TKinokoAnalysisScriptAnalysisStatement::Clone(void)
{
    return new TKinokoAnalysisScriptAnalysisStatement(_AnalysisSequenceBuilder);
}

string TKinokoAnalysisScriptAnalysisStatement::FirstToken(void) const
{
    return string("analysis");
}

void TKinokoAnalysisScriptAnalysisStatement::Parse(TParaTokenizer* Tokenizer, TParaStatementParser* StatementParser, TParaSymbolTable* SymbolTable) throw(TScriptException)
{
    TParaExpressionParser* ExpressionParser;
    ExpressionParser = StatementParser->ExpressionParser();

    Tokenizer->Next().MustBe(FirstToken());

    if (Tokenizer->LookAhead().Is("(")) {
        Tokenizer->Next().MustBe("(");
	_DataSourceNameExpression = ExpressionParser->Parse(Tokenizer, SymbolTable);
	Tokenizer->Next().MustBe(")");
    }
    else {
        _DataSourceNameExpression = 0;
    }
    
    _Statement = StatementParser->Parse(Tokenizer, SymbolTable);
}

TParaStatement::TExecResult TKinokoAnalysisScriptAnalysisStatement::Execute(TParaSymbolTable* SymbolTable) throw(TScriptException)
{
    if (_DataSourceNameExpression != 0) {
        _DataSourceName = _DataSourceNameExpression->Evaluate(SymbolTable).AsString();
    }
    else {
        _DataSourceName = "";
    }

    _AnalysisSequenceBuilder->OpenRootSequence(_DataSourceName);
    TExecResult Result = _Statement->Execute(SymbolTable);
    _AnalysisSequenceBuilder->CloseRootSequence();

    return Result;
}



TKinokoAnalysisScriptInvokeStatement::TKinokoAnalysisScriptInvokeStatement(TKinokoAnalysisSequenceBuilder* AnalysisSequenceBuilder)
{
    _AnalysisSequenceBuilder = AnalysisSequenceBuilder;
}

TKinokoAnalysisScriptInvokeStatement::~TKinokoAnalysisScriptInvokeStatement()
{
    for (unsigned i = 0; i < _EventArgumentExpressionList.size(); i++) {
	delete _EventArgumentExpressionList[i];
    }
}

TParaStatement* TKinokoAnalysisScriptInvokeStatement::Clone(void)
{
    return new TKinokoAnalysisScriptInvokeStatement(_AnalysisSequenceBuilder);
}

string TKinokoAnalysisScriptInvokeStatement::FirstToken(void) const
{
    return string("invoke");
}

void TKinokoAnalysisScriptInvokeStatement::Parse(TParaTokenizer* Tokenizer, TParaStatementParser* StatementParser, TParaSymbolTable* SymbolTable) throw(TScriptException)
{
    TParaExpressionParser* ExpressionParser;
    ExpressionParser = StatementParser->ExpressionParser();

    Tokenizer->Next().MustBe(FirstToken());

    _EventName = Tokenizer->Next().AsString();
    if (Tokenizer->LookAhead().Is("(")) {
	_EventArgumentExpressionList = ExpressionParser->ParseExpressionList(
	    Tokenizer, SymbolTable, "(", ")", ","
	);
    }

    Tokenizer->Next().MustBe(";");
}

TParaStatement::TExecResult TKinokoAnalysisScriptInvokeStatement::Execute(TParaSymbolTable* SymbolTable) throw(TScriptException)
{
    string EventName = _EventName;

    vector<string> EventArgumentList;
    for (unsigned i = 0; i < _EventArgumentExpressionList.size(); i++) {
	EventArgumentList.push_back(
	    _EventArgumentExpressionList[i]->Evaluate(SymbolTable).AsString()
	);
    }

    TKinokoAnalysisAction* InvokeAction =  new TKinokoAnalysisInvokeAction(
	_AnalysisSequenceBuilder->EventEmitter(), EventName, EventArgumentList
    );
    try {
	_AnalysisSequenceBuilder->AddAction(InvokeAction);
    }
    catch (TKinokoException &e) {
	delete InvokeAction;
	throw TScriptException("invoke: " + e.Message());
    }

    return TParaStatement::TExecResult();
}



TKinokoAnalysisScriptWhenStatement::TKinokoAnalysisScriptWhenStatement(TKinokoAnalysisSequenceBuilder* AnalysisSequenceBuilder)
{
    _AnalysisSequenceBuilder = AnalysisSequenceBuilder;

    _ConditionExpression = 0;
    _Statement = 0;
}

TKinokoAnalysisScriptWhenStatement::~TKinokoAnalysisScriptWhenStatement()
{
    delete _Statement;
    delete _ConditionExpression;
}

TParaStatement* TKinokoAnalysisScriptWhenStatement::Clone(void)
{
    return new TKinokoAnalysisScriptWhenStatement(_AnalysisSequenceBuilder);
}

string TKinokoAnalysisScriptWhenStatement::FirstToken(void) const
{
    return string("when");
}

void TKinokoAnalysisScriptWhenStatement::Parse(TParaTokenizer* Tokenizer, TParaStatementParser* StatementParser, TParaSymbolTable* SymbolTable) throw(TScriptException)
{
    TParaExpressionParser* ExpressionParser;
    ExpressionParser = StatementParser->ExpressionParser();

    Tokenizer->Next().MustBe("when");
    Tokenizer->Next().MustBe("(");
    _ConditionExpression = ExpressionParser->Parse(Tokenizer, SymbolTable);
    Tokenizer->Next().MustBe(")");

    _Statement = StatementParser->Parse(Tokenizer, SymbolTable);
}

TParaStatement::TExecResult TKinokoAnalysisScriptWhenStatement::Execute(TParaSymbolTable* SymbolTable) throw(TScriptException)
{
    TParaValue& ConditionValue = _ConditionExpression->Evaluate(SymbolTable);

    if (! ConditionValue.IsObject("AnalysisCondition")) {
	throw TScriptException(
	    _ConditionExpression->Position() +
	    "when: invalid condition expression"
	);
    }

    TKinokoDataChecker* DataChecker;
    try {
	DataChecker = ((TKinokoAnalysisConditionMessenger*) ConditionValue.AsObject())->GetDataChecker();
    }
    catch (TScriptException &e) {
	throw TScriptException(_ConditionExpression->Position() + e.Message());
    }

    TKinokoAnalysisSequence* Sequence = new TKinokoConditionalAnalysisSequence(
	DataChecker
    );

    _AnalysisSequenceBuilder->OpenSubSequence(Sequence);
    TExecResult Result = _Statement->Execute(SymbolTable);
    _AnalysisSequenceBuilder->CloseSubSequence();

    return Result;
}
