/* KinokoAnalysisSequence.cc */
/* Created by Enomoto Sanshiro on 17 September 2001 */
/* Last updated by Enomoto Sanshiro on 17 September 2001. */


#include <iostream>
#include <string>
#include "KinokoDataAnalyzer.hh"
#include "KinokoDataChecker.hh"
#include "KinokoAnalysisSequence.hh"

using namespace std;


TKinokoAnalysisSequence::TKinokoAnalysisSequence(void)
{
    _SequenceListCapacity = 16;
    _NumberOfSequences = 0;

    _SequenceList = new TKinokoAnalysisSequence* [_SequenceListCapacity];
}

TKinokoAnalysisSequence::~TKinokoAnalysisSequence()
{
    delete[] _SequenceList;
}

int TKinokoAnalysisSequence::ProcessTrailerPacket(void* Packet, TKinokoDataSource* DataSource) throw(TKinokoException)
{
    return TKinokoDataAnalyzerList::ProcessTrailerPacket(Packet, DataSource);
}

void TKinokoAnalysisSequence::AddSequence(TKinokoAnalysisSequence* Sequence)
{
    if (_NumberOfSequences >= _SequenceListCapacity) {
	TKinokoAnalysisSequence** OldSequenceList = _SequenceList;
	_SequenceListCapacity *= 2;
	_SequenceList = new TKinokoAnalysisSequence* [_SequenceListCapacity];
	for (int i = 0; i < _NumberOfSequences; i++) {
	    _SequenceList[i] = OldSequenceList[i];
	}
	delete[] OldSequenceList;
    }

    TKinokoDataAnalyzerList::AddAnalyzer(Sequence);
    _SequenceList[_NumberOfSequences++] = Sequence;
}

void TKinokoAnalysisSequence::Flush(void)
{
    for (int i = 0; i < _NumberOfSequences; i++) {
	_SequenceList[i]->Flush();
    }
}

void TKinokoAnalysisSequence::Dump(ostream& os, const string& Indent)
{
    os << Indent << ".sequence" << endl;
    TKinokoDataAnalyzerList::Dump(os, Indent + "->\t");
    os << Indent << ".end" << endl;
}



TKinokoSingleEventAnalysisSequence::TKinokoSingleEventAnalysisSequence(void)
{
    _ProcessorState = ProcessorState_WaitingForNewEvent;
    _NumberOfProcessedDataPackets = 0;
    _IsLastEventCompleted = true;
}

TKinokoSingleEventAnalysisSequence::~TKinokoSingleEventAnalysisSequence()
{
}

int TKinokoSingleEventAnalysisSequence::ProcessDataPacket(void* DataPacket, TKinokoDataSource* DataSource, TKinokoDataSection* DataSection) throw(TKinokoException)
{
    _IsLastEventCompleted = false;

    if (_ProcessorState == ProcessorState_ReadingEventData) {
	int Result = TKinokoAnalysisSequence::ProcessDataPacket(
	    DataPacket, DataSource, DataSection
	);
	if (Result > 0) {
	    _NumberOfProcessedDataPackets++;
	}
	return Result;
    }
    else {
	return 0;
    }
}

int TKinokoSingleEventAnalysisSequence::ProcessTrailerPacket(void* Packet, TKinokoDataSource* DataSource) throw(TKinokoException)
{
    _IsLastEventCompleted = true;

    if (_ProcessorState == ProcessorState_WaitingForNewEvent) {
        _ProcessorState = ProcessorState_ReadingEventData;
    }
    else if (_ProcessorState == ProcessorState_ReadingEventData) {
	if (_NumberOfProcessedDataPackets > 0) {
	    _ProcessorState = ProcessorState_HoldingEventData;
	    return TKinokoAnalysisSequence::ProcessTrailerPacket(
		Packet, DataSource
	    );
	}
    }

    return 0;
}

void TKinokoSingleEventAnalysisSequence::Flush(void)
{
    // if single-event sequence is flushed() before an event is completed,
    // ignore the flush()
    if (_ProcessorState != ProcessorState_HoldingEventData) {
	return;
    }

    TKinokoAnalysisSequence::Flush();
    TKinokoAnalysisSequence::Reset();
    _NumberOfProcessedDataPackets = 0;
    
    if (_IsLastEventCompleted) {
	_ProcessorState = ProcessorState_ReadingEventData;
    }
    else {
	_ProcessorState = ProcessorState_WaitingForNewEvent;
    }
}

void TKinokoSingleEventAnalysisSequence::Dump(ostream& os, const string& Indent)
{
    os << Indent << ".sequence single_event" << endl;
    TKinokoDataAnalyzerList::Dump(os, Indent + "->\t");
    os << Indent << ".end" << endl;
}



TKinokoConditionalAnalysisSequence::TKinokoConditionalAnalysisSequence(TKinokoDataChecker* DataChecker)
{
    _DataChecker = DataChecker;
}

TKinokoConditionalAnalysisSequence::~TKinokoConditionalAnalysisSequence()
{
    delete _DataChecker;
}

void TKinokoConditionalAnalysisSequence::ReadDataSource(TKinokoDataSource* DataSource) throw(TKinokoException)
{
    TKinokoAnalysisSequence::ReadDataSource(DataSource);
    _DataChecker->ReadDataSource(DataSource);
}

int TKinokoConditionalAnalysisSequence::ProcessDataPacket(void* DataPacket, TKinokoDataSource* DataSource, TKinokoDataSection* DataSection) throw(TKinokoException)
{
    if (_DataChecker->ProcessDataPacket(DataPacket) <= 0) {
#if 1   //...
	return 0;
#endif
    }

    if (! _DataChecker->IsConditionSatisfied()) {
	return 0;
    }

    return TKinokoAnalysisSequence::ProcessDataPacket(
	DataPacket, DataSource, DataSection
    );
}

void TKinokoConditionalAnalysisSequence::Dump(ostream& os, const string& Indent)
{
    os << Indent << ".sequence conditional (";
    _DataChecker->Dump(os);
    os << ")" << endl;

    TKinokoDataAnalyzerList::Dump(os, Indent + "->\t");
    os << Indent << ".end" << endl;
}
