/* KinokoAnalysisSequenceBuilder.cc */
/* Created by Enomoto Sanshiro on 20 September 2001. */
/* Last updated by Enomoto Sanshiro on 16 June 2002. */


#include <vector>
#include <deque>
#include "KinokoPlatform.hh"
#include "KinokoDataDistributor.hh"
#include "KinokoDataAnalyzer.hh"
#include "KinokoSectionDataAnalyzer.hh"
#include "KinokoAnalysisSequence.hh"
#include "KinokoAnalysisSequenceBuilder.hh"

using namespace std;


TKinokoAnalysisSequenceBuilder::TKinokoAnalysisSequenceBuilder(TKinokoEventEmitter* EventEmitter, TKinokoRegistry* Registry)
{
    _EventEmitter = EventEmitter;
    _Registry = _Registry;

    _DataDistributor = new TKinokoDataDistributor();

    _CurrentSequence = 0;
    _CurrentSingleEventSequence = 0;
}

TKinokoAnalysisSequenceBuilder::~TKinokoAnalysisSequenceBuilder()
{
    _DataDistributor->DestroyAnalyzer();
}

TKinokoDataDistributor* TKinokoAnalysisSequenceBuilder::DataDistributor(void)
{
    return _DataDistributor;
}

vector<TKinokoAnalysisSequence*>& TKinokoAnalysisSequenceBuilder::RootSequenceList(void)
{
    return _RootSequenceList;
}

TKinokoEventEmitter* TKinokoAnalysisSequenceBuilder::EventEmitter(void)
{
    return _EventEmitter;
}

TKinokoRegistry* TKinokoAnalysisSequenceBuilder::Registry(void)
{
    return _Registry;
}

void TKinokoAnalysisSequenceBuilder::OpenRootSequence(const string& DataSourceName)
{
    if (! _AnalysisSequenceStack.empty()) {
	throw TKinokoException(
	    "TKinokoAnalysisSequenceBuilder::OpenRootSequence()", 
	    "badly configured sequence stack (internal)"
	);
    }

    _CurrentSequence = LookupRootSequence(DataSourceName);
    _CurrentSingleEventSequence = 0;
}

void TKinokoAnalysisSequenceBuilder::CloseRootSequence(void)
{
    if (! _AnalysisSequenceStack.empty()) {
	throw TKinokoException(
	    "TKinokoAnalysisSequenceBuilder::CloseRootSequence()", 
	    "badly configured sequence stack (internal)"
	);
    }

    _CurrentSequence = 0;
    _CurrentSingleEventSequence = 0;
}

void TKinokoAnalysisSequenceBuilder::OpenSubSequence(TKinokoAnalysisSequence* SubSequence) throw(TKinokoException)
{
    if (_CurrentSequence == 0) {
	throw TKinokoException(
	    "TKinokoAnalysisSequenceBuilder::OpenSubSequence()", 
	    "sub-sequence outside sequence"
	);
    }

    _CurrentSequence->AddSequence(SubSequence);

    _AnalysisSequenceStack.push_front(_CurrentSequence);
    _SingleEventAnalysisSequenceStack.push_front(_CurrentSingleEventSequence);

    _CurrentSequence = SubSequence;
    _CurrentSingleEventSequence = 0;
}

void TKinokoAnalysisSequenceBuilder::CloseSubSequence(void) throw(TKinokoException)
{
    if (_AnalysisSequenceStack.empty()) {
	throw TKinokoException(
	    "TKinokoAnalysisSequenceBuilder::CloseSubSequence()", 
	    "sequence stack empty (internal)"
	);
    }

    _CurrentSequence = _AnalysisSequenceStack.front();
    _CurrentSingleEventSequence = _SingleEventAnalysisSequenceStack.front();

    _AnalysisSequenceStack.pop_front();
    _SingleEventAnalysisSequenceStack.pop_front();
}

void TKinokoAnalysisSequenceBuilder::AddAction(TKinokoDataAnalyzer* DataAnalyzer) throw(TKinokoException)
{
    if (_CurrentSequence == 0) {
	throw TKinokoException(
	    "TKinokoAnalysisSequenceBuilder::AddAction()", 
	    "analysis action outside sequence"
	);
    }

    _CurrentSequence->AddAnalyzer(DataAnalyzer);
}

void TKinokoAnalysisSequenceBuilder::AddSingleEventAction(TKinokoDataAnalyzer* DataAnalyzer) throw(TKinokoException)
{
    if (_CurrentSequence == 0) {
	throw TKinokoException(
	    "TKinokoAnalysisSequenceBuilder::AddSingleEventAction()", 
	    "analysis action outside sequence"
	);
    }

    if (_CurrentSingleEventSequence == 0) {
	_CurrentSingleEventSequence = new TKinokoSingleEventAnalysisSequence();
	_CurrentSequence->AddSequence(_CurrentSingleEventSequence);
    }

    _CurrentSingleEventSequence->AddAnalyzer(DataAnalyzer);
}

TKinokoAnalysisSequence* TKinokoAnalysisSequenceBuilder::LookupRootSequence(const string& DataSourceName)
{
    TKinokoAnalysisSequence* AnalysisSequence;

    if (_RootSequenceTable.count(DataSourceName) == 0) {
	AnalysisSequence = new TKinokoAnalysisSequence();
	_RootSequenceTable[DataSourceName] = AnalysisSequence;
	_RootSequenceList.push_back(AnalysisSequence);
	_DataDistributor->RegisterAnalyzer(DataSourceName, AnalysisSequence);
    }
    else {
	AnalysisSequence = _RootSequenceTable[DataSourceName];
    }

    return AnalysisSequence;
}
