/* KinokoDataDistributor.cc */
/* Created by Enomoto Sanshiro on 23 October 1999. */
/* Last updated by Enomoto Sanshiro on 1 September 2001. */


#include <string>
#include <map>
#include "KinokoStream.hh"
#include "KinokoDataDescriptor.hh"
#include "KinokoDataAnalyzer.hh"
#include "KinokoDataDistributor.hh"

using namespace std;


TKinokoDataDistributor::TKinokoDataDistributor(void)
{
    _IsAnonymousAnalyzerAvailable = false;
    _DataDescriptor = 0;

    _AnalyzerListSize = 4096;
    _AnalyzerList = new TKinokoDataAnalyzer* [_AnalyzerListSize];
    for (int i = 0; i < _AnalyzerListSize; i++) {
	_AnalyzerList[i] = 0;
    }
}

TKinokoDataDistributor::~TKinokoDataDistributor()
{
}

void TKinokoDataDistributor::RegisterAnalyzer(const string& DataSourceName, TKinokoDataAnalyzer* Analyzer)
{
    for (unsigned i = 0; i < _AnalyzerNameList.size(); i++) {
	if (_AnalyzerNameList[i].first == DataSourceName) {
	    throw TKinokoException(
		"TKinokoDataDistributor::RegisterAnalyzer()",
		"doubly registered data analyzer: " + DataSourceName +
		" (use TKinokoAnalyzerList if necessary)"
	    );
	}
    }

    _AnalyzerNameList.push_back(make_pair(DataSourceName, Analyzer));
    
    if (DataSourceName.empty()) {
	_IsAnonymousAnalyzerAvailable = true;
    }
}

void TKinokoDataDistributor::RegisterCommonAnalyzer(TKinokoDataAnalyzer* Analyzer)
{
    if (! Analyzer->IsSharable()) {
	throw TKinokoException(
	    "TKinokoDataDistributor::RegisterCommonAnalyzer()",
	    "registering unsharable analyzer as 'common'"
	);
    }

    if (_AnalyzerList[0] != 0) {
	throw TKinokoException(
	    "TKinokoDataDistributor::RegisterCommonAnalyzer()",
	    "doubly registered data analyzer: [common]"
	    " (use TKinokoCommonAnalyzerList if necessary)"
	);
    }

    _AnalyzerList[0] = Analyzer;
}

void TKinokoDataDistributor::DestroyAnalyzer(void)
{
    for (unsigned i = 0; i < _AnalyzerNameList.size(); i++) {
	_AnalyzerNameList[i].second->Destroy();
	delete _AnalyzerNameList[i].second;
    }
    
    if (_AnalyzerList[0] != 0) {
	_AnalyzerList[0]->Destroy();
	delete _AnalyzerList[0];
    }
    
    for (int i = 0; i < _AnalyzerListSize; i++) {
	_AnalyzerList[i] = 0;
    }

    _OtherAnalyzerTable.erase(
	_OtherAnalyzerTable.begin(), _OtherAnalyzerTable.end()
    );
    _AnalyzerNameList.erase(
	_AnalyzerNameList.begin(), _AnalyzerNameList.end()
    );
}

void TKinokoDataDistributor::ProcessDataDescriptor(TKinokoDataDescriptor* DataDescriptor) throw(TKinokoException)
{
    _DataDescriptor = DataDescriptor;

    for (unsigned i = 0; i < _AnalyzerNameList.size(); i++) {
	string DataSourceName = _AnalyzerNameList[i].first;
	TKinokoDataAnalyzer* Analyzer = _AnalyzerNameList[i].second;

	TKinokoDataSource* DataSource = 0;
	if (DataSourceName.empty() && _IsAnonymousAnalyzerAvailable) {
	    DataSource = DataDescriptor->DataSourceList()[0];
	    _IsAnonymousAnalyzerAvailable = false;
	}
	else {
	    DataSource = DataDescriptor->DataSource(DataSourceName);
	}

	if (DataSource != 0) {
	    int DataSourceId = DataSource->DataSourceId();
	    if (DataSourceId < _AnalyzerListSize) {
		if (_AnalyzerList[DataSourceId] == 0) {
		    _AnalyzerList[DataSourceId] = Analyzer;
		    Analyzer->ReadDataSource(DataSource);
		}
	    }
	    else {
		if (_OtherAnalyzerTable[DataSourceId] == 0) {
		    _OtherAnalyzerTable[DataSourceId] = Analyzer;
		    Analyzer->ReadDataSource(DataSource);
		}
	    }
	}
    }

    // common analyzer //
    if (_AnalyzerList[0] != 0) {
	const vector<TKinokoDataSource*>& DataSourceList = DataDescriptor->DataSourceList();
	for (unsigned i = 0; i < DataSourceList.size(); i++) {
	    _AnalyzerList[0]->ReadDataSource(DataSourceList[i]);
	}
    }
}

int TKinokoDataDistributor::ProcessDataPacket(void* Packet) throw(TKinokoException)
{
    int Result = 0;

    int DataSourceId = TKinokoDataStreamScanner::DataSourceIdOf(Packet);

    TKinokoDataAnalyzer* Analyzer;
    if (DataSourceId < _AnalyzerListSize) {
	Analyzer = _AnalyzerList[DataSourceId];
    }
    else {
	Analyzer = _OtherAnalyzerTable[DataSourceId];
    }

    if (Analyzer != 0) {
	Result = Analyzer->ProcessDataPacket(Packet);
    }

    // common analyzer //
    if (_AnalyzerList[0] != 0) {
	TKinokoDataSource* DataSource = _DataDescriptor->DataSource(DataSourceId);
	Result += _AnalyzerList[0]->ProcessDataPacket(Packet, DataSource);
    }

    return Result;
}

int TKinokoDataDistributor::ProcessTrailerPacket(void* Packet) throw(TKinokoException)
{
    int Result = 0;

    int DataSourceId = TKinokoDataStreamScanner::DataSourceIdOf(Packet);

    TKinokoDataAnalyzer* Analyzer;
    if (DataSourceId < _AnalyzerListSize) {
	Analyzer = _AnalyzerList[DataSourceId];
    }
    else {
	Analyzer = _OtherAnalyzerTable[DataSourceId];
    }

    if (Analyzer != 0) {
	Result = Analyzer->ProcessTrailerPacket(Packet);
    }

    // common analyzer //
    if (_AnalyzerList[0] != 0) {
	TKinokoDataSource* DataSource = _DataDescriptor->DataSource(DataSourceId);
	Result += _AnalyzerList[0]->ProcessTrailerPacket(Packet, DataSource);
    }

    return Result;
}

int TKinokoDataDistributor::ProcessCommandPacket(void* Packet) throw(TKinokoException)
{
    int Result = 0;

    int DataSourceId = TKinokoDataStreamScanner::DataSourceIdOf(Packet);

    TKinokoDataAnalyzer* Analyzer;
    if (DataSourceId < _AnalyzerListSize) {
	Analyzer = _AnalyzerList[DataSourceId];
    }
    else {
	Analyzer = _OtherAnalyzerTable[DataSourceId];
    }

    if (Analyzer != 0) {
	Result = Analyzer->ProcessCommandPacket(Packet);
    }

    // common analyzer //
    if (_AnalyzerList[0] != 0) {
	TKinokoDataSource* DataSource = _DataDescriptor->DataSource(DataSourceId);
	Result += _AnalyzerList[0]->ProcessCommandPacket(Packet, DataSource);
    }

    return Result;
}

TKinokoDataDescriptor* TKinokoDataDistributor::DataDescriptor(void)
{
    return _DataDescriptor;
}

void TKinokoDataDistributor::Dump(ostream& os)
{
    for (unsigned i = 0; i < _AnalyzerNameList.size(); i++) {
	string DataSourceName = _AnalyzerNameList[i].first;
	TKinokoDataAnalyzer* Analyzer = _AnalyzerNameList[i].second;

	os << "analysis (" << DataSourceName << "):" << endl;
	Analyzer->Dump(os, "\t");
	os << endl;
    }

    if (_AnalyzerList[0] != 0) {
	os << "analysis (COMMON):" << endl;
	_AnalyzerList[0]->Dump(os, "\t");
	os << endl;
    }
}
