/* KinokoArena.cc */
/* Created by Enomoto Sanshiro on 19 April 1998. */
/* Updated by Enomoto Sanshiro on 15 January 2002. */
/* Last Updated by Enomoto Sanshiro on 16 January 2010. */


#include <iostream>
#include <sstream>
#include <sys/types.h>
#include "KinokoArena.hh"


#ifndef __KinokoArena_cc__
#define __KinokoArena_cc__


INLINE TKinokoArenaEntry::TKinokoArenaEntry(int EntryIndex)
{
    _EntryIndex = EntryIndex;
}

INLINE TKinokoArenaEntry::~TKinokoArenaEntry()
{
}

INLINE void TKinokoArenaEntry::Attach(off_t Offset, size_t Capacity, int NumberOfAttached)
{
    _Offset = Offset;
    _Capacity = Capacity;
    _ReferenceCount = NumberOfAttached;
    _PrevEntry = 0;
    _NextEntry = 0;
}

INLINE void TKinokoArenaEntry::Detach(void) 
{
    if (--_ReferenceCount <= 0) {
	if (_PrevEntry != 0) {
	    _PrevEntry->_NextEntry = _NextEntry;
	}
	if (_NextEntry != 0) {
	    _NextEntry->_PrevEntry = _PrevEntry;
	}
    }
}

INLINE void TKinokoArenaEntry::AddNext(TKinokoArenaEntry* Entry)
{
    _NextEntry = Entry;
    Entry->_PrevEntry = this;
}

INLINE int TKinokoArenaEntry::EntryIndex(void) const 
{
    return _EntryIndex;
}

INLINE off_t TKinokoArenaEntry::Offset(void) const 
{
    return _Offset;
}

INLINE int TKinokoArenaEntry::ReferenceCount(void) const 
{
    return _ReferenceCount;
}

INLINE size_t TKinokoArenaEntry::Capacity(void) const 
{
    return _Capacity;
}

INLINE TKinokoArenaEntry* TKinokoArenaEntry::Prev(void) const 
{
    return _PrevEntry;
}

INLINE TKinokoArenaEntry* TKinokoArenaEntry::Next(void) const 
{
    return _NextEntry;
}    



INLINE void TKinokoArenaEntryPool::Initialize(unsigned InitialCapacity)
{
    _EntryTableSize = InitialCapacity;
    _NumberOfOccupied = 0;
    _NextEntryIndex = 0;

    for (unsigned Index = 0; Index < _EntryTableSize; Index++) {
	_EntryTable.push_back(TKinokoArenaEntry(Index));
	_OccupancyTable.push_back(false);
    }
}

INLINE void TKinokoArenaEntryPool::Finalize(void)
{
    _OccupancyTable.erase(_OccupancyTable.begin(), _OccupancyTable.end());
    _EntryTable.erase(_EntryTable.begin(), _EntryTable.end());
}

INLINE TKinokoArenaEntry* TKinokoArenaEntryPool::AllocateEntry(void)
{
    int EntryIndex;

    if (_NumberOfOccupied >= _EntryTableSize) {
	EntryIndex = _EntryTableSize;
	_EntryTable.push_back(TKinokoArenaEntry(EntryIndex));
	_OccupancyTable.push_back(true);
	_EntryTableSize++;
    }
    else {
	for (
	    EntryIndex = _NextEntryIndex;
	    _OccupancyTable[EntryIndex];
	    EntryIndex++, EntryIndex %= _EntryTableSize
	){
	    ;
	}
    }
    _NextEntryIndex = (EntryIndex + 1) % _EntryTableSize;

    _NumberOfOccupied++;
    _OccupancyTable[EntryIndex] = true;

    return &_EntryTable[EntryIndex];
}

INLINE void TKinokoArenaEntryPool::ReleaseEntry(TKinokoArenaEntry* Entry)
{
    _OccupancyTable[Entry->EntryIndex()] = false;
    _NumberOfOccupied--;
}



INLINE void TKinokoArena::Initialize(size_t ArenaSize, unsigned InitialEntryTableSize)
{
    _ArenaSize = ArenaSize;
    
    _HeadEntry = 0;
    _TailEntry = 0;

    _NumberOfEntries = 0;
    _UsedSize = 0;

    _EntryPool.Initialize(InitialEntryTableSize);
}

INLINE void TKinokoArena::Finalize(void)
{
    _EntryPool.Finalize();
}

INLINE bool TKinokoArena::CreateEntry(size_t Capacity, off_t& OffsetAddress, int NumberOfAttached)
{
    if (NumberOfAttached == 0) {
	return false;
    }

    if (_HeadEntry == 0) {	
	OffsetAddress = 0;
    }
    else {
	OffsetAddress = _TailEntry->Offset() + _TailEntry->Capacity();
	if (AvailableSize(OffsetAddress) < Capacity) {
	    OffsetAddress = 0;
	}
    }

    if (AvailableSize(OffsetAddress) < Capacity) {
#if 0
	throw TKinokoException(
	    "TKinokoArena::CreateEntry()", "buffer is exhausted"
	);
#else
	return false;
#endif
    }

    TKinokoArenaEntry* Entry = _EntryPool.AllocateEntry();
    if (Entry == 0) {
#if 0
	throw TKinokoException(
	    "TKinokoArena::CreateEntry()", "unable to create entry"
	);
#else
	return false;
#endif
    }

    Entry->Attach(OffsetAddress, Capacity, NumberOfAttached);
    if (_HeadEntry == 0) {
	_HeadEntry = Entry;
    }
    else {
	_TailEntry->AddNext(Entry);
    }
    _TailEntry = Entry;

    _NumberOfEntries++;
    _UsedSize += Capacity;

    return true;
}

INLINE bool TKinokoArena::DetachEntry(off_t OffsetAddress)
{
    TKinokoArenaEntry* Entry;
    for (Entry = _HeadEntry; Entry != 0; Entry = Entry->Next()) {
	if (Entry->Offset() == OffsetAddress) {
	    break;
	}
    } 
    
    if (Entry != 0) {
	Entry->Detach();
	if (Entry->ReferenceCount() == 0) {
	    if (_HeadEntry == Entry) {
		_HeadEntry = Entry->Next();
	    }
	    if (_TailEntry == Entry) {
		_TailEntry = Entry->Prev();
	    }

	    _NumberOfEntries--;
	    _UsedSize -= Entry->Capacity();

	    _EntryPool.ReleaseEntry(Entry);
	}
    }
    else {
	std::ostringstream os;
	os << "invalid offset address: " << OffsetAddress;
#if 0
	throw TKinokoException("TKinokoArena::DetachEntry()", os.str());
#else
	std::cerr << "ERROR: KinokoArena::DetachEntry(): ";
	std::cerr << os.str() << std::endl;
	return false;
#endif
    }

    return true;
}

INLINE size_t TKinokoArena::Capacity(void) const
{
    return _ArenaSize;
}

INLINE size_t TKinokoArena::AvailableSize(void) const
{
    size_t Size;
    off_t TailOffset;

    if (_HeadEntry == 0) {	
	Size = _ArenaSize;
    }
    else {
	TailOffset = _TailEntry->Offset() + _TailEntry->Capacity();
	Size = AvailableSize(0) + AvailableSize(TailOffset);
    }

    return Size;
}

INLINE size_t TKinokoArena::AvailableSize(off_t StartOffset) const
{
    if (_HeadEntry == 0) {
	return _ArenaSize;
    }
    
    size_t Size;
    off_t HeadOffset = _HeadEntry->Offset();
    off_t TailOffset = _TailEntry->Offset() + _TailEntry->Capacity();
    
    // Notation: ...: Used, ___: Free, ---: AvailableContinuousSize //

    if (HeadOffset < TailOffset) {
	if (StartOffset < HeadOffset) {
	    // _____S---H...T_____
	    Size = HeadOffset - StartOffset;
	}
	else if (StartOffset < TailOffset) {
	    // _____H...S...T_____
	    Size = 0;
	}
	else {
	    // _____H...T___S-----
	    Size = _ArenaSize - StartOffset;
	}
    }
    else {
	if (StartOffset < TailOffset) {
	    // .....S...T___H.....
	    Size = 0;
	}
	else if (StartOffset < HeadOffset) {
	    // .....T___S---H.....
	    Size = HeadOffset - StartOffset;
	}
	else {
	    // .....T___H...S.....
	    Size = 0;
	}
    }
    
    return Size;
}

INLINE bool TKinokoArena::IsAllocatable(size_t Size) const
{
    if (_HeadEntry == 0) {
	return (Size <= AvailableSize(0));
    }
    else {
	off_t TailOffset = _TailEntry->Offset() + _TailEntry->Capacity();
	return (Size <= AvailableSize(TailOffset)) || (Size <= AvailableSize(0));
    }
}

INLINE int TKinokoArena::NumberOfEntries(void) const
{
    return _NumberOfEntries;
}

INLINE size_t TKinokoArena::UsedSize(void) const
{
    return _UsedSize;
}

INLINE off_t TKinokoArena::DataHeadOffset(void) const
{
    if (_HeadEntry == 0) {
	return 0;
    }
    else {
	return _HeadEntry->Offset();
    }
}

INLINE off_t TKinokoArena::NextStartOffset(void) const
{
    if (_HeadEntry == 0) {
	return 0;
    }
    else {
	return _TailEntry->Offset() + _TailEntry->Capacity();
    }
}



INLINE void TKinokoArenaBinder::SetBaseAddress(void* BaseAddress)
{
    _BaseAddress = BaseAddress;
}

INLINE bool TKinokoArenaBinder::CreateEntry(size_t Capacity, void*& Address, int NumberOfAttached)
{
    off_t Offset;
    if (_Arena.CreateEntry(Capacity, Offset, NumberOfAttached)) {
	Address = (char*) _BaseAddress + Offset;
	return true;
    }
    else {
	return false;
    }
}

INLINE bool TKinokoArenaBinder::DetachEntry(const void* Address)
{
    return _Arena.DetachEntry((const char*) Address - (char*) _BaseAddress);
}


#endif
