/* KinokoBufferServer.hh */
/* Created by Enomoto Sanshiro on 19 April 1998. */
/* Updated by Enomoto Sanshiro on 21 August 2002. */
/* Last updated by Enomoto Sanshiro on 16 January 2010. */


#ifndef __KinokoBufferServer_hh__
#define __KinokoBufferServer_hh__


#include <vector>
#include <deque>
#include "MushDefs.hh"
#include "MushTimer.hh"
#include "MushIpc.hh"
#include "KinokoDefs.hh"
#include "KinokoArena.hh"
#include "KinokoBufferLogger.hh"



struct TKinokoBufferPacket {
  public:
    long ReceiverAddress;
    long SenderAddress;
    off_t OffsetAddress;
    size_t Size;
    int PacketType;
  public:
    enum TKinokoBufferPacketType {
	ptWriteRequest,
	ptEntryReady,
	ptDataStrobe,
	ptReadRequest,
	ptDataAcknowledge,
	ptReaderAttachRequest,
	ptReaderDetachRequest,
	ptReaderDetachAcknowledge,
	ptWriterAttachRequest,
	ptWriterDetachRequest,
	_NumberOfPacketTypes
    };
};


class TKinokoBufferRequest {
  public:
    TKinokoBufferRequest(const TKinokoBufferPacket &KinokoBufferPacket) {
	WriterMessageAddress = KinokoBufferPacket.SenderAddress;
	OffsetAddress = KinokoBufferPacket.OffsetAddress;
	Size = KinokoBufferPacket.Size;
    }
  public:
    long WriterMessageAddress;
    size_t OffsetAddress;
    size_t Size;
};


class TKinokoBufferServer {
  public:
    TKinokoBufferServer(int SharedMemoryProjectId, int MessageQueueProjectId, const std::string& ProjectPath, unsigned long BufferSize, int InitialEntryTableSize, TKinokoBufferLogger* Logger) throw(TKinokoException);
    ~TKinokoBufferServer();
    int DoTransaction(TMushIntervalTimer *TimeKeeper);
    static long MessageQueueAddress(void);
    int SharedMemoryId(void);
    int MessageQueueId(void);
    void ReportStatus(void);
    void GetStatistics(float& BufferUsage, float& NumberOfEntries, float& DataRate, float& PacketRate);
  private:
    void DispatchMessage(const TKinokoBufferPacket& Packet);
    void ProcessPendingRequests(void);
    bool ProcessWriteRequest(const TKinokoBufferRequest& WriteRequest);
    bool ProcessReadRequest(const TKinokoBufferRequest& ReadRequest);
    void SendEntryReady(const TKinokoBufferRequest& WriteRequest, off_t OffsetAddress);
    void SendReadRequest(const TKinokoBufferRequest& ReadRequest);
    void DetachEntry(off_t OffsetAddress);
    void AttachReader(long ReaderAddress);
    void DetachReader(long ReaderAddress);
    void AttachWriter(long WriterAddress);
    void DetachWriter(long WriterAddress);
  private:
    TKinokoArena _Arena;
    size_t _BufferSize;
    TMushServerSharedMemory* _SharedMemory;
    TMushServerMessageQueue* _MessageQueue;
    TKinokoBufferLogger* _Logger;
    TKinokoBufferLogger* _PrescaledLogger;
    int _NumberOfReaders, _NumberOfWriters;
    std::vector<long> _ReaderAddressList;
  private:
    std::deque<TKinokoBufferRequest> _WriteRequestQueue;
    std::deque<TKinokoBufferRequest> _ReadRequestQueue;
    int _NumberOfUnreadPackets, _MaxTolerableNumberOfUnreadPackets;
    unsigned long _WrittenDataSize, _NumberOfWrittenPackets;
    long _LastReportTime;
    float _DataRate, _PacketRate;
  private:
    TKinokoBufferPacket _Packet;
};


#endif
