/* KinokoBuilderConfiguration.cc */
/* Created by Enomoto Sanshiro on 12 September 2009. */
/* Updated by Enomoto Sanshiro on 19 May 2010. */
/* Last updated by Enomoto Sanshiro on 7 July 2010. */

#include <string>
#include <sstream>
#include "KinokoHorizontalBuilder.hh"
#include "KinokoVerticalBuilder.hh"
#include "KinokoBuilderConfiguration.hh"

using namespace std;


TKinokoKiscBuilderConfiguration::TKinokoKiscBuilderConfiguration(TKiscConfigFile& ConfigFile)
: _ConfigFile(ConfigFile)
{
}

TKinokoKiscBuilderConfiguration::~TKinokoKiscBuilderConfiguration()
{
}

void TKinokoKiscBuilderConfiguration::ConfigureBuilder(TKinokoBuilder* Builder, const std::string& BuilderName) throw(TKinokoException)
{
    try {
	_ConfigFile.SetTargetSection(BuilderName);
    }
    catch (TScriptException &e) {
	throw TKinokoException(
	    BuilderName, "Config file syntax error: " + e.Message()
	);
    }
    ConfigureCommon(Builder, BuilderName);

    string Type;
    try {
	Type = _ConfigFile["Type"].AsString();
    }
    catch (TScriptException &e) {
	throw TKinokoException(BuilderName, "'Type' not defined");
    }
    if (Type == "Horizontal") {
	TKinokoHorizontalBuilder* HorizontalBuilder = (
	    dynamic_cast<TKinokoHorizontalBuilder*>(Builder)
	);
	if (HorizontalBuilder == 0) {
	    throw TKinokoException(
		BuilderName, "Builder type mismatch: " + Type
	    );
	}
	ConfigureHorizontal(HorizontalBuilder, BuilderName);
    }
    else if (Type == "Vertical") {
	TKinokoVerticalBuilder* VerticalBuilder = (
	    dynamic_cast<TKinokoVerticalBuilder*>(Builder)
	);
	if (VerticalBuilder == 0) {
	    throw TKinokoException(
		BuilderName, "Builder type mismatch: " + Type
	    );
	}
	ConfigureVertical(VerticalBuilder, BuilderName);
    }
    else {
	throw TKinokoException(
	    BuilderName, "Unknown builder type: " + Type
	);
    }
}

void TKinokoKiscBuilderConfiguration::ConfigureCommon(TKinokoBuilder* Builder, const std::string& BuilderName) throw(TKinokoException)
{
    Builder->SetOutputDataSourceName(
	_ConfigFile.GetValueOfWithDefault("Name", BuilderName).AsString()
    );

    try {
	long BufferSize = (
	    _ConfigFile.GetValueOfWithDefault("BufferSize", "0").AsLong()
	);
	Builder->SetBufferSize(BufferSize);
    }
    catch (TScriptException &e) {
	throw TKinokoException(BuilderName, "Bad BufferSize literal");
    }

    try {
	long StartDataSize = (
	    _ConfigFile.GetValueOfWithDefault("StartDataSize", "0").AsLong()
	);
	if (StartDataSize != 0) {
	    Builder->SetStartDataSize(StartDataSize);
	}
    }
    catch (TScriptException &e) {
	;
    }

    try {
	long StartPacketCycles = (
	    _ConfigFile.GetValueOfWithDefault("StartPacketCycles", "0").AsLong()
	);
	if (StartPacketCycles != 0) {
	    Builder->SetStartPacketCycles(StartPacketCycles);
	}
    }
    catch (TScriptException &e) {
	;
    }

    TParaValue StartChannelListValue;
    try {
	StartChannelListValue = _ConfigFile["StartChannelList"];
    }
    catch (TScriptException &e) {
	;
    }
    vector<long> ChannelList;
    try {
	if (StartChannelListValue.IsList()) {
	    TParaListValue StartChannelList = StartChannelListValue.AsList();
	    for (unsigned i = 0; i < StartChannelList.ListSize(); i++) {
		ChannelList.push_back(StartChannelList.ValueOf(i).AsLong());
	    }
	}
	else if (! StartChannelListValue.IsVoid()) {
	    ChannelList.push_back(StartChannelListValue.AsLong());
	}
    }
    catch (TScriptException &e) {
	throw TKinokoException(
	    "invalid channel list: " + StartChannelListValue.AsString()
	);
    }
    if (! ChannelList.empty()) {
	Builder->SetStartChannelList(ChannelList);
    }

    try {
	bool IsInputPreserved = (
	    _ConfigFile.GetValueOfWithDefault("IsInputPreserved", "").AsBool()
	);
	Builder->SetInputPreservation(IsInputPreserved);
    }
    catch (TScriptException &e) {
	;
    }
    try {
	int ActivityWindowLength = (
	    _ConfigFile.GetValueOfWithDefault("ActivityWindowLength", "0").AsLong()
	);
	if (ActivityWindowLength != 0) {
	    Builder->SetActivityWindowLength(ActivityWindowLength);
	}
    }
    catch (TScriptException &e) {
	;
    }
}

void TKinokoKiscBuilderConfiguration::ConfigureHorizontal(TKinokoHorizontalBuilder* Builder, const std::string& BuilderName) throw(TKinokoException)
{
    string TargetDataSource;
    try {
	TargetDataSource = _ConfigFile["TargetDataSource"].AsString();
    }
    catch (TScriptException &e) {
	throw TKinokoException(
	    BuilderName, "'TargetDataSource' not defined"
	);
    }
    Builder->SetTargetDataSource(TargetDataSource);

    TParaValue TargetSection;
    try {
	TargetSection = _ConfigFile["TargetDataSection"];
    }
    catch (TScriptException &e) {
	throw TKinokoException(
	    BuilderName, "'TargetDataSection' not defined"
	);
    }
    if (TargetSection.IsList()) {
	TParaListValue SectionList = TargetSection.AsList();
	for (unsigned i = 0; i < SectionList.ListSize(); i++) {
	    TKinokoEventPiece::TStreamKey StreamKey = -1;
	    if (SectionList.HasKeyIndex()) {
		string StreamKeyString = SectionList.KeyOf(i);
		if (! StreamKeyString.empty()) {
		    istringstream is(StreamKeyString);
		    if (! (is >> StreamKey)) {
			throw TKinokoException(
			    BuilderName, "Bad StreamKey: " + StreamKeyString
			);
		    }
		}
	    }
	    Builder->AddTargetDataSection(
		SectionList.ValueOf(i).AsString(), StreamKey
	    );
	}
    }
    else {
	TKinokoEventPiece::TStreamKey StreamKey = -1;
	Builder->AddTargetDataSection(TargetSection.AsString(), StreamKey);
    }
}

void TKinokoKiscBuilderConfiguration::ConfigureVertical(TKinokoVerticalBuilder* Builder, const std::string& BuilderName) throw(TKinokoException)
{
    TParaValue TargetSource;
    try {
	TargetSource = _ConfigFile["TargetDataSource"];
    }
    catch (TScriptException &e) {
	throw TKinokoException("'TargetDataSource' not defined");
    }
    if (TargetSource.IsList()) {
	TParaListValue SourceList = TargetSource.AsList();
	for (unsigned i = 0; i < SourceList.ListSize(); i++) {
	    TKinokoEventPiece::TStreamKey StreamKey = -1;
	    if (SourceList.HasKeyIndex()) {
		string StreamKeyString = SourceList.KeyOf(i);
		if (! StreamKeyString.empty()) {
		    istringstream is(StreamKeyString);
		    if (! (is >> StreamKey)) {
			throw TKinokoException(
			    BuilderName, "Bad StreamKey: " + StreamKeyString
			);
		    }
		}
	    }
	    Builder->AddTargetDataSource(
		SourceList.ValueOf(i).AsString(), StreamKey
	    );
	}
    }
    else {
	TKinokoEventPiece::TStreamKey StreamKey = -1;
	Builder->AddTargetDataSource(TargetSource.AsString(), StreamKey);
    }
}
