/* KinokoBuilderInlet.cc */
/* Created by Enomoto Sanshiro on 26 April 2010. */
/* Last updated by Enomoto Sanshiro on 26 April 2010. */


#include <string>
#include <iostream>
#include <vector>
#include <map>
#include "KinokoDataProcessor.hh"
#include "KinokoBuilderInlet.hh"

using namespace std;

//#define DEBUG(x) x
#define DEBUG(x)


TKinokoBuilderInlet::TKinokoBuilderInlet(TKinokoEventFragmentProcessor* FragmentProcessor)
{
    _Processor = FragmentProcessor;

    _NullLogger = new TKinokoBuilderNullLogger();
    _Logger = _NullLogger;

    _DataDescriptor = new TKinokoDataDescriptor(); 
    _StreamScanner = new TKinokoDataStreamScanner();

    _RunBeginCount = 0;
    _IsConstructed = false;

    DEBUG(cout << "debug flag enabled for KinokoBuilderInlet.cc" << endl);
}

TKinokoBuilderInlet::~TKinokoBuilderInlet()
{
    delete _StreamScanner;
    delete _DataDescriptor;
    delete _NullLogger;
}

void TKinokoBuilderInlet::SetLogger(TKinokoBuilderLogger* Logger)
{
    if (Logger) {
	_Logger = Logger;
    }
}

void TKinokoBuilderInlet::SetTargetDataSource(const std::string& DataSourceName)
{
    _DataSourceName = DataSourceName;
}

void TKinokoBuilderInlet::AddTargetDataSection(const std::string& DataSectionName, TStreamKey StreamKey)
{
    _SectionNameList.push_back(make_pair(DataSectionName, StreamKey));
}

void TKinokoBuilderInlet::Construct(void) throw(TKinokoException) 
{
    if (_IsConstructed) {
	return;
    }
    _IsConstructed = true;

    if (_DataSourceName.empty()) {
	throw TKinokoException(
	    "TKinokoBuilderInlet::OnConstruct()",
	    "No target DataSourceName specified"
	);
    }

    TKinokoDataSource* DataSource = _DataDescriptor->DataSource(
	_DataSourceName
    );
    if (DataSource == 0) {
	_Logger->WriteWarning(
	    "No target Datasource found: " + _DataSourceName
	);
	return;
    }

    _DataSourceId = DataSource->DataSourceId();
    TKinokoDataSection* DataSection;
    for (unsigned i = 0; i < _SectionNameList.size(); i++) {
	string SectionName = _SectionNameList[i].first;
	TStreamKey StreamKey = _SectionNameList[i].second;
	DataSection = DataSource->DataSection(SectionName);
	if (DataSection == 0) {
	    _Logger->WriteWarning(
		"No target DataSection found: " + 
		_DataSourceName + "." + SectionName
	    );
	    continue;
	}
	if (StreamKey < 0) {
	    StreamKey = DataSection->SectionId();
	}
	_StreamKeyTable[DataSection->SectionId()] = StreamKey;
    }
}

void TKinokoBuilderInlet::ProcessRunBegin(void) throw(TKinokoException)
{
    if (! _IsConstructed) {
	Construct();
    }

    _Processor->ProcessRunBegin();
}

void TKinokoBuilderInlet::ProcessRunEnd(void) throw(TKinokoException)
{
    _Processor->ProcessRunEnd();
}

bool TKinokoBuilderInlet::ProcessPacket(void* Packet, long PacketSize) throw(TKinokoException)
{
    if (_StreamScanner->IsDataDescriptorPacket(Packet)) {
	return ProcessDescriptorPacket(Packet, PacketSize);
    }
    else if (_StreamScanner->IsCommandPacket(Packet)) {
	// do this first for ProcessRunBegin() //
	ProcessCommandPacket(Packet, PacketSize);
    }

    int DataSourceId = TKinokoDataStreamScanner::DataSourceIdOf(Packet);
    if (DataSourceId != _DataSourceId) {
	return false;
    }
    if (_StreamScanner->IsDataPacket(Packet)) {
	return ProcessDataPacket(Packet, PacketSize);
    }
    if (_StreamScanner->IsTrailerPacket(Packet)) {
	return ProcessTrailerPacket(Packet, PacketSize);
    }

    return false;
}
    
bool TKinokoBuilderInlet::ProcessDescriptorPacket(void* Packet, long PacketSize) throw(TKinokoException)
{
    char* DescriptorText = _StreamScanner->DataDescriptorOf(Packet);
    istringstream DescriptorTextStream(DescriptorText);
    try {
	_DataDescriptor->ReadFrom(DescriptorTextStream);
    }
    catch (TKinokoException &e) {
	throw TKinokoException(
	    "TKinokoBufferInlet::ProcessDescriptorPacket()",
	    "badly formatted data descriptor: " + e.Message()
	);
    }

    return false;
}

bool TKinokoBuilderInlet::ProcessCommandPacket(void* Packet, long PacketSize) throw(TKinokoException)
{
    int CommandValue = _StreamScanner->CommandValueOf(Packet);
    if (CommandValue == TKinokoDataStreamScanner::Command_RunBegin) {
	_RunBeginCount++;
	if (_RunBeginCount == 1) {
	    ProcessRunBegin();
	}
    }
    else if (CommandValue == TKinokoDataStreamScanner::Command_RunEnd) {
	_RunBeginCount--;
	if (_RunBeginCount == 0) {
	    ProcessRunEnd();
	}
    }

    return false;
}

bool TKinokoBuilderInlet::ProcessDataPacket(void* Packet, long PacketSize) throw(TKinokoException)
{
    int SectionId = TKinokoDataSectionScanner::SectionIdOf(Packet);

    map<int, TStreamKey>::iterator Entry = _StreamKeyTable.find(SectionId);
    if (Entry == _StreamKeyTable.end()) {
	// non-target section; forward it //
	return false;
    }
    TStreamKey StreamKey = Entry->second;

    char* DataArea = (char*) TKinokoDataSectionScanner::DataAreaOf(Packet);
    int DataSize = TKinokoDataSectionScanner::DataSizeOf(Packet);
    if (DataSize <= 0) {
	return true;
    }

    TKinokoDataChunk DataChunk(DataArea, DataSize);
    if (_Processor->ProcessFragment(StreamKey, DataChunk) == 0) {
	// not accepted by the builder; forward it //
	return false;
    }

    return true;
}

bool TKinokoBuilderInlet::ProcessTrailerPacket(void* Packet, long PacketSize) throw(TKinokoException)
{
    int TrailerValue = _StreamScanner->TrailerValueOf(Packet);
    if (TrailerValue == TKinokoDataStreamScanner::Trailer_Event) {
	_Processor->ProcessEventEnd();
    }

    return true;
}
