/* KinokoEventChopper.cc */
/* Created by Enomoto Sanshiro on 11 August 2009. */
/* Last updated by Enomoto Sanshiro on 7 March 2010. */


#include <iostream>
#include <sstream>
#include <vector>
#include <map>
#include <set>
#include "KinokoDataChunk.hh"
#include "KinokoEventPiece.hh"
#include "KinokoEventPieceScanner.hh"
#include "KinokoBuilderProcessor.hh"
#include "KinokoEventChopper.hh"

using namespace std;

//#define DEBUG(x) x
#define DEBUG(x)


TKinokoCarryOverTable::TKinokoCarryOverTable(void)
{
}

TKinokoCarryOverTable::~TKinokoCarryOverTable()
{
    for (unsigned i = 0; i < _BufferList.size(); i++) {
	delete[] _BufferList[i];
    }
}

void TKinokoCarryOverTable::Put(TStreamKey StreamKey, const TKinokoDataChunk& DataChunk)
{
    if (DataChunk.IsEmpty()) {
	return;
    }

    DEBUG(cout << "Chopper: Put CarryOver: ");
    DEBUG(cout << "stream=" << StreamKey << ", ");
    DEBUG(cout << "size=" << DataChunk.Size() << endl);

    int Index;
    map<TStreamKey, int>::iterator Entry = _KeyIndexTable.find(StreamKey);
    if (Entry == _KeyIndexTable.end()) {
	Index = _BufferList.size();
	_KeyIndexTable[StreamKey] = Index; 
	_BufferList.push_back(0);
	_CapacityList.push_back(0);
	_SizeList.push_back(0);
    }
    else {
	Index = Entry->second;
    }

    size_t Size = DataChunk.Size();
    char* Buffer = _BufferList[Index];

    if (_CapacityList[Index] < Size) {
	DEBUG(cout << "Chopper: Reallocating Carry-Over Buffer for ");
	DEBUG(cout << "stream=" << StreamKey << "..." << flush);
	_CapacityList[Index] = (int) (1.2 * Size);
	Buffer = new char[_CapacityList[Index]];
	// do not delete old buffer here because the DataChunk may use it //
	DEBUG(cout << "done. size=" << _CapacityList[Index] << endl);
    }

    if (Size > 0) {
	DataChunk.WriteTo(Buffer);
    }
    if (Buffer != _BufferList[Index]) {
	delete[] _BufferList[Index];
	_BufferList[Index] = Buffer;
    }

    _SizeList[Index] = Size;
}

bool TKinokoCarryOverTable::Get(TStreamKey StreamKey, TKinokoDataChunk& DataChunk)
{
    map<TStreamKey, int>::iterator Entry = _KeyIndexTable.find(StreamKey);
    if (Entry == _KeyIndexTable.end()) {
	return false;
    }
    if (_SizeList[Entry->second] == 0) {
	return false;
    }

    DataChunk.PushFront(
	_BufferList[Entry->second], _SizeList[Entry->second]
    );
    _SizeList[Entry->second] = 0;

    DEBUG(cout << "Chopper: Get CarryOver: ");
    DEBUG(cout << "stream=" << StreamKey << ", ");
    DEBUG(cout << "size=" << DataChunk.Size() << endl);

    return true;
}

set<TKinokoCarryOverTable::TStreamKey> TKinokoCarryOverTable::StreamKeySet(void)
{
    set<TStreamKey> Set;
    for (
	map<TStreamKey, int>::iterator Element = _KeyIndexTable.begin();
	Element != _KeyIndexTable.end();
	Element++
    ){
	Set.insert(Element->first);
    }

    return Set;
}



TKinokoEventChopper::TKinokoEventChopper(TKinokoEventPieceProcessor* PieceProcessor, TKinokoEventPieceScanner* PieceScanner)
{
    _PieceProcessor = PieceProcessor;
    _PieceScanner = PieceScanner;

    DEBUG(cout << "debug flag enabled for KinokoEventChopper.cc" << endl);
}

TKinokoEventChopper::~TKinokoEventChopper()
{
}

void TKinokoEventChopper::ProcessRunBegin(void)
{
    _PieceProcessor->ProcessRunBegin();
}

void TKinokoEventChopper::ProcessRunEnd(void)
{
    set<TStreamKey> StreamKeySet = _CarryOverTable.StreamKeySet();
    for (
	set<TStreamKey>::iterator StreamKey = StreamKeySet.begin();
	StreamKey != StreamKeySet.end();
	StreamKey++
    ){
	TKinokoEventPiece EventPiece;
	if (_CarryOverTable.Get(*StreamKey, EventPiece)) {
	    ostringstream os;
	    os << "Fragments left over on run-end: stream=" << *StreamKey;
	    _Logger->WriteWarning(os.str());

	    EventPiece.PieceType = PieceType_Corrupted;
	    EventPiece.ChannelKey = _LocationTable[*StreamKey].first;
	    EventPiece.EventKey = _LocationTable[*StreamKey].second;
	    _PieceProcessor->ProcessPiece(EventPiece);
	}
    }

    _PieceProcessor->ProcessRunEnd();
}

int TKinokoEventChopper::ProcessFragment(TStreamKey StreamKey, const TKinokoDataChunk& DataChunk)
{
    DEBUG(cout << "Chopper::ProcessFragment(): ");
    DEBUG(cout << "stream=" << StreamKey << ", ");
    DEBUG(cout << "size=" << DataChunk.Size() << endl);

    TKinokoDataChunk ThisDataChunk(DataChunk);
    _CarryOverTable.Get(StreamKey, ThisDataChunk);

    while (! ThisDataChunk.IsEmpty()) {
	DEBUG(cout << "Chopper: Fragment: ");
	DEBUG(cout << "stream=" << StreamKey << ", ");
	DEBUG(cout << "size=" << ThisDataChunk.Size() << endl);

	TKinokoEventPiece EventPiece(ThisDataChunk);
	int PieceSize = _PieceScanner->Scan(StreamKey, EventPiece);
	if ((PieceSize <= 0) || (PieceSize > EventPiece.Size())) {
	    // piece imcomplete //
	    break;
	}
	DEBUG(cout << "Chopper: Piece: ");
	DEBUG(cout << "stream=" << StreamKey << ", ");
	DEBUG(cout << "size=" << PieceSize << ", ");
	DEBUG(cout << "type=" << EventPiece.PieceType << ", ");
	DEBUG(cout << "channel=" << EventPiece.ChannelKey << ", ");
	DEBUG(cout << "event=" << EventPiece.EventKey << endl);
	EventPiece.CutBackTo(PieceSize);
	
	if (EventPiece.PieceType == PieceType_Padding) {
	    DEBUG(cout << "Chopper: padding piece discarded: ");
	    DEBUG(cout << "size=" << EventPiece.Size() << endl);
	    ThisDataChunk.CutFrontBy(PieceSize);
	    continue;
	}

	if (EventPiece.ChannelKey < 0) {
	    EventPiece.ChannelKey = _LocationTable[StreamKey].first;
	}
	if (EventPiece.EventKey < 0) {
	    EventPiece.EventKey = _LocationTable[StreamKey].second;
	}
	_LocationTable[StreamKey] = make_pair(
	    EventPiece.ChannelKey, EventPiece.EventKey
	);

	if (EventPiece.PieceType == PieceType_Corrupted) {
	    DEBUG(cout << "Chopper: data corruption: ");
	    DEBUG(cout << "stream=" << StreamKey << ", ");
	    DEBUG(cout << "channel=" << EventPiece.ChannelKey << ", ");
	    DEBUG(cout << "event=" << EventPiece.EventKey << ", ");
	    DEBUG(cout << "size=" << EventPiece.Size() << endl);
	    if (_CorruptionCountTable[StreamKey] == 0) {
		ostringstream os;
		os << "Data corruption: stream=" << StreamKey;
		_Logger->WriteWarning(os.str());
	    }
	    _CorruptionCountTable[StreamKey]++;
	}

	_PieceProcessor->ProcessPiece(EventPiece);
	ThisDataChunk.CutFrontBy(PieceSize);
    }

    if (! ThisDataChunk.IsEmpty()) {
	if (ThisDataChunk.Size() <= _PieceScanner->MaximumPieceSize()) {
	    _CarryOverTable.Put(StreamKey, ThisDataChunk);
	}
	else {
	    if (_CorruptionCountTable[StreamKey] == 0) {
		ostringstream os;
		os << "Data fragment too large: stream=" << StreamKey;
		_Logger->WriteError(os.str());
	    }
	    _CorruptionCountTable[StreamKey]++;
	    
	    TKinokoEventPiece EventPiece(ThisDataChunk);
	    EventPiece.PieceType = PieceType_Corrupted;
	    EventPiece.ChannelKey = _LocationTable[StreamKey].first;
	    EventPiece.EventKey = _LocationTable[StreamKey].second;
	    _PieceProcessor->ProcessPiece(EventPiece);
	}
    }

    return 1;
}
