/* KinokoEventEnvelope.cc */
/* Created by Enomoto Sanshiro on 3 September 2009. */
/* Last updated by Enomoto Sanshiro on 7 March 2010. */


#include <string>
#include <iostream>
#include <cstdlib>
#include "KinokoDataProcessor.hh"
#include "KinokoTaggedDataSection.hh"
#include "KinokoBlockDataSection.hh"
#include "KinokoEventEnvelope.hh"

using namespace std;

//#define DEBUG(x) x
#define DEBUG(x)


static const string SegmentSectionName = "event_segment";
static string SegmentFormatName = "EventSegment";
static const string FormatVersion = "1.0";
static const int AcceptableContentFormatMajorVersion = 1;


TKinokoEventEnvelopeWriter::TKinokoEventEnvelopeWriter(TKinokoDataSender* DataSender)
{
    _DataSender = DataSender;
    DEBUG(cout << "debug flag enabled for KinokoEventEnvelope.cc" << endl);
}

TKinokoEventEnvelopeWriter::~TKinokoEventEnvelopeWriter()
{
}

void TKinokoEventEnvelopeWriter::BuildDataSource(TKinokoDataSource* DataSource)
{
    TKinokoBlockDataSection* SegmentSection = (
	new TKinokoBlockDataSection(DataSource, SegmentSectionName)
    );
    DataSource->AddDataSection(SegmentSection);
    SegmentSection->AddAttribute("ContentFormat", SegmentFormatName);
    SegmentSection->AddAttribute("ContentFormatVersion", FormatVersion);
    _SegmentFormatter = SegmentSection->Formatter();
}

void TKinokoEventEnvelopeWriter::ProcessRunBegin(void)
{
    _DataSender->SendRunBeginPacket();
}

void TKinokoEventEnvelopeWriter::ProcessRunEnd(void)
{
    _DataSender->SendRunEndPacket();
}

void TKinokoEventEnvelopeWriter::ProcessEventBegin(TEventKey EventKey, int PieceType, int NumberOfPieces, size_t TotalSize)
{
}

void TKinokoEventEnvelopeWriter::ProcessEventEnd(void)
{
    _DataSender->SendEventTrailerPacket();
}

int TKinokoEventEnvelopeWriter::ProcessSegment(const TKinokoEventSegment& EventSegment)
{
    DEBUG(cout << "Envelope Writer: Begin Event: ");
    DEBUG(cout << "EventKey=" << EventSegment.EventKey << ", ");
    DEBUG(cout << "PieceType=" << EventSegment.PieceType << ", ");
    DEBUG(cout << "SegmentSize=" << EventSegment.Size() << endl);

    int SegmentSize = EventSegment.Size();
    int PacketSize = _SegmentFormatter->PacketSizeFor(SegmentSize);
    void* Packet = _DataSender->AllocatePacket(PacketSize);
    U32bit* DataArea = (U32bit*) _SegmentFormatter->DataAreaOf(Packet);

    _SegmentFormatter->WriteHeaderTo(Packet, SegmentSize);
    EventSegment.WriteTo(DataArea);

    _DataSender->FlushPacket(Packet, PacketSize);

    return 1;
}



TKinokoEventEnvelopeReader::TKinokoEventEnvelopeReader(TKinokoEventSegmentProcessor* SegmentProcessor)
{
    _Processor = SegmentProcessor;

    _NullLogger = new TKinokoBuilderNullLogger();
    _Logger = _NullLogger;

    _DataDescriptor = new TKinokoDataDescriptor(); 
    _StreamScanner = new TKinokoDataStreamScanner();

    _RunBeginCount = 0;
    _IsConstructed = false;

    DEBUG(cout << "debug flag enabled for KinokoEventEnvelope.cc" << endl);
}

TKinokoEventEnvelopeReader::~TKinokoEventEnvelopeReader()
{
    delete _StreamScanner;
    delete _DataDescriptor;
    delete _NullLogger;
}

void TKinokoEventEnvelopeReader::SetLogger(TKinokoBuilderLogger* Logger)
{
    if (Logger) {
	_Logger = Logger;
    }
}

void TKinokoEventEnvelopeReader::AddTargetDataSource(const std::string& DataSourceName, TStreamKey StreamKey)
{
    _DataSourceNameList.push_back(make_pair(DataSourceName, StreamKey));
}

const std::vector<std::pair<std::string, TKinokoEventEnvelopeReader::TStreamKey> >& TKinokoEventEnvelopeReader::InputDataSourceNameList(void)
{
    if (! _IsConstructed) {
	Construct();
    }

    return _ValidDataSourceNameList;
}

void TKinokoEventEnvelopeReader::Construct(void) throw(TKinokoException)
{
    if (_IsConstructed) {
	return;
    }
    _IsConstructed = true;

    for (unsigned i = 0; i < _DataSourceNameList.size(); i++) {
	string DataSourceName = _DataSourceNameList[i].first;

	TKinokoDataSource* DataSource = _DataDescriptor->DataSource(
	    DataSourceName
	);
	if (DataSource == 0) {
	    _Logger->WriteWarning(
		"No input DataSource found: " + DataSourceName
	    );
	    continue;
	}
	int DataSourceId = DataSource->DataSourceId();

	TKinokoBlockDataSection* Section = (TKinokoBlockDataSection*) (
	    DataSource->DataSection(SegmentSectionName)
	);
	if (Section == 0) {
	    _Logger->WriteError(
		"No '" + SegmentSectionName + "' section found: " + 
		DataSourceName
	    );
	    continue;
	}
	string Format;
	if (
	    (! Section->GetAttributeOf("ContentFormat", Format)) ||
	    (Format != SegmentFormatName)
	){
	    _Logger->WriteWarning(
		"Not an event segment: " + 
		(DataSourceName + "." + SegmentSectionName)
	    );
	    continue;
	}
	string VersionStr;
	float Version;
	if (
	    (! Section->GetAttributeOf("ContentFormatVersion", VersionStr)) ||
	    (! (istringstream(VersionStr) >> Version))
	){
	    _Logger->WriteError(
		"Unable to get event segment format version: " +
		(DataSourceName + "." + SegmentSectionName)
	    );
	    continue;
	}
	if ((int) Version > AcceptableContentFormatMajorVersion) {
	    _Logger->WriteError(
		"Unknown event segment format version "
		"(this KiNOKO is too old?): " + 
		(DataSourceName + "." + SegmentSectionName) +
		"(format ver " + VersionStr + ")"
	    );
	    continue;
	}
	int SectionId = Section->SectionId();

	TStreamKey StreamKey = _DataSourceNameList[i].second;
	if (StreamKey < 0) {
	    StreamKey = DataSourceId;
	}
	int DataSourceIndex = _SegmentSectionIdTable.size();

	_DataSourceIndexTable[DataSourceId] = DataSourceIndex;
	_SegmentSectionIdTable.push_back(SectionId);
	_StreamKeyTable.push_back(StreamKey);
	_CurrentEventKeyTable.push_back(-1);
	_ValidDataSourceNameList.push_back(
	    make_pair(DataSourceName, StreamKey)
	);

	DEBUG(cout << "DataSource Registered: ");
	DEBUG(cout << DataSourceName << "[" << DataSourceId << "]: ");
	DEBUG(cout << "segment_header=" << HeaderSection->SectionId() << ", ");
	DEBUG(cout << "segment_data=" << Section->SectionId() << endl);
    }
}

void TKinokoEventEnvelopeReader::ProcessRunBegin(void) throw(TKinokoException)
{
    if (! _IsConstructed) {
	Construct();
    }

    _Processor->ProcessRunBegin();
}

void TKinokoEventEnvelopeReader::ProcessRunEnd(void) throw(TKinokoException)
{
    _Processor->ProcessRunEnd();
}

bool TKinokoEventEnvelopeReader::ProcessPacket(void* Packet, long PacketSize) throw(TKinokoException)
{
    if (_StreamScanner->IsDataDescriptorPacket(Packet)) {
	return ProcessDescriptorPacket(Packet, PacketSize);
    }
    else if (_StreamScanner->IsCommandPacket(Packet)) {
	// do this first for ProcessRunBegin() //
	ProcessCommandPacket(Packet, PacketSize);
    }

    int DataSourceId = TKinokoDataStreamScanner::DataSourceIdOf(Packet);
    if (_DataSourceIndexTable.count(DataSourceId) == 0) {
	return false;
    }
    if (_StreamScanner->IsDataPacket(Packet)) {
	return ProcessDataPacket(Packet, PacketSize);
    }
    if (_StreamScanner->IsTrailerPacket(Packet)) {
	return ProcessTrailerPacket(Packet, PacketSize);
    }

    return false;
}
    
bool TKinokoEventEnvelopeReader::ProcessDescriptorPacket(void* Packet, long PacketSize) throw(TKinokoException)
{
    char* DescriptorText = _StreamScanner->DataDescriptorOf(Packet);
    istringstream DescriptorTextStream(DescriptorText);
    try {
	_DataDescriptor->ReadFrom(DescriptorTextStream);
    }
    catch (TKinokoException &e) {
	throw TKinokoException(
	    "TKinokoEventEnvelopeReader::ProcessDescriptorPacket()",
	    "badly formatted data descriptor: " + e.Message()
	);
    }

    return false;
}

bool TKinokoEventEnvelopeReader::ProcessCommandPacket(void* Packet, long PacketSize) throw(TKinokoException)
{
    int CommandValue = _StreamScanner->CommandValueOf(Packet);
    if (CommandValue == TKinokoDataStreamScanner::Command_RunBegin) {
	_RunBeginCount++;
	if (_RunBeginCount == 1) {
	    ProcessRunBegin();
	}
    }
    else if (CommandValue == TKinokoDataStreamScanner::Command_RunEnd) {
	_RunBeginCount--;
	if (_RunBeginCount == 0) {
	    ProcessRunEnd();
	}
    }

    return false;
}

bool TKinokoEventEnvelopeReader::ProcessDataPacket(void* Packet, long PacketSize) throw(TKinokoException)
{
    int DataSourceId = TKinokoDataStreamScanner::DataSourceIdOf(Packet);
    map<int, int>::iterator DataSourceEntry = _DataSourceIndexTable.find(
	DataSourceId
    );
    if (DataSourceEntry == _DataSourceIndexTable.end()) {
	// this cannot happen //
	return false;
    }
    int DataSourceIndex = DataSourceEntry->second;
    TStreamKey StreamKey = _StreamKeyTable[DataSourceIndex];

    int SectionId = TKinokoDataSectionScanner::SectionIdOf(Packet);
    if (SectionId != _SegmentSectionIdTable[DataSourceIndex]) {
	return false;
    }

    DEBUG(cout << "Receive SegmentBlock: ");
    DEBUG(cout << "@" << DataSourceIndex << ", " << DataSize << endl);

    char* DataArea = (char*) TKinokoDataSectionScanner::DataAreaOf(Packet);
    int DataSize = TKinokoDataSectionScanner::DataSizeOf(Packet);
    TKinokoDataChunk DataChunk(DataArea, DataSize);

    TKinokoEventSegment EventSegment(DataChunk);
    EventSegment.StreamKey = StreamKey;

    if (EventSegment.PieceType() == 0) {
	ostringstream os;
	os << "event segment broken: byte order swapped?: ";
	os << "stream=" << StreamKey;
	_Logger->WriteError(os.str());
	return false;
    }

    TEventKey EventKey = EventSegment.EventKey();
    if (EventKey != _CurrentEventKeyTable[DataSourceIndex]) {
	if (_CurrentEventKeyTable[DataSourceIndex] >= 0) {
	    _Processor->ProcessEventEnd();
	}
	_CurrentEventKeyTable[DataSourceIndex] = EventKey;
	_Processor->ProcessEventBegin(
	    EventSegment.EventKey(), EventSegment.PieceType(), 
	    EventSegment.NumberOfPieces(), EventSegment.TotalSize()
	);
    }
    
    _Processor->ProcessSegment(EventSegment);

    return true;
}

bool TKinokoEventEnvelopeReader::ProcessTrailerPacket(void* Packet, long PacketSize) throw(TKinokoException)
{
    int TrailerValue = _StreamScanner->TrailerValueOf(Packet);
    if (TrailerValue == TKinokoDataStreamScanner::Trailer_Event) {
	int DataSourceId = TKinokoDataStreamScanner::DataSourceIdOf(Packet);
	map<int, int>::iterator DataSourceEntry = _DataSourceIndexTable.find(
	    DataSourceId
	);
	if (DataSourceEntry == _DataSourceIndexTable.end()) {
	    // this cannot happen //
	    return false;
	}
	int DataSourceIndex = DataSourceEntry->second;
	_Processor->ProcessEventEnd();
	_CurrentEventKeyTable[DataSourceIndex] = -1;
    }

    return true;
}
