/* KinokoEventSorter.cc */
/* Created by Enomoto Sanshiro on 15 August 2009. */
/* Last updated by Enomoto Sanshiro on 7 March 2010. */


#include <iostream>
#include <cstdlib>
#include "KinokoEventPiece.hh"
#include "KinokoEventSorter.hh"
#include "KinokoArena.hh"
#include "KinokoBuilderProcessor.hh"

using namespace std;

//#define DEBUG(x) x
#define DEBUG(x)


TKinokoEventSorter::TKinokoEventSorter(TKinokoEventPieceProcessor* OutputPieceProcessor, void* Buffer, TKinokoArena& Arena)
: _Arena(Arena), _ArenaBinder(Arena)
{
    _OutputPieceProcessor = OutputPieceProcessor;

    _ArenaBinder.SetBaseAddress(Buffer);

    _LastBuiltEventKey = -1;
    _LastReceivedEventKey = -1;
    _NumberOfBuiltEvents = 0;

    _BufferReserveSize = (int) (0.01 * _Arena.Capacity());

    DEBUG(cout << "debug flag enabled for KinokoEventSorter.cc" << endl);
}

TKinokoEventSorter::~TKinokoEventSorter()
{
    _Arena.Finalize();
}

int TKinokoEventSorter::RegisterChannel(TChannelKey ChannelKey)
{
    int ChannelIndex = TKinokoEventSorterContext::RegisterChannel(ChannelKey);

    _EventPieceQueueList.push_back(deque<pair<void*, size_t> >());
    _EarlyArrivalPieceSetList.push_back(
	multimap<TEventKey, pair<void*, size_t> >()
    );

    return ChannelIndex;
}

const TKinokoArena& TKinokoEventSorter::Arena(void) const
{
    return _Arena;
}

void TKinokoEventSorter::ProcessRunBegin(void)
{
    _OutputPieceProcessor->ProcessRunBegin();
}

void TKinokoEventSorter::ProcessRunEnd(void)
{
    for (unsigned i = 0; i < _ChannelKeyList.size(); i++) {
	// flush Early-Arrivals //
	multimap<TEventKey, pair<void*, size_t> >::iterator EventEntry;
	for (
	    EventEntry = _EarlyArrivalPieceSetList[i].begin();
	    EventEntry != _EarlyArrivalPieceSetList[i].end();
	    EventEntry++
	){
	    _EventKeyQueueList[i].push_back(EventEntry->first);
	    _EventPieceQueueList[i].push_back(EventEntry->second);
	}
	_EarlyArrivalPieceSetList[i].clear();

	// mark everything completed //
	_CompleteEventKeyList[i] = _LastReceivedEventKey;
    }

    BuildAll();

    _OutputPieceProcessor->ProcessRunEnd();
}

int TKinokoEventSorter::ProcessPiece(const TKinokoEventPiece& EventPiece)
{
    int PieceType = EventPiece.PieceType;
    int PieceFlag = EventPiece.PieceFlag;
    TChannelKey ChannelKey = EventPiece.ChannelKey;
    TEventKey EventKey = EventPiece.EventKey;
    bool IsPossiblyEarlyArrival = PieceFlag & PieceFlag_PossiblyEarlyArrival;
    bool IsLastOfTheEvent = PieceFlag & PieceFlag_LastOfTheEvent;

    _BufferReserveSize = max(_BufferReserveSize, 3*EventPiece.Size());

    int ChannelIndex;
    map<TChannelKey, int>::iterator ChannelIndexEntry = (
	_ChannelIndexTable.find(ChannelKey)
    );
    if (ChannelIndexEntry == _ChannelIndexTable.end()) {
	ChannelIndex = RegisterChannel(ChannelKey);
	if (_LastBuiltEventKey >= 0) {
	    ostringstream os;
	    os << "Late registration: channel=" << ChannelKey;
	    _Logger->WriteWarning(os.str());
	}
    }
    else {
	ChannelIndex = ChannelIndexEntry->second;
    }
    _ChannelPieceCountList[ChannelIndex]++;

    DEBUG(cout << "Sorter::ProcessPiece(): ");
    DEBUG(cout << "type=" << PieceType << ", ");
    DEBUG(cout << "flag=" << PieceFlag << ", ");
    DEBUG(cout << "channel=" << ChannelKey << ", ");
    DEBUG(cout << "index=" << ChannelIndex << ", ");
    DEBUG(cout << "event=" << EventPiece.EventKey << ", ");
    DEBUG(cout << "size=" << EventPiece.Size());
    DEBUG(cout << (IsPossiblyEarlyArrival ? ", early" : ""));
    DEBUG(cout << (IsLastOfTheEvent ? ", completed" : ""));
    DEBUG(cout << endl);

    if (PieceType == PieceType_Normal) {
	;
    }
    else if (PieceType == PieceType_Padding) {
	_Logger->WriteWarning("Padding piece discarded by sorter");
	return 1;
    }
    else if (PieceType == PieceType_Unaligned) {
	DEBUG(cout << "Sorter: Unaligned Piece Received" << endl);
	;
    }
    else if (PieceType == PieceType_Corrupted) {
	DEBUG(cout << "Sorter: Corrupted Piece Forwarded" << endl);
	return _OutputPieceProcessor->ProcessPiece(EventPiece);
    }
    else {
	cerr << "INTERNAL ERROR: ";
	cerr << "TKinokoEventSorter::ProcessEventPiece(): ";
	cerr << "invalid piece type" << endl;
	abort();
    }

    // check out-of-order arrivals //
    if (
	(! _EventKeyQueueList[ChannelIndex].empty()) &&
	(EventKey < _EventKeyQueueList[ChannelIndex].back())
    ){
	ostringstream os;
	os << "Out-of-order arrival: ";
	os << "channel=" << ChannelKey << ", event=" << EventKey;
	os << ", last_event=" << _EventKeyQueueList[ChannelIndex].back();
	_Logger->WriteWarning(os.str());
	TKinokoEventPiece NewEventPiece(EventPiece);
	NewEventPiece.PieceType = PieceType_Unaligned;
	return _OutputPieceProcessor->ProcessPiece(NewEventPiece);
    }

    // check late arrivals //
    if (EventKey <= _LastBuiltEventKey) {
	if (_LateArrivalLengthList[ChannelIndex] == 0) {
	    ostringstream os;
	    os << "Late arrival: ";
	    os << "channel=" << ChannelKey << ", event=" << EventKey;
	    os << ", last_built=" << _LastBuiltEventKey;
	    _Logger->WriteWarning(os.str());
	}
	_LateArrivalLengthList[ChannelIndex]++;

	TKinokoEventPiece NewEventPiece(EventPiece);
	NewEventPiece.PieceType = PieceType_Unaligned;
	return _OutputPieceProcessor->ProcessPiece(NewEventPiece);
    }
    else {
	_LateArrivalLengthList[ChannelIndex] = 0;
    }

    // check early arrivals //
    if (PieceType == PieceType_Unaligned) {
	// unaligned, but not too late //
#if 0
	ostringstream os;
	os << "Late arrival salvaged: ";
	os << "channel=" << ChannelKey << ", event=" << EventKey;
	_Logger->WriteInformation(os.str());
#endif
	IsPossiblyEarlyArrival = true;
    }
    if (IsPossiblyEarlyArrival) {
	if (
	    (! _EventKeyQueueList[ChannelIndex].empty()) &&
	    (EventKey == _EventKeyQueueList[ChannelIndex].back()) 
	){
	    IsPossiblyEarlyArrival = false;
	}
    }

    // test event-completion //
    if (! IsPossiblyEarlyArrival) {
	if (IsLastOfTheEvent) {
	    _CompleteEventKeyList[ChannelIndex] = EventKey;
	    _ChannelEventCountList[ChannelIndex]++;
	}
	else if (
	    (! _EventKeyQueueList[ChannelIndex].empty()) &&
	    (EventKey != _EventKeyQueueList[ChannelIndex].back()) 
	){
	    _CompleteEventKeyList[ChannelIndex] = (
		_EventKeyQueueList[ChannelIndex].back()
	    );
	    _ChannelEventCountList[ChannelIndex]++;
	}
    }
    _LastReceivedEventKey = max(_LastReceivedEventKey, EventKey);

    // copy to buffer //
    // TODO: put early arrivals somewhere other than normal buffer //
    void* DataPiece;
    size_t DataPieceSize = EventPiece.Size();
    if (! _ArenaBinder.CreateEntry(DataPieceSize, DataPiece)) {
	ostringstream os;
	os << "Out of memory: piece unbuilt: ";
	os << "event=" << EventPiece.EventKey << ", ";
	os << "channel=" << ChannelKey << ", ";
	os << "size=" << EventPiece.Size();
	_Logger->WriteWarning(os.str());
	TKinokoEventPiece NewEventPiece(EventPiece);
	NewEventPiece.PieceType = PieceType_Unaligned;
	return _OutputPieceProcessor->ProcessPiece(NewEventPiece);
    }
    EventPiece.WriteTo(DataPiece);
    
    // put into queue //
    if (IsPossiblyEarlyArrival) {
	// early arrivals into early arrival set //
	DEBUG(cout << "Sorter: early arrival: ");
	DEBUG(cout << ChannelKey << " " << EventKey << endl);
	_EarlyArrivalEventKeySetList[ChannelIndex].insert(EventKey);
	_EarlyArrivalPieceSetList[ChannelIndex].insert(
	    make_pair(EventKey, make_pair(DataPiece, DataPieceSize))
	);
    }
    else {
	// put into regular queue //
	_EventKeyQueueList[ChannelIndex].push_back(EventKey);
	_EventPieceQueueList[ChannelIndex].push_back(
	    make_pair(DataPiece, DataPieceSize)
	);
    }
    
    _HoldingEventSet[EventKey].first += 1;
    _HoldingEventSet[EventKey].second += DataPieceSize;

    if (_Arena.AvailableSize() < (size_t) _BufferReserveSize) {
	// TODO: flush early arrivals first //
	vector<TEventKey> BuiltEventKeyList;
	while (_Arena.AvailableSize() < (size_t) _BufferReserveSize) {
	    TEventKey EventKey = _HoldingEventSet.begin()->first;
	    BuiltEventKeyList.push_back(EventKey);
	    BuildOne();
	}
	ostringstream os;
	os << "Memory shortage: building forced. event(s)=";
	for (unsigned i = 0; i < BuiltEventKeyList.size(); i++) {
	    os << EventKey << " ";
	}
	_Logger->WriteWarning(os.str());
    }

    return 1;
}

int TKinokoEventSorter::BuildOne(TEventKey EventKey)
{
    map<TEventKey, pair<int, size_t> >::iterator EventEntry;

    if (EventKey < 0) {
	if (_HoldingEventSet.empty()) {
	    return -1;
	}
	EventEntry = _HoldingEventSet.begin();
	EventKey = EventEntry->first;
    }
    else {
	EventEntry = _HoldingEventSet.find(EventKey);
	if (EventEntry == _HoldingEventSet.end()) {
	    return -1;
	}
    }

    _OutputPieceProcessor->ProcessEventBegin(
	EventKey, PieceType_Normal, 
	EventEntry->second.first, EventEntry->second.second
    );
    
    // loop with the ChannelKey order //
    for (
	map<TChannelKey, int>::iterator Channel = _ChannelIndexTable.begin();
	Channel != _ChannelIndexTable.end();
	Channel++
    ){
	TChannelKey ChannelKey = Channel->first;
	int ChannelIndex = Channel->second;
	bool IsChannelInvolved = false;

	// process early arrivals first //
	while (! _EarlyArrivalPieceSetList[ChannelIndex].empty()) {
	    multimap<TEventKey, pair<void*, size_t> >::iterator Entry = (
		_EarlyArrivalPieceSetList[ChannelIndex].begin()
	    );
	    if (Entry->first != EventKey) {
		break;
	    }

	    DEBUG(cout << "Sorter: early arrival inserted: ");
	    DEBUG(cout << _ChannelKeyList[ChannelIndex] << " ");
	    DEBUG(cout << Entry->first << endl);

	    pair<void*, size_t> DataPiece = Entry->second;
	    TKinokoEventPiece EventPiece(
		TKinokoDataChunk(DataPiece.first, DataPiece.second)
	    );
	    EventPiece.PieceType = PieceType_Normal;
	    EventPiece.ChannelKey = ChannelKey;
	    EventPiece.EventKey = EventKey;
	    _OutputPieceProcessor->ProcessPiece(EventPiece);

	    _ArenaBinder.DetachEntry(DataPiece.first);
	    _EarlyArrivalPieceSetList[ChannelIndex].erase(Entry);
	    _EarlyArrivalEventKeySetList[ChannelIndex].erase(Entry->first);
	    IsChannelInvolved = true;
	}
	
	// process normal piece queue //
	while (
	    (! _EventKeyQueueList[ChannelIndex].empty()) && 
	    (_EventKeyQueueList[ChannelIndex].front() == EventKey) 
	){
	    pair<void*, size_t> DataPiece = (
		_EventPieceQueueList[ChannelIndex].front()
	    );
	    TKinokoEventPiece EventPiece(
		TKinokoDataChunk(DataPiece.first, DataPiece.second)
	    );
	    EventPiece.PieceType = PieceType_Normal;
	    EventPiece.ChannelKey = ChannelKey;
	    EventPiece.EventKey = EventKey;
	    _OutputPieceProcessor->ProcessPiece(EventPiece);

	    _ArenaBinder.DetachEntry(DataPiece.first);
	    _EventKeyQueueList[ChannelIndex].pop_front();
	    _EventPieceQueueList[ChannelIndex].pop_front();
	    IsChannelInvolved = true;
	}
	
	bool IsChannelAlive = _ChannelWatcherList[ChannelIndex].IsAlive();
	if (IsChannelInvolved) {
	    _ChannelWatcherList[ChannelIndex].AddInvolvedEvent();
	}
	else {
	    _ChannelWatcherList[ChannelIndex].AddAbsentEvent();
	}
	if (_CompleteEventKeyList[ChannelIndex] < EventKey) {
	    if (IsChannelAlive) {
		DEBUG(cout << "Sorter: insufficient piece collection" << endl);
	    }
	    _ChannelWatcherList[ChannelIndex].AddAbandonedEvent();
	}

	if (_ChannelWatcherList[ChannelIndex].IsAlive() != IsChannelAlive) {
	    if (IsChannelAlive) {
		ostringstream os;
		os << "Channel died: channel=" << ChannelKey;
		_Logger->WriteWarning(os.str());
	    }
	    else {
		ostringstream os;
		os << "Channel come back to alive: channel=" << ChannelKey;
		_Logger->WriteInformation(os.str());
	    }
	}
    }

    _OutputPieceProcessor->ProcessEventEnd();
    DEBUG(cout << "Sorter::BuildOne(): Built: event=" << EventKey << endl);

    _HoldingEventSet.erase(EventKey);
    _NumberOfBuiltEvents++;
    _LastBuiltEventKey = EventKey;

    return EventKey;
}

int TKinokoEventSorter::BuildReadies(void)
{
    DEBUG(DumpStatusTo(cout));

    int NumberOfBuilt = 0;

    while (! _HoldingEventSet.empty()) {
	TEventKey EventKeyToBuild = _HoldingEventSet.begin()->first;
	bool IsThisReady = true;
	for (unsigned i = 0; i < _ChannelKeyList.size(); i++) {
	    if (
		_ChannelWatcherList[i].IsAlive() &&
		(_CompleteEventKeyList[i] < EventKeyToBuild)
            ){
		IsThisReady = false;
		break;
	    }
	}
	if (! IsThisReady) {
	    break;
	}

	BuildOne(EventKeyToBuild);
	NumberOfBuilt++;
    }

    DEBUG(cout << "Sorter::BuildReadies(): ");
    DEBUG(cout << NumberOfBuilt << " events built" << endl);

    return NumberOfBuilt;
}

int TKinokoEventSorter::BuildAll(void)
{
    int NumberOfBuilt = BuildReadies();

    while (BuildOne() >= 0) {
	NumberOfBuilt++;
    }

    return NumberOfBuilt;
}



TKinokoEventSorterContext::TKinokoEventSorterContext(void)
{
}

TKinokoEventSorterContext::~TKinokoEventSorterContext()
{
}

int TKinokoEventSorterContext::RegisterChannel(TChannelKey ChannelKey)
{
    int ChannelIndex = _ChannelKeyList.size();

    _ChannelIndexTable[ChannelKey] = ChannelIndex;
    _ChannelKeyList.push_back(ChannelKey);
    _ChannelWatcherList.push_back(TKinokoChannelWatcher());
    _EventKeyQueueList.push_back(deque<TEventKey>());
    _CompleteEventKeyList.push_back(-1);
    _ChannelEventCountList.push_back(0);
    _ChannelPieceCountList.push_back(0);
    _EarlyArrivalEventKeySetList.push_back(set<TEventKey>());
    _LateArrivalLengthList.push_back(0);

    return ChannelIndex;
}

void TKinokoEventSorterContext::DumpStatusTo(std::ostream& os)
{
    os << "========= Sorter Context ==========" << endl;
    os << "  NumberOfBuiltEvents: " << _NumberOfBuiltEvents << endl;
    os << "  LastBuiltEventKey: " << _LastBuiltEventKey << endl;
    os << "  LastReceivedEventKey: " << _LastReceivedEventKey << endl;

    std::map<TEventKey, std::pair<int, size_t> >::iterator Event;
    for (
	Event = _HoldingEventSet.begin(); 
	Event != _HoldingEventSet.end(); 
	Event++
    ){
	os << "  --------------------" << endl;
	os << "  EventKey: " << Event->first << endl;
	os << "  NPieces: " << Event->second.first << endl;
	os << "  TotalSize: " << Event->second.second << endl;
    }

    for (
	map<TChannelKey, int>::iterator Channel = _ChannelIndexTable.begin();
	Channel != _ChannelIndexTable.end();
	Channel++
    ){
	int i = Channel->second;
	os << "  --------------------" << endl;
	os << "  Channel: " << _ChannelKeyList[i] << endl;
	os << "  IsAlive: " << _ChannelWatcherList[i].IsAlive() << endl;
	os << "  CompleteEvent: " << _CompleteEventKeyList[i] << endl;

	os << "  QueuedEvents: ";
	for (unsigned j = 0; j < _EventKeyQueueList[i].size(); j++) {
	    os << _EventKeyQueueList[i][j] << " ";
	}
	os << endl;

	os << "  EarlyArrivals: ";
	set<TEventKey>::iterator EventKey;
	for (
	    EventKey = _EarlyArrivalEventKeySetList[i].begin();
	    EventKey != _EarlyArrivalEventKeySetList[i].begin();
	    EventKey++
	){
	    os << *EventKey << endl;
	}
	os << endl;
    }
	
    os << "===================================" << endl;
}

void TKinokoEventSorterContext::DumpChannelStatTo(std::ostream& os)
{
    for (
	map<TChannelKey, int>::iterator Channel = _ChannelIndexTable.begin();
	Channel != _ChannelIndexTable.end();
	Channel++
    ){
	int ChannelIndex = Channel->second;
	TStreamKey ChannelNumber = _ChannelKeyList[ChannelIndex];
	long ChannelEventCount = _ChannelEventCountList[ChannelIndex];
	int NumberOfPieces = _EventKeyQueueList[ChannelIndex].size();
	os << ChannelNumber << ": ";
	os << ChannelEventCount << " ";
	for (int i = 0; i < NumberOfPieces; i++) {
	    os << "*";
	}
	os << endl;
    }
    os << endl;
}
