/* KinokoSorterController.cc */
/* Created by Enomoto Sanshiro on 8 May 2010. */
/* Last updated by Enomoto Sanshiro on 18 May 2010. */


#include <iostream>
#include <iomanip>
#include "KinokoEventSorter.hh"
#include "KinokoSorterController.hh"

using namespace std;

//#define DEBUG(x) x
#define DEBUG(x)


static const int DefaultStartPacketCycles = 20;


TKinokoSorterController::TKinokoSorterController(TKinokoEventSorter& Sorter, const TKinokoArena& Arena)
: _Sorter(Sorter), _Arena(Arena)
{
    _StartDataSize = 0;
    _StartPacketCycles = 0;
    _IsReadyToStart = false;

    DEBUG(cout << "debug flag enabled for KinokoSorterController.cc" << endl);
}

TKinokoSorterController::~TKinokoSorterController()
{
}

void TKinokoSorterController::SetStartDataSize(size_t StartDataSize)
{
    _StartDataSize = StartDataSize;
}

void TKinokoSorterController::SetStartPacketCycles(int NumberOfCycles)
{
    _StartPacketCycles = NumberOfCycles;
}

void TKinokoSorterController::RegisterStream(TStreamKey StreamKey)
{
    DEBUG(cout << "Register Stream: stream=" << StreamKey << endl);

    int ChannelIndex = _Sorter.RegisterChannel(StreamKey);
    _RegisteredChannelIndexList.push_back(ChannelIndex);
}

void TKinokoSorterController::ProcessRunBegin(void)
{
    // default testing method is number of packet cycles per channel //
    if (
	_RegisteredChannelIndexList.empty() && 
	(_StartDataSize == 0) && 
	(_StartPacketCycles == 0)
    ){
	_StartPacketCycles = DefaultStartPacketCycles;
    }
}

void TKinokoSorterController::ProcessRunEnd(void)
{
    if (! _IsReadyToStart) {
	_IsReadyToStart = true;
	DEBUG(cout << "Building started: by RunEnd" << endl);
	DoTransaction();
    }
}

void TKinokoSorterController::DoTransaction(void)
{
    DEBUG(_Sorter.DumpChannelStatTo(cout));

    if (! _IsReadyToStart) {
	TestReadinessToStart();
    }
    if (_IsReadyToStart) {
	//... TODO: this doesn't have to be on every transaction ...//
	_Sorter.BuildReadies();
    }
}

void TKinokoSorterController::TestReadinessToStart(void)
{
    if (_StartDataSize > 0) {
	if (_Arena.UsedSize() > _StartDataSize) {
	    DEBUG(cout << "Building started: ");
	    DEBUG(cout << "by accumulated data size: " << _StartDataSize);
	    DEBUG(cout << endl);
	    _IsReadyToStart = true;
	    return;
	}
    }

    if (! _RegisteredChannelIndexList.empty()) {
	TestReadinessToStartByChannelTable();
	if (_IsReadyToStart) {
	    DEBUG(cout << "Building started: ");
	    DEBUG(cout << "by arrival of all channel data" << endl);
	    return;
	}
    }

    if (_StartPacketCycles > 0) {
	TestReadinessToStartByPacketCycles();
	if (_IsReadyToStart) {
	    DEBUG(cout << "Building started: ");
	    DEBUG(cout << "by number of packet cycles per channel: ");
	    DEBUG(cout << _StartPacketCycles << endl);
	    return;
	}
    }

    if (_Sorter.NumberOfBuiltEvents() > 0) {
	DEBUG(cout << "Building started (forced): ");
	DEBUG(cout << "by buffer almost full" << endl);
	_IsReadyToStart = true;
	return;
    }
}

void TKinokoSorterController::TestReadinessToStartByChannelTable(void)
{
    unsigned Count = 0;
    for (unsigned i = 0; i < _RegisteredChannelIndexList.size(); i++) {
	int ChannelIndex = _RegisteredChannelIndexList[i];
	if (_Sorter.EventKeyQueueList()[ChannelIndex].empty()) {
	    break;
	}
	Count++;
    }

    if (Count == _RegisteredChannelIndexList.size()) {
	_IsReadyToStart = true;
    }
}

void TKinokoSorterController::TestReadinessToStartByPacketCycles(void)
{
    if (_Sorter.ChannelPieceCountList().empty()) {
	return;
    }
    if (_ChannelPieceCountList.size() < _Sorter.EventKeyQueueList().size()) {
	// new channel registered; reset the packet counter //
	_ChannelPieceCountList = _Sorter.ChannelPieceCountList();
	_ChannelPacketCountList.assign(_ChannelPieceCountList.size(), 0);
	DEBUG(cout << "SorterController: New Channel Registered. ");
	DEBUG(cout << "n=" << _ChannelPacketCountList.size() << endl);
	return;
    }

    unsigned NumberOfFilledChannels = 0;
    for (unsigned i = 0; i < _ChannelPieceCountList.size(); i++) {
	if (_Sorter.ChannelPieceCountList()[i] > _ChannelPieceCountList[i]) {
	    _ChannelPacketCountList[i]++;
	}
	if (_ChannelPacketCountList[i] >= _StartPacketCycles) {
	    NumberOfFilledChannels++;
	}
	DEBUG(cout << _Sorter.ChannelKeyList()[i] << ": ");
	DEBUG(cout << _Sorter.ChannelPieceCountList()[i] << ",");
	DEBUG(cout << _Sorter.ChannelEventCountList()[i] << ": ");
	DEBUG(cout << _ChannelPacketCountList[i] << endl);
    }
    if (NumberOfFilledChannels == _ChannelPieceCountList.size()) {
	_IsReadyToStart = true;
    }

    _ChannelPieceCountList = _Sorter.ChannelPieceCountList();
}
