/* KinokoCollectorCom.cc */
/* Created by Enomoto Sanshiro on 11 October 2000. */
/* Last updated by Enomoto Sanshiro on 18 January 2002. */


#include <iostream>
#include <sstream>
#include <string>
#include <vector>
#include "MushMisc.hh"
#include "KinokoDaqFrontend.hh"
#include "KinokoStreamSourceComponent.hh"
#include "KinokoCollectorCom.hh"

using namespace std;


static const string Comment = "Collector: DAQ front-end process";


TKinokoCollectorCom::TKinokoCollectorCom(void)
: TKinokoStreamSourceComponent("KinokoCollectorCom")
{
    _DaqFrontend = new TKinokoNullDaqFrontend();

    _IsDisabled = false;

    _IsStopped = false;
    _MaxEventCounts = 0;
    _EventCounts = 0;
    _RunLength = 0;
    _StartTime = 0;
}

TKinokoCollectorCom::~TKinokoCollectorCom()
{
    delete _DaqFrontend;
}

void TKinokoCollectorCom::BuildDescriptor(TKcomComponentDescriptor& Descriptor)
{
    TKinokoStreamSourceComponent::BuildDescriptor(Descriptor);
    Descriptor.AddComment(Comment);

    TKcomEventDeclaration SetScriptEvent("setReadoutScript");
    SetScriptEvent.AddArgument(TKcomPropertyDeclaration(
        "file_name", TKcomPropertyDeclaration::Type_String
    ));
    SetScriptEvent.AddArgument(TKcomPropertyDeclaration(
        "datasource_name", TKcomPropertyDeclaration::Type_String
    ));
    Descriptor.RegisterEventSlot(EventId_SetReadoutScript, SetScriptEvent);

    TKcomEventDeclaration SetMaxEventCountsEvent("setMaxEventCounts");
    SetMaxEventCountsEvent.AddArgument(TKcomPropertyDeclaration(
        "number_of_events", TKcomPropertyDeclaration::Type_Int
    ));
    Descriptor.RegisterEventSlot(EventId_SetMaxEventCounts, SetMaxEventCountsEvent);

    TKcomEventDeclaration SetRunLengthEvent("setRunLength");
    SetRunLengthEvent.AddArgument(TKcomPropertyDeclaration(
        "run_length_sec", TKcomPropertyDeclaration::Type_Int
    ));
    Descriptor.RegisterEventSlot(EventId_SetRunLength, SetRunLengthEvent);

    TKcomEventDeclaration ExecuteCommandEvent("executeCommand");
    ExecuteCommandEvent.AddArgument(TKcomPropertyDeclaration(
        "command_name", TKcomPropertyDeclaration::Type_String
    ));
    ExecuteCommandEvent.AddArgument(TKcomPropertyDeclaration(
        "parameter", TKcomPropertyDeclaration::Type_Int
    ));
    Descriptor.RegisterEventSlot(EventId_ExecuteCommand, ExecuteCommandEvent);

    TKcomEventDeclaration DisableEvent("disable");
    Descriptor.RegisterEventSlot(EventId_Disable, DisableEvent);

    TKcomEventDeclaration FinishedEvent("dataAcquisitionFinished");
    Descriptor.RegisterEventSource(EventId_DataAcquisitionFinished, FinishedEvent);
}

int TKinokoCollectorCom::ProcessEvent(int EventId, TKcomEvent& Event, TKcomEventResponse& EventResponse)
{
    int Result = 0;

    switch (EventId) {
      case EventId_SetReadoutScript:
	Result = ProcessSetReadoutScriptEvent(Event);
	break;

      case EventId_SetMaxEventCounts:
	Result = ProcessSetMaxEventCountsEvent(Event);
	break;

      case EventId_SetRunLength:
	Result = ProcessSetRunLengthEvent(Event);
	break;

      case EventId_ExecuteCommand:
	Result = ProcessExecuteCommandEvent(Event);
	break;

      case EventId_Disable:
	Result = ProcessDisableEvent(Event);
	break;

      default:
	Result = TKinokoStreamSourceComponent::ProcessEvent(
	    EventId, Event, EventResponse
	);
    }

    return Result;
}

int TKinokoCollectorCom::ProcessSetReadoutScriptEvent(TKcomEvent& Event)
{
    if (Event.ArgumentList().size() < 1) {
	_Logger->WriteError(
	    ComponentName(), "setReadoutScript(): too few arguments"
        );
	return 1;
    }

    _ScriptFileName = Event.ArgumentList()[0];
    if (Event.ArgumentList().size() > 1) {
	_DataSourceName = Event.ArgumentList()[1];
    }
    
    _Logger->WriteDebug(
	ComponentName(), 
	"setReadoutScript(): " + _ScriptFileName + ", " + _DataSourceName
    );

    return 1;
}

int TKinokoCollectorCom::ProcessSetMaxEventCountsEvent(TKcomEvent& Event)
{
    if (Event.ArgumentList().size() < 1) {
	_Logger->WriteError(
	    ComponentName(), "setMaxEventCounts(): too few arguments"
        );
    }
    else {
	string MaxEventCountsString = Event.ArgumentList()[0];
	_Logger->WriteDebug(
	    ComponentName(), "setMaxEventCounts(): " + MaxEventCountsString
	);

	if (! (istringstream(MaxEventCountsString) >> _MaxEventCounts)) {
	    _MaxEventCounts = 0;
	    _Logger->WriteError(
		ComponentName(), "setMaxEventCounts(): invalid argument"
	    );
	}
    }

    return 1;
}

int TKinokoCollectorCom::ProcessSetRunLengthEvent(TKcomEvent& Event)
{
    if (Event.ArgumentList().size() < 1) {
	_Logger->WriteError(
	    ComponentName(), "setRunLength(): too few arguments"
        );
    }
    else {
	string RunLengthString = Event.ArgumentList()[0];
	_Logger->WriteDebug(
	    ComponentName(), "setRunLength(): " + RunLengthString
	);

	if (! (istringstream(RunLengthString) >> _RunLength)) {
	    _RunLength = 0;
	    _Logger->WriteError(
		ComponentName(), "setRunLength(): invalid argument"
	    );
	}
    }

    return 1;
}

int TKinokoCollectorCom::ProcessDisableEvent(TKcomEvent& Event)
{
    _IsDisabled = true;
    _Logger->WriteDebug(ComponentName(), "disable()");

    return 1;
}

void TKinokoCollectorCom::Construct(void) throw(TKinokoException)
{
    if (_ScriptFileName.empty()) {
	_Logger->WriteError(
	    ComponentName(), 
	    "initialization fault: script file is not specified."
        );
	return;
    }

    if (! _IsDisabled) {
	delete _DaqFrontend;
	_DaqFrontend = new TKinokoDaqFrontend(TypeName());
    }

    try {
	_DaqFrontend->Construct(
	    _ScriptFileName, _DataSourceName, DataSourceId(), 
	    _OutputDataStream, _EventEmitter, _Registry
	);
    }
    catch (TKinokoException&e) {
	delete _DaqFrontend;
	_DaqFrontend = new TKinokoNullDaqFrontend();

	_Logger->WriteError(
	    ComponentName(), "initialization fault: " + e.Message()
        );
    }
}

void TKinokoCollectorCom::Destruct(void) throw(TKinokoException)
{
    _DaqFrontend->Destruct();
    
    delete _DaqFrontend;
    _DaqFrontend = new TKinokoNullDaqFrontend();

    _IsDisabled = false;

    _IsStopped = false;
    _MaxEventCounts = 0;
    _RunLength = 0;
}

void TKinokoCollectorCom::OnStart(void) throw(TKinokoException)
{
    _Logger->WriteRemarkable(ComponentName(), "starting data taking");

    try {
	_DaqFrontend->OnStart();
    }
    catch (TKinokoException &e) {
	_Logger->WriteError(ComponentName(), e.Message());

	_Logger->WriteNotice(ComponentName(), "component disabled");
	_DaqFrontend->Destruct();
	delete _DaqFrontend;
	_DaqFrontend = new TKinokoNullDaqFrontend();
    }

    _IsStopped = false;
    _IsStopRequested = false;
    _EventCounts = 0;
    _StartTime = TMushDateTime::SecSinceEpoch();
}

void TKinokoCollectorCom::OnStop(void) throw(TKinokoException)
{
    try {
	_DaqFrontend->OnStop();
    }
    catch (TKinokoException &e) {
	_Logger->WriteError(ComponentName(), e.Message());

	_Logger->WriteNotice(ComponentName(), "component disabled");
	_DaqFrontend->Destruct();
	delete _DaqFrontend;
	_DaqFrontend = new TKinokoNullDaqFrontend();
    }

    _Logger->WriteRemarkable(ComponentName(), "data taking stopped");
}

int TKinokoCollectorCom::ProcessData(void)throw(TKinokoException)
{
    if ((_MaxEventCounts > 0) && (_EventCounts >= _MaxEventCounts)) {
	_IsStopRequested = true;
    }
    if ((_RunLength > 0) && (TMushDateTime::SecSince(_StartTime) >= _RunLength)) {
	_IsStopRequested = true;
    }

    if (_IsStopRequested) {
	if (! _IsStopped) {
	    TKcomEvent Event;
	    EmitEventOneWay(EventId_DataAcquisitionFinished, Event);
	    _IsStopped = true;
	}
	return 0;
    }

    int Result = _DaqFrontend->DoTransaction();
    if (Result) {
	_EventCounts++;
    }
    else {
	_IsStopRequested = _DaqFrontend->IsStopRequested();
    }

    return Result;
}

int TKinokoCollectorCom::ProcessExecuteCommandEvent(TKcomEvent& Event)
{
    if (Event.ArgumentList().size() < 1) {
	_Logger->WriteError(
	    ComponentName(), 
	    "executeCommand(string command, ...): too few arguments"
        );
	return 1;
    }

    string Command = Event.ArgumentList()[0];

    int NumberOfParameters = Event.ArgumentList().size() - 1;

    static const int MaxNumberOfParameters = 128;   //...
    static int ParameterList[MaxNumberOfParameters];
    if (NumberOfParameters > MaxNumberOfParameters) {
	_Logger->WriteError(
	    ComponentName(), 
	    "executeCommand(string command, ...): too many arguments"
	);
	NumberOfParameters = MaxNumberOfParameters;
    } 

    for (int i = 0; i < NumberOfParameters; i++) {
	string ParameterString = Event.ArgumentList()[i + 1];
	istringstream ParameterStream(ParameterString);
	if (! (ParameterStream >> ParameterList[i])) {
	    _Logger->WriteError(
		ComponentName(), 
		"executeCommand(string command, ...): invalid arguments"
	    );
	    ParameterList[i] = 0;
	}
    }

    try {
	_DaqFrontend->ExecuteDaqCommand(
	    Command, ParameterList, NumberOfParameters
	);
    }
    catch (TKinokoException &e) {
	_Logger->WriteError(ComponentName(), e.Message());
    }

    return 1;
}

