/* KinokoViewerCom.cc */
/* Created by Enomoto Sanshiro on 31 October 2000. */
/* Last updated by Enomoto Sanshiro on 24 June 2002. */


#include <iostream>
#include <iomanip>
#include <string>
#include "MushFileSystem.hh"
#include "MushNetworkSocket.hh"
#include "KinokoViewer.hh"
#include "KinokoStreamSinkComponent.hh"
#include "KinokoViewerCom.hh"

using namespace std;


static const string Comment = "Viewer: Kinoko Data Visualizer";


TKinokoViewerCom::TKinokoViewerCom(void)
: TKinokoStreamSinkComponent("KinokoViewerCom")
{
    _Viewer = 0;
}

TKinokoViewerCom::~TKinokoViewerCom()
{
    delete _Viewer;
}

void TKinokoViewerCom::BuildDescriptor(TKcomComponentDescriptor& Descriptor)
{
    TKinokoStreamSinkComponent::BuildDescriptor(Descriptor);
    Descriptor.AddComment(Comment);

    TKcomEventDeclaration SetScriptEvent("setViewScript");
    SetScriptEvent.AddArgument(TKcomPropertyDeclaration(
        "file_name", TKcomPropertyDeclaration::Type_String
    ));
    SetScriptEvent.AddArgument(TKcomPropertyDeclaration(
        "display_name", TKcomPropertyDeclaration::Type_String
    ));
    Descriptor.RegisterEventSlot(EventId_SetViewScript, SetScriptEvent);

    TKcomEventDeclaration ResetEvent("reset");
    Descriptor.RegisterEventSlot(EventId_Reset, ResetEvent);

    TKcomEventDeclaration ExecuteCommandEvent("executeCommand");
    ExecuteCommandEvent.AddArgument(TKcomPropertyDeclaration(
        "command_name", TKcomPropertyDeclaration::Type_String
    ));
    Descriptor.RegisterEventSlot(EventId_ExecuteCommand, ExecuteCommandEvent);

    TKcomEventDeclaration SaveImageEvent("saveImage");
    SaveImageEvent.AddArgument(TKcomPropertyDeclaration(
        "file_name", TKcomPropertyDeclaration::Type_String
    ));
    Descriptor.RegisterEventSlot(EventId_SaveImage, SaveImageEvent);
}

void TKinokoViewerCom::Initialize(void) throw(TKcomException)
{
    TKinokoStreamSinkComponent::Initialize();

    _Viewer = new TKinokoViewer();
    _Viewer->AttachPlatform(
	InputStream(), OutputStream(), _EventEmitter, _Registry, _Logger
    );

    OutputStream() << ".start;" << endl;
    OutputStream() << ".set title " << ComponentName() << ";" << endl;
}

void TKinokoViewerCom::Finalize(void) throw(TKcomException)
{
    OutputStream() << ".stop;" << endl;
    OutputStream() << ".quit;" << endl;

    TKinokoStreamSinkComponent::Finalize();
}

int TKinokoViewerCom::ProcessEvent(int EventId, TKcomEvent& Event, TKcomEventResponse& EventResponse)
{
    int Result = 0;

    switch (EventId) {
      case EventId_SetViewScript:
	Result = ProcessSetViewScriptEvent(Event);
	break;

      case EventId_Reset:
	Result = ProcessResetEvent(Event);
	break;

      case EventId_ExecuteCommand:
	Result = ProcessExecuteCommandEvent(Event);
	break;

      case EventId_SaveImage:
	Result = ProcessSaveImageEvent(Event);
	break;

      default:
	Result = TKinokoStreamSinkComponent::ProcessEvent(
	    EventId, Event, EventResponse
	);
    }

    return Result;
}

int TKinokoViewerCom::ProcessSetViewScriptEvent(TKcomEvent& Event)
{
    if (Event.ArgumentList().size() < 1) {
	_Logger->WriteError(
	    ComponentName(), "setViewScript(): too few arguments"
        );
	return 0;
    }

    _ScriptFileName = Event.ArgumentList()[0];
    if (Event.ArgumentList().size() > 1) {
	_DisplayEntryName = Event.ArgumentList()[1];
    }
    
    _Logger->WriteDebug(
	ComponentName(), "setViewScript(): " + _ScriptFileName
    );

    return 1;
}

int TKinokoViewerCom::ProcessResetEvent(TKcomEvent& Event)
{
    _Viewer->Reset();
    return 1;
}

int TKinokoViewerCom::ProcessExecuteCommandEvent(TKcomEvent& Event)
{
    if (Event.ArgumentList().size() < 1) {
	_Logger->WriteError(
	    ComponentName(), 
	    "executeCommand(string command, ...): too few arguments"
        );
	return 0;
    }

    string Command = Event.ArgumentList()[0];

    try {
	_Viewer->ExecuteCommand(Command);
    }
    catch (TKinokoException &e) {
	_Logger->WriteError(ComponentName(), e.Message());
    }

    return 1;
}

int TKinokoViewerCom::ProcessSaveImageEvent(TKcomEvent& Event)
{
    if (Event.ArgumentList().size() < 1) {
	_Logger->WriteError(
	    ComponentName(), "saveImage(): too few arguments"
        );
	return 0;
    }

    string ImageFileName = Event.ArgumentList()[0];
#if 0
    _Logger->WriteDebug(ComponentName(), "saveImage(): " + ImageFileName);
#endif

    int NumberOfImages;
    try {
	NumberOfImages = _Viewer->SaveImage(ImageFileName);
    }
    catch (TKinokoException &e) {
	_Logger->WriteError(ComponentName(), e.Message());
    }    

    if (NumberOfImages > 0) {
	string Host = TMushNetworkSocket::LocalIPAddress();
	string Path = TMushFileAttribute(ImageFileName).PathName();
	string Name = TMushFileAttribute(ImageFileName).FileRootName();
	string Extension = TMushFileAttribute(ImageFileName).Extension();
	if (Path.empty()) {
	    Path = TMushFileSystem::CurrentDirectory() + "/";
	}
	string RegistryPath = "/files/image/" + ComponentName() + "/" + Name;

	Registry()->SetValue(RegistryPath + "/host", Host);
	Registry()->SetValue(RegistryPath + "/path", Path);
	Registry()->SetLongValue(RegistryPath + "/pages", NumberOfImages);

	for (int Page = 0; Page < NumberOfImages; Page++) {
	    ostringstream RegistryPathStream, FileNameStream;
	    FileNameStream << Name;
	    if (NumberOfImages > 1) {
		FileNameStream << "-" << Page;
	    }
	    FileNameStream << "." << Extension;
	    
	    RegistryPathStream << RegistryPath;
	    RegistryPathStream << "/page" << Page << "/filename";
	    
	    Registry()->SetValue(
		RegistryPathStream.str(), FileNameStream.str()
	    );
	}
    }

    return 1;
}

void TKinokoViewerCom::Construct(void) throw(TKinokoException)
{
    if (_ScriptFileName.empty()) {
	_Logger->WriteError(
	    ComponentName(), 
	    "initialization fault: script file is not specified."
        );
	return;
    }
    
    try {
	_Viewer->SetStreamCommandProcessor(_StreamCommandProcessor);
	_Viewer->SetScript(_ScriptFileName, _DisplayEntryName);
	_Viewer->Construct(ComponentName(), _InputDataStream);

	if (_IsOutputStreamUsed) {
	    _Viewer->SetChainedOutput(_OutputDataStream);
	}
    }
    catch (TScriptException &e) {
	_Logger->WriteError(
	    ComponentName(), "initialization fault: " + e.Message()
	);
    }
    catch (TKinokoException &e) {
	_Logger->WriteError(
	    ComponentName(), "initialization fault: " + e.Message()
	);
    }

    return;
}

void TKinokoViewerCom::Destruct(void) throw(TKinokoException)
{
    try {
	_Viewer->Destruct();
    }
    catch (TKinokoException &e) {
	_Logger->WriteError(ComponentName(), e.Message());
    }
}

int TKinokoViewerCom::ProcessData(void) throw(TKinokoException)
{
    int Result;

    try {
	Result = _Viewer->ProcessData();
	if (Result <= 0) {
	    _TimeKeeper->Suspend();
	}
    }
    catch (TKinokoException &e) {
	_Logger->WriteError(ComponentName(), e.Message());
    }

    return Result;
}
