/* KinokoIndexedDataSection.cc */
/* Created by Enomoto Sanshiro on 19 November 2000. */
/* Last updated by Enomoto Sanshiro on 12 July 2007. */


#include <iostream>
#include <string>
#include "ParaTokenizer.hh"
#include "KinokoDataSource.hh"
#include "KinokoDataSection.hh"
#include "KinokoIndexedDataSection.hh"

using namespace std;


TKinokoIndexedDataSection::TKinokoIndexedDataSection(TKinokoDataSource* DataSource, const string& SectionName)
: TKinokoDataSection(DataSource, SectionName)
{
    _AddressWidth = 0;
    _SubAddressWidth = 0;
    _DataWidth = 0;
    _ElementWidth = 0;

    _ElementDepth = 0;
    _MaxElementDepth = 0;
    _DataType = ElementType_Integer;

    _Formatter = 0;
    _Scanner = 0;
}

TKinokoIndexedDataSection::TKinokoIndexedDataSection(TKinokoDataSource* DataSource, const string& SectionName, int SectionId)
: TKinokoDataSection(DataSource, SectionName, SectionId)
{
    _AddressWidth = 0;
    _SubAddressWidth = 0;
    _DataWidth = 0;
    _ElementWidth = 0;

    _ElementDepth = 0;
    _MaxElementDepth = 0;
    _DataType = ElementType_Integer;

    _Formatter = 0;
    _Scanner = 0;
}

TKinokoIndexedDataSection::~TKinokoIndexedDataSection()
{
    delete _Formatter;
    delete _Scanner;
}

void TKinokoIndexedDataSection::SetWidth(unsigned AddressWidth, unsigned DataWidth, unsigned SubAddressWidth)
{
    _AddressWidth = AddressWidth;
    _SubAddressWidth = SubAddressWidth;
    _DataWidth = DataWidth;

    _ElementWidth = _AddressWidth + _SubAddressWidth + _DataWidth;
}

void TKinokoIndexedDataSection::SetElementType(int ElementType)
{
    _DataType = ElementType;
}

void TKinokoIndexedDataSection::SetElementDepth(unsigned ElementDepth)
{
    _ElementDepth = ElementDepth;
}

void TKinokoIndexedDataSection::SetMaxElementDepth(unsigned MaxElementDepth)
{
    _MaxElementDepth = MaxElementDepth;
}

void TKinokoIndexedDataSection::SetDataBitOffset(unsigned DataBitOffset)
{
    _DataBitOffset = DataBitOffset;
}

unsigned TKinokoIndexedDataSection::AddressWidth(void) const
{
    return _AddressWidth;
}

unsigned TKinokoIndexedDataSection::DataWidth(void) const
{
    return _DataWidth;
}

unsigned TKinokoIndexedDataSection::ElementWidth(void) const
{
    return _ElementWidth;
}

int TKinokoIndexedDataSection::DataType(void) const
{
    return _DataType;
}

unsigned TKinokoIndexedDataSection::SubAddressWidth(void) const
{
    return _SubAddressWidth;
}

unsigned TKinokoIndexedDataSection::ElementDepth(void) const
{
    return _ElementDepth;
}

unsigned TKinokoIndexedDataSection::MaxElementDepth(void) const
{
    return _MaxElementDepth;
}

unsigned TKinokoIndexedDataSection::DataBitOffset(void) const
{
    return _DataBitOffset;
}

int TKinokoIndexedDataSection::SectionType(void) const
{
    return TKinokoDataSection::SectionType_Indexed;
}

void TKinokoIndexedDataSection::ReadFrom(TParaTokenizer& Tokenizer) throw(TKinokoException)
{
    unsigned AddressWidth = 0, DataWidth = 0, SubAddressWidth = 0;
    unsigned ElementDepth = 0, MaxElementDepth = 0;
    unsigned DataBitOffset = 0;
    int ElementType = ElementType_Integer;

    _ElementWidth = 0;

    try {
	TParaToken Token;

	//... for backward compatibility (KDF2) ...//
	if (Tokenizer.LookAhead().Is("(")) {
	    Tokenizer.Next().MustBe("(");
	    Tokenizer.Next().MustBe("address");
	    Tokenizer.Next().MustBe(":");
	    Tokenizer.Next().MustBe("int");
	    Tokenizer.Next().MustBe("-");
	    AddressWidth = Tokenizer.Next().AsLong();
	    Tokenizer.Next().MustBe("bit");
	    Tokenizer.Next().MustBe(",");
	    Tokenizer.Next().MustBe("data");
	    Tokenizer.Next().MustBe(":");
	    Tokenizer.Next().MustBe("int");
	    Tokenizer.Next().MustBe("-");
	    DataWidth = Tokenizer.Next().AsLong();
	    Tokenizer.Next().MustBe("bit");
	    Tokenizer.Next().MustBe(")");

	    if (AddressWidth + DataWidth > 32) {
		AddressWidth = 32;
		DataWidth = 32;
		DataBitOffset = 32;
	    }
	    else {
		AddressWidth = 32 - DataWidth;
		DataBitOffset = 0;
	    }
	}

	if (Tokenizer.LookAhead().Is("{")) {
	    Tokenizer.Next().MustBe("{");
	    while ((Token = Tokenizer.Next()).IsNot("}")) {
		if (Token.Is("attribute")) {
		    Tokenizer.Unget(Token);
		    ReadAttribute(Tokenizer);
		}
		else if (Token.Is("address")) {
		    Tokenizer.Next().MustBe(":");
		    Tokenizer.Next().MustBe("int");
		    Tokenizer.Next().MustBe("-");
		    AddressWidth = Tokenizer.Next().AsLong();
		    Tokenizer.Next().MustBe("bit");
		    Tokenizer.Next().MustBe(";");
		}
		else if (Token.Is("subaddress")) {
		    Tokenizer.Next().MustBe(":");
		    Tokenizer.Next().MustBe("int");
		    Tokenizer.Next().MustBe("-");
		    SubAddressWidth = Tokenizer.Next().AsLong();
		    Tokenizer.Next().MustBe("bit");
		    Tokenizer.Next().MustBe(";");
		}
		else if (Token.Is("data")) {
		    Tokenizer.Next().MustBe(":");
		    Token = Tokenizer.Next();
		    if (Token.Is("int")) {
			ElementType = ElementType_Integer;
		    }
		    else if (Token.Is("float")) {
			ElementType = ElementType_Float;
		    }
		    else {
			ElementType = ElementType_Unknown;
		    }
		    Tokenizer.Next().MustBe("-");
		    DataWidth = Tokenizer.Next().AsLong();
		    Tokenizer.Next().MustBe("bit");
		    Tokenizer.Next().MustBe(";");
		}
		else if (Token.Is("depth")) {
		    Tokenizer.Next().MustBe(":");
		    ElementDepth = Tokenizer.Next().AsLong();
		    Tokenizer.Next().MustBe(";");
		}
		else if (Token.Is("max_depth")) {
		    Tokenizer.Next().MustBe(":");
		    MaxElementDepth = Tokenizer.Next().AsLong();
		    Tokenizer.Next().MustBe(";");
		}
		else if (Token.Is("element_width")) {
		    Tokenizer.Next().MustBe(":");
		    _ElementWidth = Tokenizer.Next().AsLong();
		    Tokenizer.Next().MustBe("bit");
		    Tokenizer.Next().MustBe(";");
		}
		else {
		    Tokenizer.Unget(Token);
		    TKinokoDataSource::SkipUnknownEntry(Tokenizer);
		}
	    }

	    if ((AddressWidth == 0) || (DataWidth == 0)) {
		throw TKinokoException(
		    "TKinokoIndexedDataSection::ReadFrom()",
		    "address/data width not specified"
		);
	    }
	}
	else {
	    Tokenizer.Next().MustBe(";");
	}
    }
    catch (TScriptException &e) {
	throw TKinokoException(
	    "TKinokoIndexedDataSection::ReadFrom()",
	    "script exception: " + e.Message()
	);
    }

    if (_ElementWidth == 0) {
	// "element_width" is used to skip unknown fields.
	_ElementWidth = AddressWidth + SubAddressWidth + DataWidth;
    }

    SetWidth(AddressWidth, DataWidth, SubAddressWidth);
    SetElementType(ElementType);
    SetElementDepth(ElementDepth);
    SetMaxElementDepth(MaxElementDepth);

    SetDataBitOffset(DataBitOffset);
}

void TKinokoIndexedDataSection::WriteTo(ostream& os, const string& Indent)
{
    os << Indent << "section \"" << _SectionName << "\"";
    os << "<" << _SectionId << ">";
    os << ": indexed";

#if USE_KDF2
    os << "(";
    os << "address: int-" << _AddressWidth << "bit";
    os << ", ";
    os << "data: int-" << _DataWidth << "bit";
    os << ");" << endl;
#else
    os << " {" << endl;
    WriteAttributeList(os, Indent + "    ");
    os << Indent << "    ""address: int-" << _AddressWidth << "bit;" << endl;
    if (_SubAddressWidth > 0) {
	os << Indent << "    ""subaddress: int-" << _SubAddressWidth << "bit;" << endl;
    }
    if (_DataType == ElementType_Float) {
	os << Indent << "    ""data: float-32bit;" << endl;
    }
    else {
	os << Indent << "    ""data: int-" << _DataWidth << "bit;" << endl;
    }
    os << Indent << "    ""element_width: " << _ElementWidth << "bit;" << endl;
    if (_ElementDepth > 0) {
	os << Indent << "    ""depth: " << _ElementDepth << ";" << endl;
    }
    if (_MaxElementDepth > 0) {
	os << Indent << "    ""max_depth: " << _MaxElementDepth << ";" << endl;
    }
    os << Indent << "}" << endl;
#endif
}



TKinokoIndexedDataSectionFormatter::TKinokoIndexedDataSectionFormatter(TKinokoIndexedDataSection* DataSection)
: TKinokoDataSectionFormatter(DataSection)
{
    _AddressWidth = DataSection->AddressWidth();
    _SubAddressWidth = DataSection->SubAddressWidth();
    _DataWidth = DataSection->DataWidth();
    _ElementWidth = DataSection->ElementWidth();

    _DataBitOffset = 0;
    _AddressBitOffset = _DataBitOffset + _DataWidth;
    _SubAddressBitOffset = _AddressBitOffset + _AddressWidth;

#if USE_KDF2
    if (_ElementWidth > 32) {
	_AddressWidth = 32;
	_DataWidth = 32;
	_AddressBitOffset = 0;
	_DataBitOffset = 32;
	_ElementWidth = 64;
    }
    else {
	_AddressWidth = 32 - _DataWidth;
	_ElementWidth = 32;
    }
    _SubAddressWidth = 0;
    _SubAddressBitOffset = _AddressBitOffset;
#endif
}

TKinokoIndexedDataSectionFormatter::~TKinokoIndexedDataSectionFormatter()
{
}



TKinokoIndexedDataSectionScanner::TKinokoIndexedDataSectionScanner(TKinokoIndexedDataSection* DataSection)
{
    _AddressWidth = DataSection->AddressWidth();
    _SubAddressWidth = DataSection->SubAddressWidth();
    _DataWidth = DataSection->DataWidth();
    _ElementWidth = DataSection->ElementWidth();

    _DataBitOffset = DataSection->DataBitOffset();
    _AddressBitOffset = (_DataBitOffset == 0) ? _DataWidth : 0;
    _SubAddressBitOffset = _AddressBitOffset + _AddressWidth;

    _DataType = DataSection->DataType();
}

TKinokoIndexedDataSectionScanner::~TKinokoIndexedDataSectionScanner()
{
}
