/* KinokoIndexedDataSection.hh */
/* Created by Enomoto Sanshiro on 19 November 2000. */
/* Last updated by Enomoto Sanshiro on 5 March 2008. */


#ifndef __KinokoIndexedDataSection_hh__
#define __KinokoIndexedDataSection_hh__

#include <iostream>
#include <sstream>
#include <string>
#include "MushConfig.hh"
#include "ParaTokenizer.hh"
#include "KinokoDataSection.hh"


class TKinokoIndexedDataSectionFormatter;
class TKinokoIndexedDataSectionScanner;


class TKinokoIndexedDataSection: public TKinokoDataSection {
  public:
    TKinokoIndexedDataSection(TKinokoDataSource* DataSource, const std::string& SectionName);
    TKinokoIndexedDataSection(TKinokoDataSource* DataSource, const std::string& SectionName, int SectionId);
    virtual ~TKinokoIndexedDataSection();
    virtual int SectionType(void) const;
    virtual void ReadFrom(TParaTokenizer& Tokenizer) throw(TKinokoException);
    virtual void WriteTo(std::ostream& os, const std::string& Indent);
    inline TKinokoIndexedDataSectionFormatter* Formatter(void);
    inline TKinokoIndexedDataSectionScanner* Scanner(void);
  public:
    void SetWidth(unsigned AddressWidth, unsigned DataWidth, unsigned SubAddressWidth = 0);
    void SetElementType(int ElementType);
    void SetElementDepth(unsigned ElementDepth);
    void SetMaxElementDepth(unsigned MaxElementDepth);
    unsigned AddressWidth(void) const;
    unsigned SubAddressWidth(void) const;
    unsigned DataWidth(void) const;
    unsigned ElementWidth(void) const;
    int DataType(void) const;
    unsigned ElementDepth(void) const;
    unsigned MaxElementDepth(void) const;
  public:
    // internal use only //
    void SetDataBitOffset(unsigned DataBitOffset);
    unsigned DataBitOffset(void) const;
  protected:
    unsigned _AddressWidth, _DataWidth, _SubAddressWidth, _ElementWidth;
    unsigned _ElementDepth, _MaxElementDepth;
    int _DataType;
    unsigned _DataBitOffset;
    TKinokoIndexedDataSectionFormatter* _Formatter;
    TKinokoIndexedDataSectionScanner* _Scanner;
};


class TKinokoIndexedDataSectionFormatter: public TKinokoDataSectionFormatter {
  public:
    TKinokoIndexedDataSectionFormatter(TKinokoIndexedDataSection* DataSection);
    virtual ~TKinokoIndexedDataSectionFormatter();
    inline int DataSizeFor(int NumberOfElements) const;
    inline void WriteTo(void* Buffer, int Index, int Address, int Data) const;
    inline void WriteTo(void* Buffer, int Index, int Address, int SubAddress, int Data) const;
    inline void WriteFloatTo(void* Buffer, int Index, int Address, float Data) const;
    inline void WriteFloatTo(void* Buffer, int Index, int Address, int SubAddress, float Data) const;
    inline void WriteArrayTo(void* Buffer, int StartIndex, int Address, int DataArray[], int DataArrayLength) const;
    inline void WriteArrayTo(void* Buffer, int StartIndex, int Address, int SubAddress, int DataArray[], int DataArrayLength) const;
    inline void WriteFloatArrayTo(void* Buffer, int StartIndex, int Address, int SubAddress, float DataArray[], int DataArrayLength) const;
    inline void WriteFloatArrayTo(void* Buffer, int StartIndex, int Address, float DataArray[], int DataArrayLength) const;
  protected:
    unsigned _AddressWidth, _DataWidth, _SubAddressWidth, _ElementWidth;
    unsigned _AddressBitOffset, _DataBitOffset, _SubAddressBitOffset;
};


class TKinokoIndexedDataSectionScanner: public TKinokoDataSectionScanner {
  public:
    TKinokoIndexedDataSectionScanner(TKinokoIndexedDataSection* DataSection);
    virtual ~TKinokoIndexedDataSectionScanner();
    inline int NumberOfElements(void* Buffer) const;
    inline void ReadFrom(void* Buffer, int Index, int& Address, int& Data) const;
    inline void ReadFrom(void* Buffer, int Index, int& Address, int& SubAddress, int& Data) const;
    inline void ReadFloatFrom(void* Buffer, int Index, int& Address, float& Data) const;
    inline void ReadFloatFrom(void* Buffer, int Index, int& Address, int& SubAddress, float& Data) const;
    inline void ReadStringFrom(void* Buffer, int Index, int& Address, std::string& Data) const;
    inline void ReadStringFrom(void* Buffer, int Index, int& Address, int& SubAddress, std::string& Data) const;
  protected:
    unsigned _AddressWidth, _DataWidth, _SubAddressWidth, _ElementWidth;
    unsigned _AddressBitOffset, _DataBitOffset, _SubAddressBitOffset;
    int _DataType;
};



inline TKinokoIndexedDataSectionFormatter* TKinokoIndexedDataSection::Formatter(void)
{
    if (_Formatter == 0) {
	_Formatter = new TKinokoIndexedDataSectionFormatter(this);
    }

    return _Formatter;
}

inline TKinokoIndexedDataSectionScanner* TKinokoIndexedDataSection::Scanner(void)
{
    if (_Scanner == 0) {
	_Scanner = new TKinokoIndexedDataSectionScanner(this);
    }

    return _Scanner;
}



inline int TKinokoIndexedDataSectionFormatter::DataSizeFor(int NumberOfElements) const
{
    if (_ElementWidth * NumberOfElements % 8 == 0) {
	return _ElementWidth * NumberOfElements / 8;
    }
    else {
	return _ElementWidth * NumberOfElements / 8 + 1;
    }
}

inline void TKinokoIndexedDataSectionFormatter::WriteTo(void* Buffer, int Index, int Address, int Data) const
{
    unsigned ElementBitOffset = Index * _ElementWidth;
    PutElement(
	Buffer, ElementBitOffset + _DataBitOffset, _DataWidth, Data
    );
    PutElement(
	Buffer, ElementBitOffset + _AddressBitOffset, _AddressWidth, Address
    );
}

inline void TKinokoIndexedDataSectionFormatter::WriteTo(void* Buffer, int Index, int Address, int SubAddress, int Data) const
{
    unsigned ElementBitOffset = Index * _ElementWidth;
    PutElement(
	Buffer, 
	ElementBitOffset + _DataBitOffset, _DataWidth, 
	Data
    );
    PutElement(
	Buffer, 
	ElementBitOffset + _AddressBitOffset, _AddressWidth, 
	Address
    );
    PutElement(
	Buffer, 
	ElementBitOffset + _SubAddressBitOffset, _SubAddressWidth, 
	SubAddress
    );
}

inline void TKinokoIndexedDataSectionFormatter::WriteFloatTo(void* Buffer, int Index, int Address, float Data) const
{
    unsigned ElementBitOffset = Index * _ElementWidth;
    PutElement(
	Buffer, ElementBitOffset + _DataBitOffset, _DataWidth, 
        FloatToWord(Data)
    );
    PutElement(
	Buffer, ElementBitOffset + _AddressBitOffset, _AddressWidth, Address
    );
}

inline void TKinokoIndexedDataSectionFormatter::WriteFloatTo(void* Buffer, int Index, int Address, int SubAddress, float Data) const
{
    unsigned ElementBitOffset = Index * _ElementWidth;
    PutElement(
	Buffer, 
	ElementBitOffset + _DataBitOffset, _DataWidth, 
        FloatToWord(Data)
    );
    PutElement(
	Buffer, 
	ElementBitOffset + _AddressBitOffset, _AddressWidth, 
	Address
    );
    PutElement(
	Buffer, 
	ElementBitOffset + _SubAddressBitOffset, _SubAddressWidth, 
	SubAddress
    );
}

inline void TKinokoIndexedDataSectionFormatter::WriteArrayTo(void* Buffer, int StartIndex, int Address, int DataArray[], int DataArrayLength) const
{
    for (int ArrayIndex = 0; ArrayIndex < DataArrayLength; ArrayIndex++) {
	unsigned ElementBitOffset = (StartIndex + ArrayIndex) * _ElementWidth;
	PutElement(
	    Buffer, 
	    ElementBitOffset + _DataBitOffset, _DataWidth, 
	    DataArray[ArrayIndex]
	);
	PutElement(
	    Buffer, 
	    ElementBitOffset + _AddressBitOffset, _AddressWidth, 
	    Address
	);
    }
}

inline void TKinokoIndexedDataSectionFormatter::WriteArrayTo(void* Buffer, int StartIndex, int Address, int SubAddress, int DataArray[], int DataArrayLength) const
{
    for (int ArrayIndex = 0; ArrayIndex < DataArrayLength; ArrayIndex++) {
	unsigned ElementBitOffset = (StartIndex + ArrayIndex) * _ElementWidth;
	PutElement(
	    Buffer, 
	    ElementBitOffset + _DataBitOffset, _DataWidth, 
	    DataArray[ArrayIndex]
	);
	PutElement(
	    Buffer, 
	    ElementBitOffset + _AddressBitOffset, _AddressWidth, 
	    Address
	);
	PutElement(
	    Buffer, 
	    ElementBitOffset + _SubAddressBitOffset, _SubAddressWidth, 
	    SubAddress
	);
    }
}

inline void TKinokoIndexedDataSectionFormatter::WriteFloatArrayTo(void* Buffer, int StartIndex, int Address, float DataArray[], int DataArrayLength) const
{
    for (int ArrayIndex = 0; ArrayIndex < DataArrayLength; ArrayIndex++) {
	unsigned ElementBitOffset = (StartIndex + ArrayIndex) * _ElementWidth;
	PutElement(
	    Buffer, 
	    ElementBitOffset + _DataBitOffset, _DataWidth, 
	    FloatToWord(DataArray[ArrayIndex])
	);
	PutElement(
	    Buffer, 
	    ElementBitOffset + _AddressBitOffset, _AddressWidth, 
	    Address
	);
    }
}

inline void TKinokoIndexedDataSectionFormatter::WriteFloatArrayTo(void* Buffer, int StartIndex, int Address, int SubAddress, float DataArray[], int DataArrayLength) const
{
    for (int ArrayIndex = 0; ArrayIndex < DataArrayLength; ArrayIndex++) {
	unsigned ElementBitOffset = (StartIndex + ArrayIndex) * _ElementWidth;
	PutElement(
	    Buffer, 
	    ElementBitOffset + _DataBitOffset, _DataWidth, 
	    FloatToWord(DataArray[ArrayIndex])
	);
	PutElement(
	    Buffer, 
	    ElementBitOffset + _AddressBitOffset, _AddressWidth, 
	    Address
	);
	PutElement(
	    Buffer, 
	    ElementBitOffset + _SubAddressBitOffset, _SubAddressWidth, 
	    SubAddress
	);
    }
}



inline int TKinokoIndexedDataSectionScanner::NumberOfElements(void* Buffer) const
{
    return (8 * ((U32bit*) Buffer)[Offset_DataAreaSize]) / _ElementWidth;
}

inline void TKinokoIndexedDataSectionScanner::ReadFrom(void* Buffer, int Index, int& Address, int& Data) const
{
    unsigned ElementBitOffset = Index * _ElementWidth;
    GetElement(
	Buffer, ElementBitOffset + _DataBitOffset, _DataWidth, Data
    );
    GetElement(
	Buffer, ElementBitOffset + _AddressBitOffset, _AddressWidth, Address
    );

    if (_DataType == TKinokoDataSection::ElementType_Float) {
	Data = (int) WordToFloat(Data);
    }
}

inline void TKinokoIndexedDataSectionScanner::ReadFrom(void* Buffer, int Index, int& Address, int& SubAddress, int& Data) const
{
    unsigned ElementBitOffset = Index * _ElementWidth;
    GetElement(
	Buffer, ElementBitOffset + _DataBitOffset, _DataWidth, Data
    );
    GetElement(
	Buffer, ElementBitOffset + _AddressBitOffset, _AddressWidth, Address
    );
    GetElement(
	Buffer, 
	ElementBitOffset + _SubAddressBitOffset, _SubAddressWidth, 
	SubAddress
    );

    if (_DataType == TKinokoDataSection::ElementType_Float) {
	Data = (int) WordToFloat(Data);
    }
}


inline void TKinokoIndexedDataSectionScanner::ReadFloatFrom(void* Buffer, int Index, int& Address, float& Data) const
{
    int IntData;

    unsigned ElementBitOffset = Index * _ElementWidth;
    GetElement(
	Buffer, ElementBitOffset + _DataBitOffset, _DataWidth, IntData
    );
    GetElement(
	Buffer, ElementBitOffset + _AddressBitOffset, _AddressWidth, Address
    );

    if (_DataType == TKinokoDataSection::ElementType_Float) {
	Data = WordToFloat(IntData);
    }
    else {
	Data = IntData;
    }
}

inline void TKinokoIndexedDataSectionScanner::ReadFloatFrom(void* Buffer, int Index, int& Address, int& SubAddress, float& Data) const
{
    int IntData;

    unsigned ElementBitOffset = Index * _ElementWidth;
    GetElement(
	Buffer, ElementBitOffset + _DataBitOffset, _DataWidth, IntData
    );
    GetElement(
	Buffer, ElementBitOffset + _AddressBitOffset, _AddressWidth, Address
    );
    GetElement(
	Buffer, 
	ElementBitOffset + _SubAddressBitOffset, _SubAddressWidth, 
	SubAddress
    );

    if (_DataType == TKinokoDataSection::ElementType_Float) {
	Data = WordToFloat(IntData);
    }
    else {
	Data = IntData;
    }
}

inline void TKinokoIndexedDataSectionScanner::ReadStringFrom(void* Buffer, int Index, int& Address, std::string& Data) const
{
    int IntData;

    unsigned ElementBitOffset = Index * _ElementWidth;
    GetElement(
	Buffer, ElementBitOffset + _DataBitOffset, _DataWidth, IntData
    );
    GetElement(
	Buffer, ElementBitOffset + _AddressBitOffset, _AddressWidth, Address
    );

    std::ostringstream os;
    if (_DataType == TKinokoDataSection::ElementType_Float) {
	os << WordToFloat(IntData);
    }
    else {
	os << IntData;
    }

    Data = os.str();
}

inline void TKinokoIndexedDataSectionScanner::ReadStringFrom(void* Buffer, int Index, int& Address, int& SubAddress, std::string& Data) const
{
    int IntData;

    unsigned ElementBitOffset = Index * _ElementWidth;
    GetElement(
	Buffer, ElementBitOffset + _DataBitOffset, _DataWidth, IntData
    );
    GetElement(
	Buffer, ElementBitOffset + _AddressBitOffset, _AddressWidth, Address
    );
    GetElement(
	Buffer, 
	ElementBitOffset + _SubAddressBitOffset, _SubAddressWidth, 
	SubAddress
    );

    std::ostringstream os;
    if (_DataType == TKinokoDataSection::ElementType_Float) {
	os << WordToFloat(IntData);
    }
    else {
	os << IntData;
    }

    Data = os.str();
}


#endif
