/* KinokoDevicePool.cc */
/* Created by Enomoto Sanshiro on 16 April 1999. */
/* Last updated by Enomoto Sanshiro on 30 November 2000. */


#include <string>
#include <map>
#include "RoomModule.hh"
#include "RoomDeviceBase.hh"
#include "KinokoDefs.hh"
#include "KinokoDataRecord.hh"
#include "KinokoDevicePool.hh"

using namespace std;


TKinokoDevicePool::TKinokoDevicePool(void)
{
    _NextDeviceId = 1;
}

TKinokoDevicePool::~TKinokoDevicePool() 
{
}

void  TKinokoDevicePool::Destroy(void)
{
    map<int, TKinokoDataRecord*>::iterator DataRecordEntry;
    for (
        DataRecordEntry = _DataRecordTable.begin();
        DataRecordEntry != _DataRecordTable.end();
        DataRecordEntry++
    ){
	delete (*DataRecordEntry).second;
    }
    _DataRecordTable.erase(_DataRecordTable.begin(), _DataRecordTable.end());

    map<int, int*>::iterator RegisterEntry;
    for (
        RegisterEntry = _RegisterTable.begin();
        RegisterEntry != _RegisterTable.end();
        RegisterEntry++
    ){
	delete (*RegisterEntry).second;
    }
    _RegisterTable.erase(_RegisterTable.begin(), _RegisterTable.end());

    map<int, TRoomModule*>::iterator ModuleEntry;
    for (
        ModuleEntry = _ModuleTable.begin();
        ModuleEntry != _ModuleTable.end();
        ModuleEntry++
    ){
	(*ModuleEntry).second->Destroy();
	delete (*ModuleEntry).second;
    }
    _ModuleTable.erase(_ModuleTable.begin(), _ModuleTable.end());

    map<int, TRoomController*>::iterator ControllerEntry;
    for (
        ControllerEntry = _ControllerTable.begin();
        ControllerEntry != _ControllerTable.end();
        ControllerEntry++
    ){
	delete (*ControllerEntry).second;
    }
    _ControllerTable.erase(_ControllerTable.begin(), _ControllerTable.end());

    map<int, TRoomCrate*>::iterator CrateEntry;
    for (
        CrateEntry = _CrateTable.begin();
        CrateEntry != _CrateTable.end();
        CrateEntry++
    ){
	delete (*CrateEntry).second;
    }
    _CrateTable.erase(_CrateTable.begin(), _CrateTable.end());
}

int TKinokoDevicePool::RegisterCrate(const string& Name, TRoomCrate* Crate)
{
    int DeviceId = RegisterDeviceName(Name);
    _CrateTable[DeviceId] = Crate;

    return DeviceId;
}

int TKinokoDevicePool::RegisterController(const string& Name, TRoomController* Controller)
{
    int DeviceId = RegisterDeviceName(Name);
    _ControllerTable[DeviceId] = Controller;

    return DeviceId;
}

int TKinokoDevicePool::RegisterModule(const string& Name, TRoomModule* Module)
{
    int DeviceId = RegisterDeviceName(Name);
    _ModuleTable[DeviceId] = Module;

    return DeviceId;
}

int TKinokoDevicePool::RegisterRegister(const string& Name, int* Register)
{
    int DeviceId = RegisterDeviceName(Name);
    _RegisterTable[DeviceId] = Register;

    return DeviceId;
}

int TKinokoDevicePool::RegisterDataRecord(const string& Name, TKinokoDataRecord* DataRecord)
{
    int DeviceId = RegisterDeviceName(Name);
    _DataRecordTable[DeviceId] = DataRecord;

    return DeviceId;
}

TRoomModule* TKinokoDevicePool::LookupModule(const string& ModuleName)
{
    int DeviceId = DeviceIdOf(ModuleName);

    if ((DeviceId > 0) && (_ModuleTable.count(DeviceId) > 0)) {
        return _ModuleTable[DeviceId];
    }
    else {
        return 0;
    }
}

int* TKinokoDevicePool::LookupRegister(const string& RegisterName)
{
    int DeviceId = DeviceIdOf(RegisterName);

    if ((DeviceId > 0) && (_RegisterTable.count(DeviceId) > 0)) {
        return _RegisterTable[DeviceId];
    }
    else {
        return 0;
    }
}

int TKinokoDevicePool::DeviceIdOf(const string &DeviceName)
{
    if (_DeviceNameToIdTable.count(DeviceName) == 0) {
	return 0;
    }
    else {
	 return _DeviceNameToIdTable[DeviceName];
    }
}

std::string TKinokoDevicePool::DeviceNameOf(int DeviceId)
{
    if (_DeviceIdToNameTable.count(DeviceId) == 0) {
	return "";
    }
    else {
	 return _DeviceIdToNameTable[DeviceId];
    }
}

int TKinokoDevicePool::RegisterDeviceName(const string &DeviceName)
{
    if (_DeviceNameToIdTable.count(DeviceName) == 0) {
	int DeviceId = _NextDeviceId;
	_NextDeviceId++;

	_DeviceNameToIdTable[DeviceName] = DeviceId;
	_DeviceIdToNameTable[DeviceId] = DeviceName;
    }

    return _DeviceNameToIdTable[DeviceName];
}
    
void TKinokoDevicePool::ProbeDevices(void) throw(TKinokoException) 
{
    try {
	map<int, TRoomController*>::iterator ControllerEntry;
	for (
	    ControllerEntry = _ControllerTable.begin();
	    ControllerEntry != _ControllerTable.end();
	    ControllerEntry++
	){
	    if (! (*ControllerEntry).second->Probe()) {
		string DeviceName = DeviceNameOf((*ControllerEntry).first);
		throw TKinokoException(
		    "device probing fault: "
		    "no response from " + DeviceName
		);
	    }
	}
	
	map<int, TRoomModule*>::iterator ModuleEntry;
	for (
	    ModuleEntry = _ModuleTable.begin();
	    ModuleEntry != _ModuleTable.end();
	    ModuleEntry++
	){
	    if (! (*ModuleEntry).second->Probe()) {
		string DeviceName = DeviceNameOf((*ModuleEntry).first);
		throw TKinokoException(
		    "device probing fault: "
		    "no response from " + DeviceName
		);
	    }
	}
    }
    catch (THardwareException &e) {
	throw TKinokoException(
	    "TKinokoDevicePool::Initialize()",
	    "hardware exception: " + e.Message()
	);
    }
}

void TKinokoDevicePool::Initialize(void) throw(TKinokoException) 
{
    try {
	map<int, TRoomController*>::iterator ControllerEntry;
	for (
	    ControllerEntry = _ControllerTable.begin();
	    ControllerEntry != _ControllerTable.end();
	    ControllerEntry++
	){
	    (*ControllerEntry).second->Initialize();
	}
	
	map<int, TRoomModule*>::iterator ModuleEntry;
	for (
	    ModuleEntry = _ModuleTable.begin();
	    ModuleEntry != _ModuleTable.end();
	    ModuleEntry++
	){
	    (*ModuleEntry).second->Initialize();
	}
    }
    catch (THardwareException &e) {
	throw TKinokoException(
	    "TKinokoDevicePool::Initialize()",
	    "hardware exception: " + e.Message()
	);
    }
}

void TKinokoDevicePool::Finalize(void) throw(TKinokoException) 
{
    try {
	map<int, TRoomModule*>::iterator ModuleEntry;
	for (
	    ModuleEntry = _ModuleTable.begin();
	    ModuleEntry != _ModuleTable.end();
	    ModuleEntry++
	){
	    (*ModuleEntry).second->Finalize();
	}
	
	map<int, TRoomController*>::iterator ControllerEntry;
	for (
	    ControllerEntry = _ControllerTable.begin();
	    ControllerEntry != _ControllerTable.end();
	    ControllerEntry++
	){
	    (*ControllerEntry).second->Finalize();
	}
    }
    catch (THardwareException &e) {
	throw TKinokoException(
	    "TKinokoDevicePool::Finalize()",
	    "hardware exception: " + e.Message()
	);
    }
}

void TKinokoDevicePool::Clear(void) throw(TKinokoException) 
{
    map<int, TRoomModule*>::iterator ModuleEntry;
    for (
        ModuleEntry = _ModuleTable.begin();
        ModuleEntry != _ModuleTable.end();
        ModuleEntry++
    ){
	(*ModuleEntry).second->Clear();
    }

    map<int, TRoomController*>::iterator ControllerEntry;
    for (
        ControllerEntry = _ControllerTable.begin();
        ControllerEntry != _ControllerTable.end();
        ControllerEntry++
    ){
	(*ControllerEntry).second->Clear();
    }
}
