/* KinokoKdfReader.cc */
/* Created by Enomoto Sanshiro on 23 February 2002. */
/* Last updated by Enomoto Sanshiro on 6 October 2002. */


#include <string>
#include "KinokoFileStream.hh"
#include "KinokoKdfStorage.hh"
#include "KinokoDataDistributor.hh"
#include "KinokoDataProcessor.hh"
#include "KinokoKdfReader.hh"

using namespace std;


TKinokoKdfReader::TKinokoKdfReader(const string& DataFileName, bool IsRawKdf)
{
    _IsRawKdf = false;
    _DataFileName = DataFileName;

    _DataDistributor = 0;
    _Storage = 0;
    _StorageHeader = 0;
    _InputFileStream = 0;

    _IsFirstRunBeginProcessed = false;
}

TKinokoKdfReader::~TKinokoKdfReader()
{
    delete _DataDistributor;
    delete _Storage;
}

void TKinokoKdfReader::RegisterAnalyzer(const string& DataSourceName, TKinokoDataAnalyzer* DataAnalyzer)
{
    if (_AnalyzerTable.count(DataSourceName) == 0) {
	_AnalyzerTable[DataSourceName] = DataAnalyzer;
    }
    else {
	if (_AnalyzerListTable.count(DataSourceName) == 0) {
	    TKinokoDataAnalyzerList* List = new TKinokoDataAnalyzerList(); 
	    _AnalyzerListTable[DataSourceName] = List;

	    List->AddAnalyzer(_AnalyzerTable[DataSourceName]);
	    _AnalyzerTable[DataSourceName] = List;
	}

	_AnalyzerListTable[DataSourceName]->AddAnalyzer(DataAnalyzer);
    }
}

void TKinokoKdfReader::RegisterCommonAnalyzer(TKinokoDataAnalyzer* DataAnalyzer)
{
    _CommonAnalyzerList.push_back(DataAnalyzer);
}

void TKinokoKdfReader::Start(void) throw(TKinokoException)
{
    if (_Storage == 0) {
	Construct();
    }

    while (ProcessInputStream() > 0) {
	;
    }

    Destruct();
}

bool TKinokoKdfReader::ProcessNext(void) throw(TKinokoException)
{
    if (_Storage == 0) {
	Construct();
    }

    return (ProcessInputStream() > 0);
}

void TKinokoKdfReader::ProcessUntilRunBegin(void) throw(TKinokoException)
{
    if (_Storage == 0) {
	Construct();
    }

    while ((ProcessInputStream() > 0) && ! _IsFirstRunBeginProcessed) {
	;
    }
}

void TKinokoKdfReader::Construct(void) throw(TKinokoException)
{
    if (_Storage != 0) {
	return;
    }

    _Storage = new TKinokoKdfStorage(_DataFileName, _IsRawKdf);

    _StorageHeader = new TKinokoStorageHeader();
    _Storage->ReadHeader(*_StorageHeader);

    _InputFileStream = _Storage->GetInputStream();
    ConstructInlet(_InputFileStream);

    if (_AnalyzerTable.empty() && _CommonAnalyzerList.empty()) {
	return;
    }

    _DataDistributor = new TKinokoDataDistributor();

    map<string, TKinokoDataAnalyzer*>::iterator AnalyzerTableIterator;
    for (
	AnalyzerTableIterator = _AnalyzerTable.begin();
	AnalyzerTableIterator != _AnalyzerTable.end();
	AnalyzerTableIterator++
    ){
	string DataSourceName = (*AnalyzerTableIterator).first;
	TKinokoDataAnalyzer* Analyzer = (*AnalyzerTableIterator).second;

	_DataDistributor->RegisterAnalyzer(DataSourceName, Analyzer);
    }

    for (unsigned i = 0; i < _CommonAnalyzerList.size(); i++) {
	_DataDistributor->RegisterCommonAnalyzer(_CommonAnalyzerList[i]);
    }
}

void TKinokoKdfReader::Destruct(void) throw(TKinokoException)
{
    map<string, TKinokoDataAnalyzerList*>::iterator AnalyzerListTableIterator;
    for (
	AnalyzerListTableIterator = _AnalyzerListTable.begin();
	AnalyzerListTableIterator != _AnalyzerListTable.end();
	AnalyzerListTableIterator++
    ){
	delete (*AnalyzerListTableIterator).second;
    }

    delete _DataDistributor;
    delete _Storage;

    _DataDistributor = 0;
    _Storage = 0;
}

void TKinokoKdfReader::OnRunBegin(void) throw(TKinokoException)
{
    _IsFirstRunBeginProcessed = false;
}

void TKinokoKdfReader::OnReceiveDataDescriptorPacket(void* DataPacket, long PacketSize) throw(TKinokoException)
{
    if (_DataDistributor) {
	_DataDistributor->ProcessDataDescriptor(_InputDataDescriptor);
    }
}

void TKinokoKdfReader::OnReceiveDataPacket(void* DataPacket, long PacketSize) throw(TKinokoException)
{
    if (_DataDistributor) {
	_DataDistributor->ProcessDataPacket(DataPacket);
    }
}

void TKinokoKdfReader::OnReceiveTrailerPacket(void* DataPacket, long PacketSize) throw(TKinokoException) 
{
    if (_DataDistributor) {
	_DataDistributor->ProcessTrailerPacket(DataPacket);
    }
}

void TKinokoKdfReader::OnReceiveCommandPacket(void* DataPacket, long PacketSize) throw(TKinokoException) 
{
    if (_DataDistributor) {
	_DataDistributor->ProcessCommandPacket(DataPacket);
    }
}

TKinokoKdfStorage* TKinokoKdfReader::Storage(void) throw(TKinokoException)
{
    if (_Storage == 0) {
	Construct();
    }

    return _Storage;
}

TKinokoStorageHeader* TKinokoKdfReader::StorageHeader(void) throw(TKinokoException)
{
    if (_Storage == 0) {
	Construct();
    }

    return _StorageHeader;
}

TKinokoDataDescriptor* TKinokoKdfReader::DataDescriptor(void) throw(TKinokoException)
{
    if (_Storage == 0) {
	Construct();
    }

    return _InputDataDescriptor;
}

TKinokoInputFileStream* TKinokoKdfReader::InputStream(void) throw(TKinokoException)
{
    if (_Storage == 0) {
	Construct();
    }

    return _InputFileStream;
}
