/* KinokoHistogramView.cc */
/* Created by Enomoto Sanshiro on 4 November 2000. */
/* Last updated by Enomoto Sanshiro on 9 December 2009. */


#include <iostream>
#include <iomanip>
#include <cmath>
#include <complex>
#include "KameHistogram.hh"
#include "KameRepository.hh"
#include "KumaSpecialFunction.hh"
#include "KinokoHistogramView.hh"

using namespace std;
using namespace kame;
using namespace kuma;


TKinokoHistogramView::TKinokoHistogramView(TKameHistogram* Histogram)
{
    _Histogram = Histogram;
    _IsYScaleLog = false;

    _YMin = 0;
    _YMax = 1.0;

    _ReferenceHistogram = 0;
    _ConfidenceBand = 0;

    _DisplayStatisticsFlags = (
	Stat_Entries | Stat_Overflow | Stat_Underflow | Stat_Mean
    );
}

TKinokoHistogramView::~TKinokoHistogramView()
{
    delete _Histogram;

    delete _ReferenceHistogram;
    delete _ConfidenceBand;
}

void TKinokoHistogramView::SetDisplayStatistics(int StatisticsFlags)
{
    _DisplayStatisticsFlags = StatisticsFlags;
}

void TKinokoHistogramView::SetYScaleLog(void)
{
    _IsYScaleLog = true;
    _YMin = 0.5;
}

void TKinokoHistogramView::SetYScaleLinear(void)
{
    _IsYScaleLog = false;
    _YMin = 0;
}

void TKinokoHistogramView::SetAxisTitle(const std::string& XTitle, const std::string& YTitle)
{
    _XTitle = XTitle;
    _YTitle = YTitle;
}

void TKinokoHistogramView::SetReferenceHistogram(TKameHistogram* ReferenceHistogram, double ToleranceSigmas)
{
    _ReferenceHistogram = ReferenceHistogram;

    _ConfidenceBand = new TKinokoConfidenceBand(ToleranceSigmas);
}

void TKinokoHistogramView::SaveThis(TKameRepository* Repository)throw(TKinokoException)
{
    try {
	Repository->SaveHistogram(*_Histogram, _Name);
    }
    catch (TKameException &e) {
	throw TKinokoException(
	    "TKinokoHistogramView::SaveThis()", e.Message()
	);
    }
}

void TKinokoHistogramView::ClearThis(void)
{
    _Histogram->Clear();
}



TKinokoKoapHistogramView::TKinokoKoapHistogramView(TKameHistogram* Histogram, ostream* OutputStream)
: TKinokoHistogramView(Histogram)
{
    _OutputStream = OutputStream;
}

TKinokoKoapHistogramView::~TKinokoKoapHistogramView()
{
}

void TKinokoKoapHistogramView::DeployThis(void) 
{
    *_OutputStream << ".selectPage " << _PageNumber << ";" << endl;

    *_OutputStream << ".create plot " << _Name << " ";
    *_OutputStream << 100 * Left() << " " << 100 * Top() << " ";
    *_OutputStream << 100 * Width() << " " << 100 * Height() << ";" << endl;

    *_OutputStream << _Name << " set title " << _Histogram->Title() << ";" << endl;
    if (! _XTitle.empty()) {
	*_OutputStream << _Name << " set xtitle " << _XTitle << ";" << endl;
    }
    if (! _YTitle.empty()) {
	*_OutputStream << _Name << " set ytitle " << _YTitle << ";" << endl;
    }
    
    _Min = _Histogram->Scale().Min();
    _Max = _Histogram->Scale().Max();
    _NumberOfBins = _Histogram->Scale().NumberOfBins();
    _BinStep = (_Max - _Min) / _NumberOfBins;

    if (_IsYScaleLog) {
	*_OutputStream << _Name << " set yscale log;" << endl;
    }

    *_OutputStream << _Name << " set commentboxcolor background;" << endl;

    *_OutputStream << _Name << " frame ";
    *_OutputStream << _Min << " " << _Max << " ";
    *_OutputStream << _YMin << " " << _YMax << ";" << endl;
}

void TKinokoKoapHistogramView::DrawThis(void)
{
    int Order = (int) log(_Histogram->PeakCounts() + 1);
    if (! _IsYScaleLog) {
	_YMax = 1.1 * exp((float) (Order + 1));
    }
    else {
	_YMax = 2.0 * exp((float) (Order + 1));
    }

    *_OutputStream << _Name << " clear;" << endl;
    *_OutputStream << _Name << " frame ";
    *_OutputStream << _Min << " " << _Max << " ";
    *_OutputStream << _YMin << " " << _YMax << ";" << endl;

    double Scaling = -1;
    if (_ReferenceHistogram) do {
	do {
	    long AccumulationTime = _Histogram->AccumulationTime();
	    long RefAccumulationTime = _ReferenceHistogram->AccumulationTime();
	    if ((AccumulationTime > 0) && (RefAccumulationTime > 0)) {
		Scaling = (double) AccumulationTime / RefAccumulationTime;
		break;
	    }

	    double Area = _Histogram->NumberOfEntries();
	    double RefArea = _ReferenceHistogram->NumberOfEntries();
	    if ((Area > 0) && (RefArea > 0)) {
		Scaling = Area / RefArea;
		break;
	    }
	} while (0);
	if (Scaling < 0) {
	    break;
	}
	
	double Min = _ReferenceHistogram->Scale().Min();
	double Max = _ReferenceHistogram->Scale().Max();
	int NumberOfBins = _ReferenceHistogram->Scale().NumberOfBins();
	double BinStep = (Max - Min) / NumberOfBins;

	*_OutputStream << _Name << " savecontext;" << endl;
	*_OutputStream << _Name << " set color lightgray;" << endl;
	
	*_OutputStream << _Name << " bandhist " << Min << " " << BinStep;
	for (int BinIndex = 0; BinIndex < NumberOfBins; BinIndex++) {
	    double Lower, Upper;
	    double Count = _ReferenceHistogram->CountsOf(BinIndex);
	    _ConfidenceBand->GetBand(Scaling * Count, Lower, Upper);
	    *_OutputStream << " " << Lower << " " << Upper;
	}
	*_OutputStream << ";" << endl;
	*_OutputStream << _Name << " restorecontext;" << endl;

    } while (0);
    
    double Chi2 = 0;
    int Chi2Ndf = 0;
    double Chi2Probability = -1;
    if ((Scaling > 0) && _Histogram->HasData()) do {
	int NumberOfBins = _Histogram->Scale().NumberOfBins();
	if (_ReferenceHistogram->Scale().NumberOfBins() != NumberOfBins) {
	    break;
	}
	for (int Index = 0; Index < NumberOfBins; Index++) {
	    double Ni = _Histogram->CountsOf(Index);
	    double Ri = _ReferenceHistogram->CountsOf(Index);
	    if ((Ni < 10) || (Ri < 10)) {
		continue;
	    }
	    Chi2 += sqr(Ni - Ri * Scaling) / (Ni + Ri * sqr(Scaling));
	    Chi2Ndf += 1;
	}
	if (Chi2Ndf > 0) {
	    Chi2Probability = 1 - IncompleteGamma(Chi2Ndf/2.0, Chi2/2.0);
	}
    } while (0);

    if (! _Histogram->HasData()) {
        *_OutputStream << _Name << " drawtext ";
        *_OutputStream << (7 * _Min + _Max) / 8 << " ";
        *_OutputStream << (_YMin + _YMax) / 2 << " ";
        *_OutputStream << "no data;" << endl;
	return;
    }

    *_OutputStream << _Name << " hist " << _Min << " " << _BinStep;
    for (int BinIndex = 0; BinIndex < _NumberOfBins; BinIndex++) {
	*_OutputStream << " " << _Histogram->CountsOf(BinIndex);
    }
    *_OutputStream << ";" << endl;

    if (_DisplayStatisticsFlags) {
	*_OutputStream << _Name << " comment -100 0 ";
    }
    if (_DisplayStatisticsFlags & Stat_Entries) {
	*_OutputStream << "entries: " << _Histogram->NumberOfEntries() << endl;
    }
    if (_DisplayStatisticsFlags & Stat_Underflow) {
	*_OutputStream << "underflow: " << _Histogram->UnderflowCounts() << endl;
    }
    if (_DisplayStatisticsFlags & Stat_Overflow) {
	*_OutputStream << "overflow: " << _Histogram->OverflowCounts() << endl;
    }
    if (_DisplayStatisticsFlags & Stat_Mean) {
	*_OutputStream << "mean: " << _Histogram->Mean() << endl;
    }
    if (_DisplayStatisticsFlags & Stat_Deviation) {
	*_OutputStream << "rms: " << _Histogram->Deviation() << endl;
    }
    if ((_ReferenceHistogram != 0) && (_DisplayStatisticsFlags != 0)) {
	*_OutputStream << "math:#chi^2/ndf: " << ((int) (10*Chi2) / 10);
	*_OutputStream << "/" << Chi2Ndf;
	if (Chi2Ndf > 10) {
	    *_OutputStream << " (" << (int) (100*Chi2Probability) << "%)" << endl;
	}
	else {
	    *_OutputStream << " (--%)" << endl;
	}
    }
    if (_DisplayStatisticsFlags) {
	*_OutputStream << ";" << endl;
    }

    *_OutputStream << _Name << " drawframe;" << endl;
}

void TKinokoKoapHistogramView::ClearThis(void)
{
    TKinokoHistogramView::ClearThis();
    
    *_OutputStream << _Name << " clear;" << endl;
    *_OutputStream << _Name << " frame ";
    *_OutputStream << _Min << " " << _Max << " ";
    *_OutputStream << _YMin << " " << _YMax << ";" << endl;
}



static double LogGamma(double x) 
{
    static const double Pi = 3.141592653589793238;
    static const double LogPi = std::log(Pi);
    static const double SqrtOf2Pi = std::sqrt(2*Pi);

    if (std::real(std::complex<double>(x)) < 0) {
        return (
	    LogPi + std::log(1.0-x) - 
	    LogGamma(2.0-x) - std::log(std::sin(Pi*(1.0-x)))
	);
    }

    static const double CoefficientList[] = {
        1.000000000190015,
        76.18009172947146, -86.50532032941677, 24.01409824083091, 
        -1.231739572450155, 0.1208650973866179e-2, -0.5395239384953e-5
    };

    double Series = CoefficientList[0];
    for (unsigned i = 1; i < sizeof(CoefficientList)/sizeof(double); i++) {
        Series += CoefficientList[i] / (x + (double) i);
    }

    return (x+0.5) * log(x+5.5) - (x+5.5) + std::log(SqrtOf2Pi * Series/x);
}

static double LogFactorial(int n)
{
    static const int TableSize = 100;
    static double LogFactorialTable[TableSize];
    static bool IsTableInitialized = false;

    if (! IsTableInitialized) {
	int i = 0;
	double Factorial = 1;
	LogFactorialTable[i] = 0;
	for (i = 1; i < min(20, TableSize); i++) {
	    Factorial *= i;
	    LogFactorialTable[i] = log(Factorial);
	}
	for ( ; i < TableSize; i++) {
	    LogFactorialTable[i] = LogGamma(i + 1.0);
	}
	IsTableInitialized = true;
    }

    if (n < TableSize) {
	return LogFactorialTable[n];
    }
    else {
	return LogGamma(n + 1.0);
    }
}

static double CumulativeGaussian(double X)
{
    // Chebyshev Approximation (relative-error < 1.2e-7 for all x) //

    static const double SqrtOf2 = std::sqrt(2.0);
    double z = X/SqrtOf2;
    double t = 1.0 / (1.0 + fabs(z) / 2);
    double v = 1.0 - t * exp(-z*z + (
        -1.26551223 + t * (
	+1.00002368 + t * (
	+0.37409196 + t * (
	+0.09678418 + t * (
	-0.18628806 + t * (
	+0.27886807 + t * (
	-1.13520398 + t * (
	+1.48851587 + t * (
	-0.82215223 + t * (
	+0.17087277
    )))))))))));

    // returns -0.5 for X=-inf, 0 for X=0, 0.5 for X=+inf //
    return ((z >= 0) ? +1 : -1) / 2.0 * v;
}

static double Poisson(double Mean, int N)
{
    return exp(N * log(Mean) - LogFactorial(N) - Mean);
}



TKinokoConfidenceBand::TKinokoConfidenceBand(double NSigmas)
{
    _NSigmas = fabs(NSigmas);
    _HalfCoverage = CumulativeGaussian(NSigmas);

    double Mean = 0;
    int Lower = 0, Upper = 0;
    _PoissonIntervalTable.push_back(
	make_pair(Mean, make_pair(Lower, Upper))
    );

    int PrevLower = Lower, PrevUpper = Upper;
    for (Mean = 0.01; ; Mean += 0.01) {
	double Rms = sqrt(Mean);
	double GaussWidth = Rms * _NSigmas;
	if (Mean - Rms * (_NSigmas+4) > 0) {
	    Lower = (int) floor(Mean - GaussWidth);
	    Upper = (int) ceil(Mean + GaussWidth);
	    _PoissonIntervalTable.push_back(
		make_pair(Mean, make_pair(Lower, Upper))
	    );
	    break;
	}
	
	Upper = Lower = (int) floor(Mean);
	double Remainder = 2 * _HalfCoverage - Poisson(Mean, Lower);
	while (1) {
	    if (Remainder <= 0) {
		break;
	    }
	    if (Lower > 0) {
		Lower--;
		Remainder -= Poisson(Mean, Lower);
	    }

	    if (Remainder <= 0) {
		break;
	    }
	    Upper++;
	    Remainder -= Poisson(Mean, Upper);
	}

	if ((Lower > PrevLower) || (Upper > PrevUpper)) {
	    PrevLower = max(Lower, PrevLower);
	    PrevUpper = max(Upper, PrevUpper);
	    _PoissonIntervalTable.push_back(
		make_pair(Mean, make_pair(PrevLower, PrevUpper))
	    );
	}
    }
}

TKinokoConfidenceBand::~TKinokoConfidenceBand()
{
}

void TKinokoConfidenceBand::GetBand(double Mean, double& Lower, double&Upper)
{
    if (Mean >= (_PoissonIntervalTable.end() - 1)->first) {
	double Rms = sqrt(Mean);
	Lower = floor(Mean - _NSigmas*Rms);
	Upper = ceil(Mean + _NSigmas*Rms);
	return;
    }
	
    vector<pair<double, pair<double, double> > >::iterator i;
    for (
	i = _PoissonIntervalTable.begin(); 
	i != _PoissonIntervalTable.end(); 
	i++
    ){
	if (i->first >= Mean) {
	    break;
	}
    }
    Lower = i->second.first;
    Upper = i->second.second;
}
