/* KinokoHistoryView.cc */
/* Created by Enomoto Sanshiro on 5 December 2000. */
/* Last updated by Enomoto Sanshiro on 29 September 2002. */


#include <cmath>
#include "MushMisc.hh"
#include "KameHistory.hh"
#include "KameRepository.hh"
#include "KinokoPlatform.hh"
#include "KinokoHistoryView.hh"

using namespace std;
using namespace kame;


TKinokoHistoryView::TKinokoHistoryView(TKameHistory* History, double MinValue, double MaxValue, double WindowWidth)
{
    _History = History;
    _DisplayValueFieldId = FieldId_Deviation;

    _DisplayStatisticsFlags = Stat_Counts | Stat_Mean | Stat_Deviation;
    _IsDisplayStatisticsSpecified = false;

    _MinValue = MinValue;
    _MaxValue = MaxValue;

    _MaxNumberOfSamples = _History->MaxNumberOfSamples();
    _WindowWidth = WindowWidth;

    _LowerBound = 0;
    _UpperBound = 0;
    _EventEmitter = 0;
    
    _IsYScaleLog = false;
    _IsAutoHoldEnabled = true;

    _IsLastValueInRange = false;
}

TKinokoHistoryView::~TKinokoHistoryView()
{
    delete _History;
}

void TKinokoHistoryView::Draw(void)
{
    if (_IsAutoHoldEnabled) {
	Hold();
    }
    
    TKinokoView::Draw();
}

void TKinokoHistoryView::SetDisplayValue(int FieldId)
{
    _DisplayValueFieldId = FieldId;

    if (! _IsDisplayStatisticsSpecified){
	switch (_DisplayValueFieldId) {
	  case FieldId_Counts:
	    _DisplayStatisticsFlags = Stat_Counts;
	    break;
	  case FieldId_Sum:
	    _DisplayStatisticsFlags = Stat_Counts | Stat_Sum;
	    break;
	  case FieldId_Mean:
	    _DisplayStatisticsFlags = Stat_Counts | Stat_Mean;
	    break;
	  case FieldId_Deviation:
	    _DisplayStatisticsFlags = Stat_Counts | Stat_Mean | Stat_Deviation;
	    break;
	  default:
	    break;
	}
    }
}

void TKinokoHistoryView::SetDisplayStatistics(int StatisticsFlags)
{
    _DisplayStatisticsFlags = StatisticsFlags;
    _IsDisplayStatisticsSpecified = true;
}

void TKinokoHistoryView::SetYScaleLog(void)
{
    _IsYScaleLog = true;

    if (_MinValue <= 0) {
	_MinValue = 0.5;
    }
    if (_MaxValue < _MinValue) {
	_MaxValue = 10 * _MinValue;
    }
}

void TKinokoHistoryView::SetYScaleLinear(void)
{
    _IsYScaleLog = false;
}

void TKinokoHistoryView::SetAxisTitle(const std::string& XTitle, const std::string& YTitle)
{
    _XTitle = XTitle;
    _YTitle = YTitle;
}

void TKinokoHistoryView::DisableAutoHold(void)
{
    _IsAutoHoldEnabled = false;
}

void TKinokoHistoryView::Hold(void)
{
#if 0
    // use data timestamp //
    _History->HoldSample();
#else
    // use the time of hold() //
    _History->HoldSample(TMushDateTime::SecSinceEpoch());
#endif
}

void TKinokoHistoryView::SetOperationRange(double Lower, double Upper)
{
    _LowerBound = Lower;
    _UpperBound = Upper;
}

void TKinokoHistoryView::EnableAlarm(TKinokoEventEmitter* EventEmitter, const string& Message)
{
    _EventEmitter = EventEmitter;
    _AlarmMessage = Message;
}

void TKinokoHistoryView::DisableAlarm(void)
{
    _EventEmitter = 0;
}

void TKinokoHistoryView::CheckValueRange(double Value)
{
    if ((_EventEmitter == 0) || (_LowerBound >= _UpperBound)) {
	return;
    }

    if ((Value < _LowerBound) || (Value > _UpperBound)) {
        if (_IsLastValueInRange) {
	    vector<string> ArgumentList;
	    ArgumentList.push_back(_AlarmMessage);
	    _EventEmitter->EmitEvent("alarm", ArgumentList);
	}
	_IsLastValueInRange = false;
    }
    else {
	_IsLastValueInRange = true;
    }
}

void TKinokoHistoryView::SaveThis(TKameRepository* Repository)throw(TKinokoException)
{
    try {
	Repository->SaveHistory(*_History, _Name);
    }
    catch (TKameException &e) {
	throw TKinokoException(
	    "TKinokoHistoryView::SaveThis()", e.Message()
	);
    }
}

void TKinokoHistoryView::ClearThis(void)
{
    _History->Clear();
    _IsLastValueInRange = false;
}



TKinokoKoapHistoryView::TKinokoKoapHistoryView(TKameHistory* History, double MinValue, double MaxValue, double WindowWidth, ostream* OutputStream)
: TKinokoHistoryView(History, MinValue, MaxValue, WindowWidth)
{
    _OutputStream = OutputStream;
}

TKinokoKoapHistoryView::~TKinokoKoapHistoryView()
{
}

void TKinokoKoapHistoryView::DeployThis(void) 
{
    *_OutputStream << ".selectPage " << _PageNumber << ";" << endl;

    *_OutputStream << ".create plot " << _Name << " ";
    *_OutputStream << 100 * Left() << " " << 100 * Top() << " ";
    *_OutputStream << 100 * Width() << " " << 100 * Height() << ";" << endl;

    *_OutputStream << _Name << " set title " << _History->Title() << ";" << endl;
    if (! _XTitle.empty()) {
	*_OutputStream << _Name << " set xtitle " << _XTitle << ";" << endl;
    }
    if (! _YTitle.empty()) {
	*_OutputStream << _Name << " set ytitle " << _YTitle << ";" << endl;
    }

    if (_IsYScaleLog) {
	*_OutputStream << _Name << " set yscale log;" << endl;
    }

    *_OutputStream << _Name << " set commentboxcolor background;" << endl;

    *_OutputStream << _Name << " frame ";
    *_OutputStream << 0 << " " << _WindowWidth << " ";
    *_OutputStream << _MinValue << " " << _MaxValue << ";" << endl;
}

void TKinokoKoapHistoryView::DrawThis(void)
{
    long StartTime = _History->StartTime();
    long CurrentTime = TMushDateTime::SecSinceEpoch();

    double MinTime, MaxTime;
    if ((StartTime > 0) && (CurrentTime > StartTime + _WindowWidth)) {
	MaxTime = CurrentTime - StartTime;
	MinTime = MaxTime - _WindowWidth;
    }
    else {
	MaxTime = _WindowWidth;
	MinTime = 0;
    }

    *_OutputStream << _Name << " clear;" << endl;
    *_OutputStream << _Name << " frame ";
    *_OutputStream << MinTime << " " << MaxTime << " ";
    *_OutputStream << _MinValue << " " << _MaxValue << ";" << endl;

    if (_LowerBound < _UpperBound) {
	*_OutputStream << _Name << " savecontext;" << endl;
	*_OutputStream << _Name << " set color lightgray;" << endl;
	*_OutputStream << _Name << " drawrectfill ";
	*_OutputStream << MinTime << " " << _UpperBound << " ";
	*_OutputStream << MaxTime << " " << _LowerBound << ";";
	*_OutputStream << endl;
	*_OutputStream << _Name << " restorecontext;" << endl;
	*_OutputStream << _Name << " drawframe;" << endl;
    }

    if (! _History->HasData()) {
        *_OutputStream << _Name << " drawtext ";
        *_OutputStream << (7 * MinTime + MaxTime) / 8 << " ";
        *_OutputStream << (_MinValue + _MaxValue) / 2 << " ";
        *_OutputStream << "no data;" << endl;
	return;
    }

    double LastValue;
    switch (_DisplayValueFieldId) {
      case FieldId_Counts:
        LastValue = DrawCountPlot(MinTime, MaxTime);
	break;
      case FieldId_Sum:
        LastValue = DrawSumPlot(MinTime, MaxTime);
	break;
      case FieldId_Mean:
        LastValue = DrawMeanPlot(MinTime, MaxTime, false);
	break;
      case FieldId_Deviation:
        LastValue = DrawMeanPlot(MinTime, MaxTime, true);
	break;
      default:
        LastValue = 0;
	break;
    }

    if (_DisplayStatisticsFlags) {
	*_OutputStream << _Name << " comment -100 0 ";
    }
    if (_DisplayStatisticsFlags & Stat_Counts) {
	*_OutputStream << "counts: " << _History->LastCounts() << endl;
    }
    if (_DisplayStatisticsFlags & Stat_Mean) {
	*_OutputStream << "mean: " << _History->LastMean() << endl;
    }
    if (_DisplayStatisticsFlags & Stat_Deviation) {
	*_OutputStream << "rms: " << _History->LastDeviation() << endl;
    }
    if (_DisplayStatisticsFlags) {
	*_OutputStream << ";" << endl;
    }

    CheckValueRange(LastValue);
}

double TKinokoKoapHistoryView::DrawCountPlot(double MinTime, double MaxTime)
{
    *_OutputStream << _Name << " plot";

    int NumberOfSamples = _History->NumberOfSamples();

    double PassedTime, Counts;
    for (int SampleIndex = 0; SampleIndex < NumberOfSamples; SampleIndex++) {
	PassedTime = _History->PassedTimeOf(SampleIndex);
	if ((PassedTime < MinTime) || (PassedTime > MaxTime)) {
	    continue;
	}

	Counts = _History->CountsOf(SampleIndex);
	if (Counts > _MaxValue) {
	    Counts = _MaxValue;
	}
	else if (Counts < _MinValue) {
	    Counts = _MinValue;
	}
	    
	*_OutputStream << " " << PassedTime << " " << Counts;
    }
    *_OutputStream << ";" << endl;

    return _History->LastCounts();
}

double TKinokoKoapHistoryView::DrawSumPlot(double MinTime, double MaxTime)
{
    *_OutputStream << _Name << " plot";

    int NumberOfSamples = _History->NumberOfSamples();

    double PassedTime, Sum;
    for (int SampleIndex = 0; SampleIndex < NumberOfSamples; SampleIndex++) {
	PassedTime = _History->PassedTimeOf(SampleIndex);
	if ((PassedTime < MinTime) || (PassedTime > MaxTime)) {
	    continue;
	}

	Sum = _History->SumOf(SampleIndex);
	if (Sum > _MaxValue) {
	    Sum = _MaxValue;
	}
	else if (Sum < _MinValue) {
	    Sum = _MinValue;
	}
	    
	*_OutputStream << " " << PassedTime << " " << Sum;
    }
    *_OutputStream << ";" << endl;
    
    return _History->LastSum();
}

double TKinokoKoapHistoryView::DrawMeanPlot(double MinTime, double MaxTime, bool DrawsDeviation)
{
    if (DrawsDeviation) {
	*_OutputStream << _Name << " errorplot";
    }
    else {
	*_OutputStream << _Name << " plot";
    }

    int NumberOfSamples = _History->NumberOfSamples();

    double PassedTime, Counts;
    double Mean, Deviation;
    for (int SampleIndex = 0; SampleIndex < NumberOfSamples; SampleIndex++) {
        PassedTime = _History->PassedTimeOf(SampleIndex);
	Counts = _History->CountsOf(SampleIndex);
	
        if (
	    (Counts == 0) || 
	    (PassedTime < MinTime) || (PassedTime > MaxTime)
	){
	    continue;
	}

	Mean = _History->MeanOf(SampleIndex);
	Deviation = _History->DeviationOf(SampleIndex);

	if (Mean > _MaxValue) {
	    Mean = _MaxValue;
	    Deviation = 0;
	}
	else if (Mean < _MinValue) {
	    Mean = _MinValue;
	    Deviation = 0;
	}
	    
	*_OutputStream << " " << PassedTime << " " << Mean;
	if (DrawsDeviation) {
	    *_OutputStream << " 0 " << Deviation;
	}
    }
    *_OutputStream << ";" << endl;

    return _History->LastMean();
}

void TKinokoKoapHistoryView::ClearThis(void)
{
    TKinokoHistoryView::ClearThis();
    
    *_OutputStream << _Name << " clear;" << endl;
    *_OutputStream << _Name << " frame ";
    *_OutputStream << 0 << " " << _WindowWidth << " ";
    *_OutputStream << _MinValue << " " << _MaxValue << ";" << endl;
}
