/* KinokoMapView.cc */
/* Created by Enomoto Sanshiro on 2 April 2001. */
/* Last updated by Enomoto Sanshiro on 8 April 2001. */


#include <cmath>
#include <map>
#include "KameNtuple.hh"
#include "KameRepository.hh"
#include "KinokoMapView.hh"

using namespace std;
using namespace kame;


TKinokoMapView::TKinokoMapView(TKameNtuple* Ntuple, float XMin, float XMax, float YMin, float YMax, float ZMin, float ZMax)
{
    _Ntuple = Ntuple;

    _LastAddress = -1;

    _XMin = XMin; _XMax = XMax;
    _YMin = YMin; _YMax = YMax;
    _ZMin = ZMin; _ZMax = ZMax;

    _PointRadius = 1.0;
}

TKinokoMapView::~TKinokoMapView()
{
    delete _Ntuple;
}

void TKinokoMapView::AddPoint(int Address, float PositionX, float PositionY)
{
    _PositionTable[Address] = make_pair(PositionX, PositionY);
}

void TKinokoMapView::SetPointSize(float Radius)
{
    _PointRadius = Radius;
}

bool TKinokoMapView::IsValidAddress(int Address)
{
    if (Address != _LastAddress) {
	_LastAddress = Address;
	_LastPosition = _PositionTable.find(Address);
    }

    return _LastPosition != _PositionTable.end();
}

float TKinokoMapView::PositionXOf(int Address)
{
    if (Address != _LastAddress) {
	_LastAddress = Address;
	_LastPosition = _PositionTable.find(Address);
    }

    if (_LastPosition == _PositionTable.end()) {
	return 0;
    }

    return (*_LastPosition).second.first;
}

float TKinokoMapView::PositionYOf(int Address)
{
    if (Address != _LastAddress) {
	_LastAddress = Address;
	_LastPosition = _PositionTable.find(Address);
    }

    if (_LastPosition == _PositionTable.end()) {
	return 0;
    }

    return (*_LastPosition).second.second;
}

float TKinokoMapView::PointRadius(void)
{
    return _PointRadius;
}

void TKinokoMapView::SaveThis(TKameRepository* Repository)throw(TKinokoException)
{
    try {
	Repository->SaveNtuple(*_Ntuple, _Name);
    }
    catch (TKameException &e) {
	throw TKinokoException(
	    "TKinokoMapView::SaveThis()", e.Message()
	);
    }
}

void TKinokoMapView::ClearThis(void)
{
    _Ntuple->Clear();
}


TKinokoKoapMapView::TKinokoKoapMapView(TKameNtuple* Ntuple, float XMin, float XMax, float YMin, float YMax, float ZMin, float ZMax, ostream* OutputStream)
: TKinokoMapView(Ntuple, XMin, XMax, YMin, YMax, ZMin, ZMax)
{
    _OutputStream = OutputStream;
}

TKinokoKoapMapView::~TKinokoKoapMapView()
{
}

void TKinokoKoapMapView::DeployThis(void) 
{
    *_OutputStream << ".selectPage " << _PageNumber << ";" << endl;

    (*_OutputStream) << ".create mapplot " << _Name << " ";
    (*_OutputStream) << 100 * Left() << " " << 100 * Top() << " ";
    (*_OutputStream) << 100 * Width() << " " << 100 * Height() << ";" << endl;

    (*_OutputStream) << _Name << " set title " << _Ntuple->Title() << ";" << endl;

    (*_OutputStream) << _Name << " frame ";
    (*_OutputStream) << _XMin << " " << _XMax << " ";
    (*_OutputStream) << _YMin << " " << _YMax << " ";
    (*_OutputStream) << _ZMin << " " << _ZMax << ";" << endl;
}

void TKinokoKoapMapView::DrawThis(void)
{
    (*_OutputStream) << _Name << " clear;" << endl;
    (*_OutputStream) << _Name << " frame ";
    (*_OutputStream) << _XMin << " " << _XMax << " ";
    (*_OutputStream) << _YMin << " " << _YMax << " ";
    (*_OutputStream) << _ZMin << " " << _ZMax << ";" << endl;

    if (_Ntuple->NumberOfRows() == 0) {
	return;
    }

    (*_OutputStream) << _Name << " plot " << _PointRadius;
    int Address;
    int NumberOfPoints = _Ntuple->NumberOfRows();
    for (int Index = 0; Index < NumberOfPoints; Index++) {
	Address = (*_Ntuple)[Index][0];
	if (! IsValidAddress(Address)) {
	    continue;
	}

	(*_OutputStream) << " " << PositionXOf(Address);
	(*_OutputStream) << " " << PositionYOf(Address);
	(*_OutputStream) << " " << (*_Ntuple)[Index][1];
    }
    (*_OutputStream) << ";" << endl;
}

void TKinokoKoapMapView::ClearThis(void)
{
    TKinokoMapView::ClearThis();
    
    (*_OutputStream) << _Name << " clear;" << endl;
    (*_OutputStream) << _Name << " frame ";
    (*_OutputStream) << _XMin << " " << _XMax << " ";
    (*_OutputStream) << _YMin << " " << _YMax << " ";
    (*_OutputStream) << _ZMin << " " << _ZMax << ";" << endl;
}
