/* KinokoTrendView.cc */
/* Created by Enomoto Sanshiro on 5 December 2000. */
/* Updated by Enomoto Sanshiro on 29 September 2002. */
/* Last updated by Enomoto Sanshiro on 13 February 2008. */


#include <cmath>
#include "MushMisc.hh"
#include "KameTrend.hh"
#include "KameRepository.hh"
#include "KinokoPlatform.hh"
#include "KinokoTrendView.hh"

using namespace std;
using namespace kame;


TKinokoTrendView::TKinokoTrendView(TKameTrend* Trend, double MinValue, double MaxValue)
{
    _Trend = Trend;
    _MinValue = MinValue;
    _MaxValue = MaxValue;
    _WindowLength = _Trend->WindowLength();

    _DataStartTime = 0;
    _DisplayStartTime = 0;

    _DisplayValueFieldId = FieldId_Deviation;

    _DisplayStatisticsFlags = Stat_Counts | Stat_Mean | Stat_Deviation;
    _IsDisplayStatisticsSpecified = false;

    _LowerBound = 0;
    _UpperBound = 0;
    _EventEmitter = 0;
    
    _IsYScaleLog = false;

    _NumberOfPointsInside = 0;
    _NumberOfPointsOutside = 0;

    _IsTimeTickInitialized = false;
    _IsTimeTickAbsolute = false;
}

TKinokoTrendView::~TKinokoTrendView()
{
    delete _Trend;
}

void TKinokoTrendView::Draw(void)
{
    if (_DisplayStartTime == 0) {
	_DisplayStartTime = TMushDateTime::SecSinceEpoch();
	_DataStartTime = _Trend->StartTime();
    }

    if (_DataStartTime != _Trend->StartTime()) {
	_DisplayStartTime += _Trend->StartTime() - _DataStartTime;
	_DataStartTime = _Trend->StartTime();
    }

    _Trend->UpdateRange(
	_DataStartTime + TMushDateTime::SecSince(_DisplayStartTime)
	- _Trend->TickInterval()
    );

    if (! _IsTimeTickInitialized) {
	long Epoch = -1;
	if (_IsTimeTickAbsolute) {
	    Epoch = _Trend->StartTime();
	}
	InitializeTimeTick(Epoch);
	_IsTimeTickInitialized = true;
    }
    
    TKinokoView::Draw();
}

void TKinokoTrendView::SetDisplayValue(int FieldId)
{
    _DisplayValueFieldId = FieldId;

    if (! _IsDisplayStatisticsSpecified){
	switch (_DisplayValueFieldId) {
	  case FieldId_Counts:
	    _DisplayStatisticsFlags = Stat_Counts;
	    break;
	  case FieldId_Sum:
	    _DisplayStatisticsFlags = Stat_Counts | Stat_Sum;
	    break;
	  case FieldId_Mean:
	    _DisplayStatisticsFlags = Stat_Counts | Stat_Mean;
	    break;
	  case FieldId_Deviation:
	    _DisplayStatisticsFlags = Stat_Counts | Stat_Mean | Stat_Deviation;
	    break;
	  default:
	    break;
	}
    }
}

void TKinokoTrendView::SetDisplayStatistics(int StatisticsFlags)
{
    _DisplayStatisticsFlags = StatisticsFlags;
    _IsDisplayStatisticsSpecified = true;
}

void TKinokoTrendView::SetYScaleLog(void)
{
    _IsYScaleLog = true;

    if (_MinValue <= 0) {
	_MinValue = 0.5;
    }
    if (_MaxValue < _MinValue) {
	_MaxValue = 10 * _MinValue;
    }
}

void TKinokoTrendView::SetYScaleLinear(void)
{
    _IsYScaleLog = false;
}

void TKinokoTrendView::SetTimeTick(const std::string& Format, const std::string& Unit, int Step)
{
    _TimeTickFormat = Format;
    _TimeTickUnit = Unit;
    _TimeTickStep = Step;

    _IsTimeTickAbsolute = true;
    _IsTimeTickInitialized = false;
}

void TKinokoTrendView::SetAxisTitle(const std::string& XTitle, const std::string& YTitle)
{
    _XTitle = XTitle;
    _YTitle = YTitle;
}

void TKinokoTrendView::SetOperationRange(double Lower, double Upper, int NumberOfPointsToJudge)
{
    _LowerBound = Lower;
    _UpperBound = Upper;
    _NumberOfPointsToJudge = NumberOfPointsToJudge;
}

void TKinokoTrendView::EnableAlarm(TKinokoEventEmitter* EventEmitter, const string& Message)
{
    _EventEmitter = EventEmitter;
    _AlarmMessage = Message;
}

void TKinokoTrendView::DisableAlarm(void)
{
    _EventEmitter = 0;
}

void TKinokoTrendView::CheckValueRange(double Value)
{
    if ((_EventEmitter == 0) || (_LowerBound >= _UpperBound)) {
	return;
    }

    if ((Value >= _LowerBound) && (Value <= _UpperBound)) {
	_NumberOfPointsInside++;
	_NumberOfPointsOutside = 0;
    }
    else {
	_NumberOfPointsOutside++;
    }
    
    if (
	(_NumberOfPointsInside > _NumberOfPointsToJudge) &&
	(_NumberOfPointsOutside == _NumberOfPointsToJudge)
    ){
	vector<string> ArgumentList;
	ArgumentList.push_back(_AlarmMessage);
	_EventEmitter->EmitEvent("alarm", ArgumentList);

	_NumberOfPointsInside = 0;
    }
}

void TKinokoTrendView::SaveThis(TKameRepository* Repository)throw(TKinokoException)
{
    try {
	Repository->SaveTrend(*_Trend, _Name);
    }
    catch (TKameException &e) {
	throw TKinokoException(
	    "TKinokoTrendView::SaveThis()", e.Message()
	);
    }
}

void TKinokoTrendView::ClearThis(void)
{
    long ClearTime;
    if (_Trend->NumberOfPoints() > 0) {
	ClearTime = _Trend->TimeOf(_Trend->NumberOfPoints()-1);
    }
    else {
	ClearTime = _Trend->StartTime();
    }

    _Trend->Clear();

    if (ClearTime > 0) {
	_Trend->Start(ClearTime);
	_DisplayStartTime = 0;
    }
    
    _NumberOfPointsInside = 0;
    _NumberOfPointsOutside = 0;
}

void TKinokoTrendView::InitializeTimeTick(int Epoch)
{
}



TKinokoKoapTrendView::TKinokoKoapTrendView(TKameTrend* Trend, double MinValue, double MaxValue, ostream* OutputStream)
: TKinokoTrendView(Trend, MinValue, MaxValue)
{
    _OutputStream = OutputStream;
}

TKinokoKoapTrendView::~TKinokoKoapTrendView()
{
}

void TKinokoKoapTrendView::DeployThis(void) 
{
    *_OutputStream << ".selectPage " << _PageNumber << ";" << endl;

    *_OutputStream << ".create plot " << _Name << " ";
    *_OutputStream << 100 * Left() << " " << 100 * Top() << " ";
    *_OutputStream << 100 * Width() << " " << 100 * Height() << ";" << endl;

    *_OutputStream << _Name << " set title " << _Trend->Title() << ";" << endl;
    if (! _XTitle.empty()) {
	*_OutputStream << _Name << " set xtitle " << _XTitle << ";" << endl;
    }
    if (! _YTitle.empty()) {
	*_OutputStream << _Name << " set ytitle " << _YTitle << ";" << endl;
    }

    if (_IsYScaleLog) {
	*_OutputStream << _Name << " set yscale log;" << endl;
    }

    *_OutputStream << _Name << " set commentboxcolor background;" << endl;

    *_OutputStream << _Name << " frame ";
    *_OutputStream << 0 << " " << _WindowLength << " ";
    *_OutputStream << _MinValue << " " << _MaxValue << ";" << endl;
}

void TKinokoKoapTrendView::DrawThis(void)
{
    double MinTime = _Trend->PassedTimeOf(0);
    double MaxTime = MinTime + _WindowLength;

    *_OutputStream << _Name << " clear;" << endl;
    *_OutputStream << _Name << " frame ";
    *_OutputStream << MinTime << " " << MaxTime << " ";
    *_OutputStream << _MinValue << " " << _MaxValue << ";" << endl;

    if (_LowerBound < _UpperBound) {
	*_OutputStream << _Name << " savecontext;" << endl;
	*_OutputStream << _Name << " set color lightgray;" << endl;
	*_OutputStream << _Name << " drawrectfill ";
	*_OutputStream << MinTime << " " << _UpperBound << " ";
	*_OutputStream << MaxTime << " " << _LowerBound << ";";
	*_OutputStream << endl;
	*_OutputStream << _Name << " restorecontext;" << endl;
	*_OutputStream << _Name << " drawframe;" << endl;
    }

    if (_Trend->NumberOfPoints() == 0) {
        *_OutputStream << _Name << " drawtext ";
        *_OutputStream << (7 * MinTime + MaxTime) / 8 << " ";
        *_OutputStream << (_MinValue + _MaxValue) / 2 << " ";
        *_OutputStream << "no data;" << endl;
	return;
    }

    int LastPointIndex = _Trend->NumberOfPoints() - 2;
    if (LastPointIndex < 0) {
	return;
    }

    double LastValue;
    switch (_DisplayValueFieldId) {
      case FieldId_Counts:
	DrawCountPlot();
	LastValue = _Trend->CountsOf(LastPointIndex);
	break;
      case FieldId_Sum:
	DrawSumPlot();
	LastValue = _Trend->SumOf(LastPointIndex);
	break;
      case FieldId_Mean:
	DrawMeanPlot(false);
	LastValue = _Trend->MeanOf(LastPointIndex);
	break;
      case FieldId_Deviation:
	DrawMeanPlot(true);
	LastValue = _Trend->MeanOf(LastPointIndex);
	break;
      default:
        LastValue = 0;
	break;
    }

    if (_DisplayStatisticsFlags) {
	*_OutputStream << _Name << " comment -100 0 ";
    }
    if (_DisplayStatisticsFlags & Stat_Counts) {
	*_OutputStream << "counts: " << _Trend->CountsOf(LastPointIndex) << endl;
    }
    if (_DisplayStatisticsFlags & Stat_Mean) {
	*_OutputStream << "mean: " << _Trend->MeanOf(LastPointIndex) << endl;
    }
    if (_DisplayStatisticsFlags & Stat_Deviation) {
	*_OutputStream << "rms: " << _Trend->DeviationOf(LastPointIndex) << endl;
    }
    if (_DisplayStatisticsFlags) {
	*_OutputStream << ";" << endl;
    }

    CheckValueRange(LastValue);
}

void TKinokoKoapTrendView::DrawCountPlot(void)
{
    *_OutputStream << _Name << " plot";

    int NumberOfPoints = _Trend->NumberOfPoints();

    int PointIndex;
    double PassedTime, Counts;
    for (PointIndex = 0; PointIndex < NumberOfPoints-1; PointIndex++) {
	PassedTime = _Trend->PassedTimeOf(PointIndex);
	Counts = _Trend->CountsOf(PointIndex);
	if (Counts > _MaxValue) {
	    Counts = _MaxValue;
	}
	else if (Counts < _MinValue) {
	    Counts = _MinValue;
	}
	    
	*_OutputStream << " " << PassedTime << " " << Counts;
    }
    *_OutputStream << ";" << endl;

    if (PointIndex >= 0) {
	PassedTime = _Trend->PassedTimeOf(PointIndex);
	Counts = _Trend->CountsOf(PointIndex);
	*_OutputStream << _Name << " plot";
	*_OutputStream << " " << PassedTime << " " << Counts;
	*_OutputStream << ";" << endl;
    }
}

void TKinokoKoapTrendView::DrawSumPlot(void)
{
    *_OutputStream << _Name << " plot";

    int NumberOfPoints = _Trend->NumberOfPoints();

    int PointIndex;
    double PassedTime, Sum;
    for (PointIndex = 0; PointIndex < NumberOfPoints-1; PointIndex++) {
	PassedTime = _Trend->PassedTimeOf(PointIndex);
	Sum = _Trend->SumOf(PointIndex);
	if (Sum > _MaxValue) {
	    Sum = _MaxValue;
	}
	else if (Sum < _MinValue) {
	    Sum = _MinValue;
	}
	    
	*_OutputStream << " " << PassedTime << " " << Sum;
    }
    *_OutputStream << ";" << endl;

    if (PointIndex >= 0) {
	PassedTime = _Trend->PassedTimeOf(PointIndex);
	Sum = _Trend->SumOf(PointIndex);
	if (Sum > _MaxValue) {
	    Sum = _MaxValue;
	}
	else if (Sum < _MinValue) {
	    Sum = _MinValue;
	}
	    
	*_OutputStream << _Name << " plot";
	*_OutputStream << " " << PassedTime << " " << Sum;
	*_OutputStream << ";" << endl;
    }
}

void TKinokoKoapTrendView::DrawMeanPlot(bool DrawsDeviation)
{
    if (DrawsDeviation) {
	*_OutputStream << _Name << " errorplot";
    }
    else {
	*_OutputStream << _Name << " plot";
    }

    int NumberOfPoints = _Trend->NumberOfPoints();

    int PointIndex;
    double PassedTime, Counts;
    double Mean, Deviation;
    for (PointIndex = 0; PointIndex < NumberOfPoints-1; PointIndex++) {
	Counts = _Trend->CountsOf(PointIndex);
        if (Counts == 0) {
	    continue;
	}
	Mean = _Trend->MeanOf(PointIndex);
	Deviation = _Trend->DeviationOf(PointIndex);
        PassedTime = _Trend->PassedTimeOf(PointIndex);

	if (Mean > _MaxValue) {
	    Mean = _MaxValue;
	    Deviation = 0;
	}
	else if (Mean < _MinValue) {
	    Mean = _MinValue;
	    Deviation = 0;
	}
	    
	*_OutputStream << " " << PassedTime << " " << Mean;
	if (DrawsDeviation) {
	    *_OutputStream << " 0 " << Deviation;
	}
    }
    *_OutputStream << ";" << endl;

    if (PointIndex >= 0) {
	PassedTime = _Trend->PassedTimeOf(PointIndex);
	Mean = _Trend->MeanOf(PointIndex);
	if (Mean > _MaxValue) {
	    Mean = _MaxValue;
	}
	else if (Mean < _MinValue) {
	    Mean = _MinValue;
	}
	    
	*_OutputStream << _Name << " plot";
	*_OutputStream << " " << PassedTime << " " << Mean;
	*_OutputStream << ";" << endl;
    }
}

void TKinokoKoapTrendView::ClearThis(void)
{
    TKinokoTrendView::ClearThis();
    
    *_OutputStream << _Name << " clear;" << endl;
    *_OutputStream << _Name << " frame ";
    *_OutputStream << 0 << " " << _WindowLength << " ";
    *_OutputStream << _MinValue << " " << _MaxValue << ";" << endl;
}

void TKinokoKoapTrendView::InitializeTimeTick(int Epoch)
{
    if (Epoch > 0) {
	*_OutputStream << _Name << " setxscaletimetick" << " ";
	*_OutputStream << Epoch << " ";
	*_OutputStream << _TimeTickUnit << " " << _TimeTickStep << " ";
	*_OutputStream << _TimeTickFormat;
	*_OutputStream << ";" << endl;
    }
    else {
	*_OutputStream << _Name << " setxscaleelapsetick;" << endl;
    }
}
