/* KinokoViewMessenger.cc */
/* Created by Enomoto Sanshiro on 31 December 2000. */
/* Last updated by Enomoto Sanshiro on 29 September 2002. */


#include <string>
#include "ParaPackage.hh"
#include "KameRepository.hh"
#include "KameRepositoryFactory.hh"
#include "KinokoAnalysisMessenger.hh"
#include "KinokoAnalysisSequenceBuilder.hh"
#include "KinokoHistogramWriter.hh"
#include "KinokoHistogram2dWriter.hh"
#include "KinokoTrendWriter.hh"
#include "KinokoHistoryWriter.hh"
#include "KinokoWaveWriter.hh"
#include "KinokoMapWriter.hh"
#include "KinokoTabularWriter.hh"
#include "KinokoView.hh"
#include "KinokoViewCanvasPage.hh"
#include "KinokoViewBuilder.hh"
#include "KinokoViewFrame.hh"
#include "KinokoViewSequence.hh"
#include "KinokoViewFactory.hh"
#include "KinokoViewMessenger.hh"


using namespace std;
using namespace kame;


TKinokoViewSystemMessenger::TKinokoViewSystemMessenger(TKinokoViewBuilder* ViewBuilder, const string& ScriptFileName)
: TParaObjectPrototype("ViewSystem")
{
    _ViewBuilder = ViewBuilder;
    _ScriptFileName = ScriptFileName;
}

TKinokoViewSystemMessenger::~TKinokoViewSystemMessenger()
{
}

TParaObjectPrototype* TKinokoViewSystemMessenger::Clone(void)
{
    return new TKinokoViewSystemMessenger(_ViewBuilder, _ScriptFileName);
}

int TKinokoViewSystemMessenger::InvokeMethodByName(const std::string& MethodName, std::vector<TParaValue*>& ArgumentList, TParaValue& ReturnValue) throw(TScriptException)
{
    if ((MethodName == "scriptFileName") || (MethodName == "ScriptFileName")) {
	return ScriptFileName(ArgumentList, ReturnValue);
    }
    else if ((MethodName == "getRegistry") || (MethodName == "GetRegistry")) {
	return GetRegistry(ArgumentList, ReturnValue);
    }
    else if ((MethodName == "setRegistry") || (MethodName == "SetRegistry")) {
	return SetRegistry(ArgumentList, ReturnValue);
    }

    return 0;
}

int TKinokoViewSystemMessenger::ScriptFileName(vector<TParaValue*>& ArgumentList, TParaValue& ReturnValue) throw(TScriptException)
{
    ReturnValue = TParaValue(_ScriptFileName);
    return 1;
}

int TKinokoViewSystemMessenger::GetRegistry(vector<TParaValue*>& ArgumentList, TParaValue& ReturnValue) throw(TScriptException)
{
    if (ArgumentList.size() < 1) {
	throw TScriptException(
	    "too few arguments: getRegistry(string registry_path)"
	);
    }
    if (! ArgumentList[0]->IsString()) {
	throw TScriptException(
	    "invalid argument:  getRegistry(string registry_path)"
	);
    }

    string RegistryPath = ArgumentList[0]->AsString();
    TKinokoRegistry* Registry = _ViewBuilder->Registry();
    ReturnValue = TParaValue(Registry->GetValue(RegistryPath));

    return 1;
}

int TKinokoViewSystemMessenger::SetRegistry(vector<TParaValue*>& ArgumentList, TParaValue& ReturnValue) throw(TScriptException)
{
    if (ArgumentList.size() < 2) {
	throw TScriptException(
	    "too few argument[s]: "
	    "setRegistry(string registry_path, string value)"
	);
    }
    if (! ArgumentList[0]->IsString()) {
	throw TScriptException(
	    "invalid argument[s]: "
	    "setRegistry(string registry_path, string value)"
	);
    }

    string RegistryPath = ArgumentList[0]->AsString();
    string Value = ArgumentList[1]->AsString();

    TKinokoRegistry* Registry = _ViewBuilder->Registry();
    Registry->SetValue(RegistryPath, Value);
    ReturnValue = TParaValue(Value);

    return 1;
}



TKinokoViewRepositoryMessenger::TKinokoViewRepositoryMessenger(TKinokoViewBuilder* ViewBuilder)
: TParaObjectPrototype("ViewRepository")
{
    _ViewBuilder = ViewBuilder;
    _Repository = 0;
}

TKinokoViewRepositoryMessenger::~TKinokoViewRepositoryMessenger()
{
}

TParaObjectPrototype* TKinokoViewRepositoryMessenger::Clone(void)
{
    return new TKinokoViewRepositoryMessenger(_ViewBuilder);
}

int TKinokoViewRepositoryMessenger::InvokeMethodByName(const std::string& MethodName, std::vector<TParaValue*>& ArgumentList, TParaValue& ReturnValue) throw(TScriptException)
{
    return 0;
}

void TKinokoViewRepositoryMessenger::Construct(const std::string& ClassName, std::vector<TParaValue*>& ArgumentList) throw(TScriptException)
{
    if (ArgumentList.size() < 1) {
	throw TScriptException(
	    "ViewRepository::ViewRepository(string file_name)",
	    "too few argument[s]"
	);
    }
    string FileName = ArgumentList[0]->AsString();
    _Repository = 0;

    TKameRepositoryFactory* Factory = 0;
    if (ClassName == "PlainTextViewRepository") {
	Factory = TKameRepositoryFactory::GetInstance("PlainText");
    }
    else if (ClassName == "RootViewRepository") {
	Factory = TKameRepositoryFactory::GetInstance("Root");
    }
    else {
	Factory = TKameRepositoryFactory::GetInstance("PlainText");
    }

    if (Factory != 0) {
	_Repository = Factory->CreateRepository(FileName);
    }
    
    if (_Repository == 0) {
	throw TScriptException(
	    "ViewRepository::ViewRepository(string file_name)",
	    "repository not available: " + ClassName
	);
    }

    _ViewBuilder->AddViewRepository(_Repository);
}

TKameRepository* TKinokoViewRepositoryMessenger::Repository(void)
{
    return _Repository;
}



TKinokoViewCanvasPageMessenger::TKinokoViewCanvasPageMessenger(TKinokoViewBuilder* ViewBuilder)
: TParaObjectPrototype("ViewCanvasPage")
{
    _ViewBuilder = ViewBuilder;
}

TKinokoViewCanvasPageMessenger::~TKinokoViewCanvasPageMessenger()
{
}

TParaObjectPrototype* TKinokoViewCanvasPageMessenger::Clone(void)
{
    return new TKinokoViewCanvasPageMessenger(_ViewBuilder);
}

void TKinokoViewCanvasPageMessenger::Construct(const string& ClassName, vector<TParaValue*>& ArgumentList) throw(TScriptException)
{
    string PageName;
    if (ArgumentList.size() >= 1) {
	PageName = ArgumentList[0]->AsString();
    }
    else {
	PageName = ObjectName();
    }
    if (PageName.empty()) {
	PageName = "untitled";
    }

    int PageNumber = _ViewBuilder->ViewCanvas()->NumberOfPages();

    _Page = _ViewBuilder->ViewFactory()->CreateViewCanvasPage(
	PageNumber, PageName
    );

    _ViewBuilder->AddPage(_Page);
}

int TKinokoViewCanvasPageMessenger::InvokeMethodByName(const string& MethodName, vector<TParaValue*>& ArgumentList, TParaValue& ReturnValue) throw(TScriptException)
{
    if (MethodName == "put") {
	return PutView(ArgumentList, ReturnValue);
    }

    return 0;
}

int TKinokoViewCanvasPageMessenger::PutView(std::vector<TParaValue*>& ArgumentList, TParaValue& ReturnValue) throw(TScriptException)
{
    if (
	(ArgumentList.size() < 1) ||
	! ArgumentList[0]->IsObject("View")
    ){	
	throw TScriptException(
	    "Page::put(View child_view)", "invalid argument[s]"
	);
    }

    _Page->AddView(
	((TKinokoViewMessenger*) ArgumentList[0]->AsObject())->View()
    );
    
    return 1;
}



TKinokoViewMessenger::TKinokoViewMessenger(TKinokoViewBuilder* ViewBuilder, TKinokoAnalysisSequenceBuilder* AnalysisSequenceBuilder)
: TParaObjectPrototype("View")
{
    _ViewBuilder = ViewBuilder;
    _AnalysisSequenceBuilder = AnalysisSequenceBuilder;

    _ViewFactory = _ViewBuilder->ViewFactory();
}

TKinokoViewMessenger::~TKinokoViewMessenger()
{
}

int TKinokoViewMessenger::InvokeMethodByName(const string& MethodName, vector<TParaValue*>& ArgumentList, TParaValue& ReturnValue) throw(TScriptException)
{
    if ((MethodName == "fill") || (MethodName == "Fill")) {
	return Fill(ArgumentList, ReturnValue);
    }
    else if ((MethodName == "fillOne") || (MethodName == "FillOne")) {
	bool IsSingleEventAction;
	return Fill(ArgumentList, ReturnValue, IsSingleEventAction = true);
    }
    else if ((MethodName == "draw") || (MethodName == "Draw")) {
	return Draw(ArgumentList, ReturnValue);
    }
    else if ((MethodName == "clear") || (MethodName == "Clear")) {
	return Clear(ArgumentList, ReturnValue);
    }
    else if ((MethodName == "save") || (MethodName == "Save")) {
	return Save(ArgumentList, ReturnValue);
    }
    else if ((MethodName == "putLine") || (MethodName == "PutLine")) {
	return PutLine(ArgumentList, ReturnValue);
    }
    else if ((MethodName == "putRectangle") || (MethodName == "PutRectangle")) {
	return PutRectangle(ArgumentList, ReturnValue);
    }
    else if ((MethodName == "putCircle") || (MethodName == "PutCircle")) {
	return PutCircle(ArgumentList, ReturnValue);
    }
    else if ((MethodName == "putText") || (MethodName == "PutText")) {
	return PutText(ArgumentList, ReturnValue);
    }
    else if ((MethodName == "putImage") || (MethodName == "PutImage")) {
	return PutImage(ArgumentList, ReturnValue);
    }
    else if ((MethodName == "setColor") || (MethodName == "SetColor")) {
	return SetColor(ArgumentList, ReturnValue);
    }
    else if ((MethodName == "setTextAdjustment") || (MethodName == "SetTextAdjustment")) {
	return SetTextAdjustment(ArgumentList, ReturnValue);
    }
    else if ((MethodName == "setFont") || (MethodName == "SetFont")) {
	return SetFont(ArgumentList, ReturnValue);
    }

#if 1
    //... for backward compatibility ...//
    else if ((MethodName == "take") || (MethodName == "Take")) {
	return Fill(ArgumentList, ReturnValue);
    }
    else if ((MethodName == "takeOne") || (MethodName == "TakeOne")) {
	bool IsSingleEventAction;
	return Fill(ArgumentList, ReturnValue, IsSingleEventAction = true);
    }
#endif

    return 0;
}

int TKinokoViewMessenger::Fill(vector<TParaValue*>& ArgumentList, TParaValue& ReturnValue, bool IsSingleEventAction) throw(TScriptException)
{
    if (ArgumentList.size() < 1) {
	throw TScriptException(
	    "View::fill(DataElement data_element): too few argument[s]"
	);
    }

    vector<TKinokoDataElementMessenger*> DataElementMessengerList;
    for (unsigned i = 0; i < ArgumentList.size(); i++) {
	if (! ArgumentList[i]->IsObject("DataElement")) {
	    throw TScriptException(
		"View::fill(DataElement data_element): invalid argument[s]"
	    );
	}

	DataElementMessengerList.push_back(
	    (TKinokoDataElementMessenger*) ArgumentList[i]->AsObject()
	);
    }

    TKinokoDataAnalyzer* Analyzer;
    Analyzer = CreateAnalysisAction(DataElementMessengerList);
    if (Analyzer == 0) {
	throw TScriptException(
	    "View::fill(DataElement data_element): invalid data element"
	);
    }

    try {
	if (! IsSingleEventAction) {
	    _AnalysisSequenceBuilder->AddAction(Analyzer);
	}
	else {
	    _AnalysisSequenceBuilder->AddSingleEventAction(Analyzer);
	}
    }
    catch (TKinokoException &e) {
	throw TScriptException(e.Message());
    }

    return 1;
}

int TKinokoViewMessenger::Draw(std::vector<TParaValue*>& ArgumentList, TParaValue& ReturnValue) throw(TScriptException)
{
    try {
	_ViewBuilder->AddAction(new TKinokoViewDrawAction(View()));
    }
    catch (TKinokoException &e) {
	throw TScriptException(e.Message());
    }

    if (ArgumentList.size() > 0) {
	string OptionString = ArgumentList[0]->AsString();
	if (! View()->SetDrawOption(OptionString)) {
	    throw TScriptException(
		"View::draw(string option)", "invalid option: " + OptionString
	    );
	}
    }

    return 1;
}

int TKinokoViewMessenger::Clear(std::vector<TParaValue*>& ArgumentList, TParaValue& ReturnValue) throw(TScriptException)
{
    try {
	_ViewBuilder->AddAction(new TKinokoViewClearAction(View()));
    }
    catch (TKinokoException &e) {
	throw TScriptException(e.Message());
    }

    return 1;
}

int TKinokoViewMessenger::Save(std::vector<TParaValue*>& ArgumentList, TParaValue& ReturnValue) throw(TScriptException)
{
    if (
	(ArgumentList.size() < 1) ||
	(! ArgumentList[0]->IsObject("ViewRepository"))
    ){
	throw TScriptException(
	    "View::Save(ViewRepository repository)", "invalid argument[s]"
	);
    }

    TKameRepository* Repository = (
	((TKinokoViewRepositoryMessenger*) ArgumentList[0]->AsObject())->Repository()
    );

    try {
	_ViewBuilder->AddAction(new TKinokoViewSaveAction(View(), Repository));
    }
    catch (TKinokoException &e) {
	throw TScriptException(e.Message());
    }

    return 1;
}

int TKinokoViewMessenger::PutLine(std::vector<TParaValue*>& ArgumentList, TParaValue& ReturnValue) throw(TScriptException)
{
    if (ArgumentList.size() < 4) {
	throw TScriptException(
	    "View::putLine(double x0, double y0, double x1, double y1)",
	    "too few argument[s]"
	);
    }

    double x0 = ArgumentList[0]->AsDouble();
    double y0 = ArgumentList[1]->AsDouble();
    double x1 = ArgumentList[2]->AsDouble();
    double y1 = ArgumentList[3]->AsDouble();

    View()->AddViewTrimming(
	_ViewFactory->CreateLineTrimming(View(), x0, y0, x1, y1)
    );

    return 1;
}

int TKinokoViewMessenger::PutRectangle(std::vector<TParaValue*>& ArgumentList, TParaValue& ReturnValue) throw(TScriptException)
{
    if (ArgumentList.size() < 4) {
	throw TScriptException(
	    "View::putRectangle(double x0, double y0, double x1, double y1)",
	    "too few argument[s]"
	);
    }

    double x0 = ArgumentList[0]->AsDouble();
    double y0 = ArgumentList[1]->AsDouble();
    double x1 = ArgumentList[2]->AsDouble();
    double y1 = ArgumentList[3]->AsDouble();

    View()->AddViewTrimming(
	_ViewFactory->CreateRectangleTrimming(View(), x0, y0, x1, y1)
    );

    return 1;
}

int TKinokoViewMessenger::PutCircle(std::vector<TParaValue*>& ArgumentList, TParaValue& ReturnValue) throw(TScriptException)
{
    if (ArgumentList.size() < 3) {
	throw TScriptException(
	    "View::putCircle(double x, double y, double r)",
	    "too few argument[s]"
	);
    }

    double x = ArgumentList[0]->AsDouble();
    double y = ArgumentList[1]->AsDouble();
    double r = ArgumentList[2]->AsDouble();

    View()->AddViewTrimming(
	_ViewFactory->CreateCircleTrimming(View(), x, y, r)
    );

    return 1;
}

int TKinokoViewMessenger::PutText(std::vector<TParaValue*>& ArgumentList, TParaValue& ReturnValue) throw(TScriptException)
{
    if (ArgumentList.size() < 3) {
	throw TScriptException(
	    "View::putText(double x, double y, string text)",
	    "too few argument[s]"
	);
    }

    double x = ArgumentList[0]->AsDouble();
    double y = ArgumentList[1]->AsDouble();
    string Text = ArgumentList[2]->AsString();

    View()->AddViewTrimming(
	_ViewFactory->CreateTextTrimming(View(), x, y, Text)
    );

    return 1;
}

int TKinokoViewMessenger::PutImage(std::vector<TParaValue*>& ArgumentList, TParaValue& ReturnValue) throw(TScriptException)
{
    if (ArgumentList.size() < 3) {
	throw TScriptException(
	    "View::putImage(double x, double y, string file_name)",
	    "too few argument[s]"
	);
    }

    double x = ArgumentList[0]->AsDouble();
    double y = ArgumentList[1]->AsDouble();
    string FileName = ArgumentList[2]->AsString();

    View()->AddViewTrimming(
	_ViewFactory->CreateImageTrimming(View(), x, y, FileName)
    );

    return 1;
}

int TKinokoViewMessenger::SetColor(std::vector<TParaValue*>& ArgumentList, TParaValue& ReturnValue) throw(TScriptException)
{
    if (ArgumentList.size() < 1) {
	throw TScriptException(
	    "View::setColor(string color_name)", "too few argument[s]"
	);
    }

    string ColorName = ArgumentList[0]->AsString();

    View()->AddViewTrimming(
	_ViewFactory->CreateColorTrimming(View(), ColorName)
    );

    return 1;
}

int TKinokoViewMessenger::SetTextAdjustment(std::vector<TParaValue*>& ArgumentList, TParaValue& ReturnValue) throw(TScriptException)
{
    if (ArgumentList.size() < 1) {
	throw TScriptException(
	    "View::setTextAdjustment(string text_adjustment)", 
	    "too few argument[s]"
	);
    }

    string TextAdjustmentString = ArgumentList[0]->AsString();

    View()->AddViewTrimming(
	_ViewFactory->CreateTextAdjustmentTrimming(View(), TextAdjustmentString)
    );

    return 1;
}

int TKinokoViewMessenger::SetFont(std::vector<TParaValue*>& ArgumentList, TParaValue& ReturnValue) throw(TScriptException)
{
    if (ArgumentList.size() < 2) {
	throw TScriptException(
	    "View::setFont(string font_name, int font_size)", 
	    "too few argument[s]"
	);
    }

    string FontName = ArgumentList[0]->AsString();
    int FontSize = ArgumentList[1]->AsLong();

    View()->AddViewTrimming(
	_ViewFactory->CreateFontTrimming(View(), FontName, FontSize)
    );

    return 1;
}



TKinokoViewGridMessenger::TKinokoViewGridMessenger(TKinokoViewBuilder* ViewBuilder, TKinokoAnalysisSequenceBuilder* AnalysisSequenceBuilder)
: TKinokoViewMessenger(ViewBuilder, AnalysisSequenceBuilder)
{
}

TKinokoViewGridMessenger::~TKinokoViewGridMessenger()
{
}

TParaObjectPrototype* TKinokoViewGridMessenger::Clone(void)
{
    return new TKinokoViewGridMessenger(_ViewBuilder, _AnalysisSequenceBuilder);
}

void TKinokoViewGridMessenger::Construct(const string& ClassName, vector<TParaValue*>& ArgumentList) throw(TScriptException)
{
    int NumberOfColumns = 0;
    if (ArgumentList.size() >= 1) {
	if (! ArgumentList[0]->IsLong()) {
	    throw TScriptException(
		"Grid::Grid(int number_of_columns)", "invalid argument[s]"
	    );
	}
	NumberOfColumns = ArgumentList[0]->AsLong();
    }

    _Grid = new TKinokoViewGridFrame(NumberOfColumns);
    _ViewBuilder->AddView(_Grid);
}

int TKinokoViewGridMessenger::InvokeMethodByName(const string& MethodName, vector<TParaValue*>& ArgumentList, TParaValue& ReturnValue) throw(TScriptException)
{
    if (MethodName == "put") {
	return PutView(ArgumentList, ReturnValue);
    }

    return TKinokoViewMessenger::InvokeMethodByName(MethodName, ArgumentList, ReturnValue);
}

int TKinokoViewGridMessenger::PutView(std::vector<TParaValue*>& ArgumentList, TParaValue& ReturnValue) throw(TScriptException)
{
    if (
	(ArgumentList.size() < 1) ||
	! ArgumentList[0]->IsObject("View")
    ){	
	throw TScriptException(
	    "Grid::put(View child_view)", "invalid argument[s]"
	);
    }

    _Grid->AddView(
	((TKinokoViewMessenger*) ArgumentList[0]->AsObject())->View()
    );
    
    return 1;
}

TKinokoView* TKinokoViewGridMessenger::View(void)
{
    return _Grid;
}

TKinokoDataAnalyzer* TKinokoViewGridMessenger::CreateAnalysisAction(vector<TKinokoDataElementMessenger*> DataElementMessengerList)
{
    return 0;
}



TKinokoViewPlacerMessenger::TKinokoViewPlacerMessenger(TKinokoViewBuilder* ViewBuilder, TKinokoAnalysisSequenceBuilder* AnalysisSequenceBuilder)
: TKinokoViewMessenger(ViewBuilder, AnalysisSequenceBuilder)
{
}

TKinokoViewPlacerMessenger::~TKinokoViewPlacerMessenger()
{
}

TParaObjectPrototype* TKinokoViewPlacerMessenger::Clone(void)
{
    return new TKinokoViewPlacerMessenger(_ViewBuilder, _AnalysisSequenceBuilder);
}

int TKinokoViewPlacerMessenger::InvokeMethodByName(const string& MethodName, vector<TParaValue*>& ArgumentList, TParaValue& ReturnValue) throw(TScriptException)
{
    if (MethodName == "put") {
	return PutView(ArgumentList, ReturnValue);
    }

    return TKinokoViewMessenger::InvokeMethodByName(MethodName, ArgumentList, ReturnValue);
}

void TKinokoViewPlacerMessenger::Construct(const string& ClassName, vector<TParaValue*>& ArgumentList) throw(TScriptException)
{
    if (ArgumentList.size() == 0) {
	_Placer = new TKinokoViewPlacerFrame();
    }
    else if (
	(ArgumentList.size() < 4) ||
	(! ArgumentList[0]->IsNumeric()) ||
	(! ArgumentList[1]->IsNumeric()) ||
	(! ArgumentList[2]->IsNumeric()) ||
	(! ArgumentList[3]->IsNumeric())
    ){
	throw TScriptException(
	    "Placer::Placer(float x0, float x1, float y0, float y1)", 
	    "invalid argument[s]"
	);
    }
    else {
	float X0 = ArgumentList[0]->AsDouble();
	float X1 = ArgumentList[1]->AsDouble();
	float Y0 = ArgumentList[2]->AsDouble();
	float Y1 = ArgumentList[3]->AsDouble();
	_Placer = new TKinokoViewPlacerFrame(X0, X1, Y0, Y1);
    }

    _ViewBuilder->AddView(_Placer);
}

int TKinokoViewPlacerMessenger::PutView(std::vector<TParaValue*>& ArgumentList, TParaValue& ReturnValue) throw(TScriptException)
{
    if (
	(ArgumentList.size() < 5) ||
	(! ArgumentList[0]->IsObject("View"))
    ){	
	throw TScriptException(
	    "Placer::put"
            "(View child_view, float x0, float x1, float y0, float y1)", 
	    "invalid argument[s]"
	);
    }

    TKinokoViewPlacement Placement;
    try {
	Placement.x0 = ArgumentList[1]->AsDouble();
	Placement.x1 = ArgumentList[2]->AsDouble();
	Placement.y0 = ArgumentList[3]->AsDouble();
	Placement.y1 = ArgumentList[4]->AsDouble();
    }
    catch (TScriptException &e) {
	throw TScriptException(
	    "Placer::put"
            "(View child_view, float x0, float x1, float y0, float y1)", 
	    "invalid argument[s]"
	);
    }

    _Placer->AddViewAt(
	((TKinokoViewMessenger*) ArgumentList[0]->AsObject())->View(),
	Placement
    );
    
    return 1;
}

TKinokoView* TKinokoViewPlacerMessenger::View(void)
{
    return _Placer;
}

TKinokoDataAnalyzer* TKinokoViewPlacerMessenger::CreateAnalysisAction(vector<TKinokoDataElementMessenger*> DataElementMessengerList)
{
    return 0;
}



TKinokoHistogramViewMessenger::TKinokoHistogramViewMessenger(TKinokoViewBuilder* ViewBuilder, TKinokoAnalysisSequenceBuilder* AnalysisSequenceBuilder)
: TKinokoViewMessenger(ViewBuilder, AnalysisSequenceBuilder)
{
    _HistogramView = 0;
    _Histogram = 0;
}

TKinokoHistogramViewMessenger::~TKinokoHistogramViewMessenger()
{
}

TParaObjectPrototype* TKinokoHistogramViewMessenger::Clone(void)
{
    return new TKinokoHistogramViewMessenger(_ViewBuilder, _AnalysisSequenceBuilder);
}

void TKinokoHistogramViewMessenger::Construct(const string& ClassName, vector<TParaValue*>& ArgumentList) throw(TScriptException)
{
    if (ArgumentList.size() < 4) {
	throw TScriptException("Histogram::Histogram(): too few argument[s]");
    }
    if (
	(! ArgumentList[1]->IsLong()) ||
	(! ArgumentList[2]->IsNumeric()) ||
	(! ArgumentList[3]->IsNumeric())
    ){
	throw TScriptException("Histogram::Histogram(): invalid argument[s]");
    }

    _Name = this->ObjectName();
    string Title = ArgumentList[0]->AsString();
    int NumberOfBins = ArgumentList[1]->AsLong();
    double Min = ArgumentList[2]->AsDouble();
    double Max = ArgumentList[3]->AsDouble();

    if ((NumberOfBins <= 0) || (Max <= Min)) {
	throw TScriptException("Histogram::Histogram(): incorrect bin range");
    }

    _Histogram = new TKameHistogram(
	TKameHistogramScale(NumberOfBins, Min, Max)
    );
    _Histogram->SetTitle(Title);

    _HistogramView = _ViewFactory->CreateHistogramView(_Histogram);
    _HistogramView->SetName(_Name);

    _ViewBuilder->AddView(_HistogramView);
}

int TKinokoHistogramViewMessenger::InvokeMethodByName(const string& MethodName, vector<TParaValue*>& ArgumentList, TParaValue& ReturnValue) throw(TScriptException)
{
    if ((MethodName == "setYScaleLog") || (MethodName == "setLogY")) {
	return SetYScaleLog(ArgumentList, ReturnValue);
    }
    else if ((MethodName == "setYScaleLinear") || (MethodName == "setLinearY")) {
	return SetYScaleLinear(ArgumentList, ReturnValue);
    }
    else if (MethodName == "setAxisTitle") {
	return SetAxisTitle(ArgumentList, ReturnValue);
    }
    else if (MethodName == "setDisplayStatistics") {
	return SetDisplayStatistics(ArgumentList, ReturnValue);
    }
    else if (MethodName == "setReferenceHistogram") {
	return SetReferenceHistogram(ArgumentList, ReturnValue);
    }
    //... for backward compatibility ...//
    else if (MethodName == "setExpectedDistribution") {
	return SetReferenceHistogram(ArgumentList, ReturnValue);
    }

    return TKinokoViewMessenger::InvokeMethodByName(MethodName, ArgumentList, ReturnValue);
}

TKinokoView* TKinokoHistogramViewMessenger::View(void)
{
    return _HistogramView;
}

TKinokoDataAnalyzer* TKinokoHistogramViewMessenger::CreateAnalysisAction(vector<TKinokoDataElementMessenger*> DataElementMessengerList)
{
    TKinokoHistogramWriter* AnalysisAction = new TKinokoHistogramWriter(
	_Histogram, 
	DataElementMessengerList[0]->SectionPath(),
	DataElementMessengerList[0]->DataElementSpecifier()
    );
    AnalysisAction->SetTargetName(_Name);

    return AnalysisAction;
}

int TKinokoHistogramViewMessenger::SetDisplayStatistics(std::vector<TParaValue*>& ArgumentList, TParaValue& ReturnValue) throw(TScriptException)
{
    int StatisticsFlag = 0;

    for (unsigned i = 0; i < ArgumentList.size(); i++) {
	string StatName = ArgumentList[i]->AsString();
	if (StatName == "entries") {
	    StatisticsFlag |= TKinokoHistogramView::Stat_Entries;
	}
	if (StatName == "underflow") {
	    StatisticsFlag |= TKinokoHistogramView::Stat_Underflow;
	}
	if (StatName == "overflow") {
	    StatisticsFlag |= TKinokoHistogramView::Stat_Overflow;
	}
	if (StatName == "mean") {
	    StatisticsFlag |= TKinokoHistogramView::Stat_Mean;
	}
	if ((StatName == "deviation") || (StatName == "rms")) {
	    StatisticsFlag |= TKinokoHistogramView::Stat_Deviation;
	}
    }

    _HistogramView->SetDisplayStatistics(StatisticsFlag);

    return 1;
}

int TKinokoHistogramViewMessenger::SetYScaleLog(vector<TParaValue*>& ArgumentList, TParaValue& ReturnValue) throw(TScriptException)
{
    _HistogramView->SetYScaleLog();
    return 1;
}

int TKinokoHistogramViewMessenger::SetYScaleLinear(vector<TParaValue*>& ArgumentList, TParaValue& ReturnValue) throw(TScriptException)
{
    _HistogramView->SetYScaleLinear();
    return 1;
}

int TKinokoHistogramViewMessenger::SetAxisTitle(vector<TParaValue*>& ArgumentList, TParaValue& ReturnValue) throw(TScriptException)
{
    string XTitle, YTitle;
    if (ArgumentList.size() > 0) {
	XTitle = ArgumentList[0]->AsString();
    }
    if (ArgumentList.size() > 1) {
	YTitle = ArgumentList[1]->AsString();
    }

    _Histogram->Scale().SetTitle(XTitle);
    _HistogramView->SetAxisTitle(XTitle, YTitle);

    return 1;
}

int TKinokoHistogramViewMessenger::SetReferenceHistogram(vector<TParaValue*>& ArgumentList, TParaValue& ReturnValue) throw(TScriptException)
{
    if (
	(ArgumentList.size() < 2) ||
	(! ArgumentList[0]->IsObject("ViewRepository")) ||
	(! ArgumentList[1]->IsString())
    ){
	throw TScriptException(
	    "Histogram::setReferenceHistogram(ViewRepository repository, string name)", "invalid argument[s]"
	);
    }

    double ToleranceSigmas = 3;
    if (ArgumentList.size() > 2) {
	if (! ArgumentList[2]->IsNumeric()) {
	    throw TScriptException(
		"Histogram::setReferenceHistogram(ViewRepository repository, string name, dobule tolerance_sigmas=3)", "invalid argument[s]"
	    );
	}
	ToleranceSigmas = ArgumentList[2]->AsDouble();
	if ((ToleranceSigmas < 1) || (ToleranceSigmas > 6)) {
	    throw TScriptException(
		"Histogram::setReferenceHistogram(ViewRepository repository, string name, dobule tolerance_sigmas=3)", 
		"invalid tolerance value: must be between 1 and 6"
	    );
	}
    }

    TKameRepository* Repository = (
	((TKinokoViewRepositoryMessenger*) ArgumentList[0]->AsObject())->Repository()
    );

    string Name = ArgumentList[1]->AsString();

    TKameHistogram* ReferenceHistogram = new TKameHistogram();
    try {
	Repository->LoadHistogram(*ReferenceHistogram, Name);
    }
    catch (TKameException &e) {
	throw TScriptException(
	    "Histogram::setReferenceHistogram()", 
	    "unable to load histogram object: " + Name
	);
    }

    _HistogramView->SetReferenceHistogram(
	ReferenceHistogram, ToleranceSigmas
    );

    return 1;
}



TKinokoHistogram2dViewMessenger::TKinokoHistogram2dViewMessenger(TKinokoViewBuilder* ViewBuilder, TKinokoAnalysisSequenceBuilder* AnalysisSequenceBuilder)
: TKinokoViewMessenger(ViewBuilder, AnalysisSequenceBuilder)
{
    _HistogramView = 0;
    _Histogram = 0;
}

TKinokoHistogram2dViewMessenger::~TKinokoHistogram2dViewMessenger()
{
}

TParaObjectPrototype* TKinokoHistogram2dViewMessenger::Clone(void)
{
    return new TKinokoHistogram2dViewMessenger(_ViewBuilder, _AnalysisSequenceBuilder);
}

void TKinokoHistogram2dViewMessenger::Construct(const string& ClassName, vector<TParaValue*>& ArgumentList) throw(TScriptException)
{
    if (ArgumentList.size() < 7) {
	throw TScriptException("Histogram2d::Histogram2d(): too few argument[s]");
    }
    if (
	(! ArgumentList[1]->IsLong()) ||
	(! ArgumentList[2]->IsNumeric()) ||
	(! ArgumentList[3]->IsNumeric()) ||
	(! ArgumentList[4]->IsLong()) ||
	(! ArgumentList[5]->IsNumeric()) ||
	(! ArgumentList[6]->IsNumeric())
    ){
	throw TScriptException("Histogram2d::Histogram2d(): invalid argument");
    }

    _Name = this->ObjectName();
    string Title = ArgumentList[0]->AsString();
    int NumberOfXBins = ArgumentList[1]->AsLong();
    double XMin = ArgumentList[2]->AsDouble();
    double XMax = ArgumentList[3]->AsDouble();
    int NumberOfYBins = ArgumentList[4]->AsLong();
    double YMin = ArgumentList[5]->AsDouble();
    double YMax = ArgumentList[6]->AsDouble();

    if (
	(NumberOfXBins <= 0) || (XMax <= XMin) || 
	(NumberOfYBins <= 0) || (YMax <= YMin)
    ){
	throw TScriptException(
	    "Histogram2d::Histogram2d(): incorrect bin range"
	);
    }
    
    _Histogram = new TKameHistogram2d(
	TKameHistogramScale(NumberOfXBins, XMin, XMax),
	TKameHistogramScale(NumberOfYBins, YMin, YMax)
    );
    _Histogram->SetTitle(Title);

    _HistogramView = _ViewFactory->CreateHistogram2dView(_Histogram);
    _HistogramView->SetName(_Name);

    _ViewBuilder->AddView(_HistogramView);
}

int TKinokoHistogram2dViewMessenger::InvokeMethodByName(const string& MethodName, vector<TParaValue*>& ArgumentList, TParaValue& ReturnValue) throw(TScriptException)
{
    if (MethodName == "setAxisTitle") {
	return SetAxisTitle(ArgumentList, ReturnValue);
    }
    else if (MethodName == "disableZScale") {
	return DisableZScale(ArgumentList, ReturnValue);
    }
    else if ((MethodName == "setZScaleLog") || (MethodName == "setLogZ")) {
	return SetZScaleLog(ArgumentList, ReturnValue);
    }
    else if ((MethodName == "setZScaleLinear") || (MethodName == "setLinearZ")) {
	return SetZScaleLinear(ArgumentList, ReturnValue);
    }

    return TKinokoViewMessenger::InvokeMethodByName(MethodName, ArgumentList, ReturnValue);
}

TKinokoView* TKinokoHistogram2dViewMessenger::View(void)
{
    return _HistogramView;
}

TKinokoDataAnalyzer* TKinokoHistogram2dViewMessenger::CreateAnalysisAction(vector<TKinokoDataElementMessenger*> DataElementMessengerList)
{
    TKinokoHistogram2dWriter* AnalysisAction = new TKinokoHistogram2dWriter(
	_Histogram,
	DataElementMessengerList[0]->SectionPath(),
	DataElementMessengerList[0]->DataElementSpecifier(),
	DataElementMessengerList[1]->SectionPath(),
	DataElementMessengerList[1]->DataElementSpecifier()
    );
    AnalysisAction->SetTargetName(_Name);

    return AnalysisAction;
}

int TKinokoHistogram2dViewMessenger::SetAxisTitle(vector<TParaValue*>& ArgumentList, TParaValue& ReturnValue) throw(TScriptException)
{
    string XTitle, YTitle;
    if (ArgumentList.size() > 0) {
	XTitle = ArgumentList[0]->AsString();
    }
    if (ArgumentList.size() > 1) {
	YTitle = ArgumentList[1]->AsString();
    }

    _HistogramView->SetAxisTitle(XTitle, YTitle);

    return 1;
}

int TKinokoHistogram2dViewMessenger::DisableZScale(vector<TParaValue*>& ArgumentList, TParaValue& ReturnValue) throw(TScriptException)
{
    _HistogramView->DisableZScale();

    return 1;
}

int TKinokoHistogram2dViewMessenger::SetZScaleLog(vector<TParaValue*>& ArgumentList, TParaValue& ReturnValue) throw(TScriptException)
{
    _HistogramView->SetZScaleLog();
    return 1;
}

int TKinokoHistogram2dViewMessenger::SetZScaleLinear(vector<TParaValue*>& ArgumentList, TParaValue& ReturnValue) throw(TScriptException)
{
    _HistogramView->SetZScaleLinear();
    return 1;
}



TKinokoTrendViewMessenger::TKinokoTrendViewMessenger(TKinokoViewBuilder* ViewBuilder, TKinokoAnalysisSequenceBuilder* AnalysisSequenceBuilder)
: TKinokoViewMessenger(ViewBuilder, AnalysisSequenceBuilder)
{
    _TrendView = 0;
    _Trend = 0;
}

TKinokoTrendViewMessenger::~TKinokoTrendViewMessenger()
{
}

TParaObjectPrototype* TKinokoTrendViewMessenger::Clone(void)
{
    return new TKinokoTrendViewMessenger(_ViewBuilder, _AnalysisSequenceBuilder);
}

void TKinokoTrendViewMessenger::Construct(const string& ClassName, vector<TParaValue*>& ArgumentList) throw(TScriptException)
{
    if (ArgumentList.size() < 4) {
	throw TScriptException(
	    "Trend::Trend(string title, double range_min, double range_max, long window_length, long tick_interval=1): too few argument[s]");
    }
    if (
	(! ArgumentList[1]->IsNumeric()) ||
	(! ArgumentList[2]->IsNumeric()) ||
	(! ArgumentList[3]->IsNumeric())
    ){
	throw TScriptException(
	    "Trend::Trend(string title, double range_min, double range_max, long window_length, long tick_interval=1): invalid argument[s]"
	);
    }

    _Name = this->ObjectName();
    string Title = ArgumentList[0]->AsString();
    double MinValue = ArgumentList[1]->AsDouble();
    double MaxValue = ArgumentList[2]->AsDouble();
    long WindowLength = ArgumentList[3]->AsLong();

    long TickInterval = 1;
    if (ArgumentList.size() > 4) {
	if (! ArgumentList[4]->IsLong()) {
	    throw TScriptException(
		"Trend::Trend(string title, double range_min, double rabge_max, long window_length, long tick_interval=1): invalid argument[s]"
	    );
	}
	TickInterval = ArgumentList[4]->AsLong();
    }

    _Trend = new TKameTrend(TickInterval, WindowLength);
    _Trend->SetTitle(Title);

    _TrendView = _ViewFactory->CreateTrendView(
	_Trend, MinValue, MaxValue
    );
    _TrendView->SetName(_Name);

    _ViewBuilder->AddView(_TrendView);
}

int TKinokoTrendViewMessenger::InvokeMethodByName(const string& MethodName, vector<TParaValue*>& ArgumentList, TParaValue& ReturnValue) throw(TScriptException)
{
    if (MethodName == "drawCounts") {
	return DrawCounts(ArgumentList, ReturnValue);
    }
    else if (MethodName == "drawSum") {
	return DrawSum(ArgumentList, ReturnValue);
    }
    else if (MethodName == "drawMean") {
	return DrawMean(ArgumentList, ReturnValue);
    }
    else if ((MethodName == "drawDeviation") || (MethodName == "drawRms")) {
	return DrawDeviation(ArgumentList, ReturnValue);
    }
    else if (MethodName == "setDisplayStatistics") {
	return SetDisplayStatistics(ArgumentList, ReturnValue);
    }
    else if ((MethodName == "setYScaleLog") || (MethodName == "setLogY")) {
	return SetYScaleLog(ArgumentList, ReturnValue);
    }
    else if ((MethodName == "setYScaleLinear") || (MethodName == "setLinearY")) {
	return SetYScaleLinear(ArgumentList, ReturnValue);
    }
    else if (MethodName == "setTimeTick") {
	return SetTimeTick(ArgumentList, ReturnValue);
    }
    else if (MethodName == "setAxisTitle") {
	return SetAxisTitle(ArgumentList, ReturnValue);
    }
    else if (MethodName == "setOperationRange") {
	return SetOperationRange(ArgumentList, ReturnValue);
    }
    else if (MethodName == "enableAlarm") {
	return EnableAlarm(ArgumentList, ReturnValue);
    }
    else if (MethodName == "disableAlarm") {
	return DisableAlarm(ArgumentList, ReturnValue);
    }
    else if ((MethodName == "fillOne") || (MethodName == "FillOne")) {
	throw TScriptException(
	    "Trend::fillOne()",
	    "interface not available (use fill() instead)"
	);
    }

    return TKinokoViewMessenger::InvokeMethodByName(MethodName, ArgumentList, ReturnValue);
}

TKinokoView* TKinokoTrendViewMessenger::View(void)
{
    return _TrendView;
}

TKinokoDataAnalyzer* TKinokoTrendViewMessenger::CreateAnalysisAction(vector<TKinokoDataElementMessenger*> DataElementMessengerList)
{
    TKinokoTrendWriter* AnalysisAction = new TKinokoTrendWriter(
	_Trend,
	DataElementMessengerList[0]->SectionPath(),
	DataElementMessengerList[0]->DataElementSpecifier()
    );
    AnalysisAction->SetTargetName(_Name);

    return AnalysisAction;
}

int TKinokoTrendViewMessenger::DrawCounts(std::vector<TParaValue*>& ArgumentList, TParaValue& ReturnValue) throw(TScriptException)
{
    try {
	_ViewBuilder->AddAction(new TKinokoTrendViewDrawAction(
            _TrendView, TKinokoTrendView::FieldId_Counts
	));
    }
    catch (TKinokoException &e) {
	throw TScriptException(e.Message());
    }

    return 1;
}

int TKinokoTrendViewMessenger::DrawSum(std::vector<TParaValue*>& ArgumentList, TParaValue& ReturnValue) throw(TScriptException)
{
    try {
	_ViewBuilder->AddAction(new TKinokoTrendViewDrawAction(
            _TrendView, TKinokoTrendView::FieldId_Sum
	));
    }
    catch (TKinokoException &e) {
	throw TScriptException(e.Message());
    }

    return 1;
}

int TKinokoTrendViewMessenger::DrawMean(std::vector<TParaValue*>& ArgumentList, TParaValue& ReturnValue) throw(TScriptException)
{
    try {
	_ViewBuilder->AddAction(new TKinokoTrendViewDrawAction(
            _TrendView, TKinokoTrendView::FieldId_Mean
	));
    }
    catch (TKinokoException &e) {
	throw TScriptException(e.Message());
    }

    return 1;
}

int TKinokoTrendViewMessenger::DrawDeviation(std::vector<TParaValue*>& ArgumentList, TParaValue& ReturnValue) throw(TScriptException)
{
    try {
	_ViewBuilder->AddAction(new TKinokoTrendViewDrawAction(
            _TrendView, TKinokoTrendView::FieldId_Deviation
	));
    }
    catch (TKinokoException &e) {
	throw TScriptException(e.Message());
    }

    return 1;
}

int TKinokoTrendViewMessenger::SetDisplayStatistics(std::vector<TParaValue*>& ArgumentList, TParaValue& ReturnValue) throw(TScriptException)
{
    int StatisticsFlag = 0;

    for (unsigned i = 0; i < ArgumentList.size(); i++) {
	string StatName = ArgumentList[i]->AsString();
	if (StatName == "counts") {
	    StatisticsFlag |= TKinokoTrendView::Stat_Counts;
	}
	if (StatName == "sum") {
	    StatisticsFlag |= TKinokoTrendView::Stat_Sum;
	}
	if (StatName == "mean") {
	    StatisticsFlag |= TKinokoTrendView::Stat_Mean;
	}
	if ((StatName == "deviation") || (StatName == "rms")) {
	    StatisticsFlag |= TKinokoTrendView::Stat_Deviation;
	}
    }

    _TrendView->SetDisplayStatistics(StatisticsFlag);

    return 1;
}

int  TKinokoTrendViewMessenger::SetYScaleLog(vector<TParaValue*>& ArgumentList, TParaValue& ReturnValue) throw(TScriptException)
{
    _TrendView->SetYScaleLog();
    return 1;
}

int  TKinokoTrendViewMessenger::SetYScaleLinear(vector<TParaValue*>& ArgumentList, TParaValue& ReturnValue) throw(TScriptException)
{
    _TrendView->SetYScaleLinear();
    return 1;
}

int  TKinokoTrendViewMessenger::SetTimeTick(vector<TParaValue*>& ArgumentList, TParaValue& ReturnValue) throw(TScriptException)
{
    string Format;   // "%H:%M", "%m %y" etc, empty for auto
    string Unit;     // "sec", "day", etc, "-" for auto
    long Step = 0;   // zero for auto

    if (ArgumentList.size() >= 1) {
	Format = ArgumentList[0]->AsString();
    }
    if (ArgumentList.size() >= 2) {
	Unit = ArgumentList[1]->AsString();
    }
    if (ArgumentList.size() >= 3) {
	Step = ArgumentList[2]->AsLong();
    }

    if (Unit.empty()) {
	Unit = "-";
    }

    _TrendView->SetTimeTick(Format, Unit, Step);

    return 1;
}

int TKinokoTrendViewMessenger::SetAxisTitle(vector<TParaValue*>& ArgumentList, TParaValue& ReturnValue) throw(TScriptException)
{
    string XTitle, YTitle;
    if (ArgumentList.size() > 0) {
	XTitle = ArgumentList[0]->AsString();
    }
    if (ArgumentList.size() > 1) {
	YTitle = ArgumentList[1]->AsString();
    }

    _TrendView->SetAxisTitle(XTitle, YTitle);

    return 1;
}

int TKinokoTrendViewMessenger::SetOperationRange(std::vector<TParaValue*>& ArgumentList, TParaValue& ReturnValue) throw(TScriptException)
{
    if (
	(ArgumentList.size() < 2) ||
	(! ArgumentList[0]->IsNumeric()) ||
	(! ArgumentList[1]->IsNumeric())
    ){
	throw TScriptException(
		"Trend::setOperationRange(float lower_bound, float upper_bound, int number_of_points_to_judge=2)"
	    "invalid argument[s]"
	);
    }
    double LowerBound = ArgumentList[0]->AsDouble();
    double UpperBound = ArgumentList[1]->AsDouble();

    int NumberOfPointsToJudge = 2;
    if (ArgumentList.size() >= 3) {
	if (! ArgumentList[2]->IsLong()) {
	    throw TScriptException(
		"Trend::setOperationRange(float lower_bound, float upper_bound, int number_of_points_to_judge)"
		"invalid argument[s]"
	    );
	}
    }

    try {
	_ViewBuilder->AddAction(
	    new TKinokoTrendViewSetOperationRangeAction(
		_TrendView, LowerBound, UpperBound, NumberOfPointsToJudge
	    )
	);
    }
    catch (TKinokoException &e) {
	throw TScriptException(e.Message());
    }

    return 1;
}

int TKinokoTrendViewMessenger::EnableAlarm(std::vector<TParaValue*>& ArgumentList, TParaValue& ReturnValue) throw(TScriptException)
{
    string Message;
    if (ArgumentList.size() > 0) {
	Message = ArgumentList[0]->AsString();
    }

    try {
	_ViewBuilder->AddAction(
	    new TKinokoTrendViewEnableAlarmAction(
		_TrendView, _ViewBuilder->EventEmitter(), Message
	    )
	);
    }
    catch (TKinokoException &e) {
	throw TScriptException(e.Message());
    }

    return 1;
}

int TKinokoTrendViewMessenger::DisableAlarm(std::vector<TParaValue*>& ArgumentList, TParaValue& ReturnValue) throw(TScriptException)
{
    try {
	_ViewBuilder->AddAction(
	    new TKinokoTrendViewDisableAlarmAction(_TrendView)
	);
    }
    catch (TKinokoException &e) {
	throw TScriptException(e.Message());
    }

    return 1;
}



#if 1
//... for backward compatibility ...//
TKinokoHistoryViewMessenger::TKinokoHistoryViewMessenger(TKinokoViewBuilder* ViewBuilder, TKinokoAnalysisSequenceBuilder* AnalysisSequenceBuilder)
: TKinokoViewMessenger(ViewBuilder, AnalysisSequenceBuilder)
{
    _HistoryView = 0;
    _History = 0;
}

TKinokoHistoryViewMessenger::~TKinokoHistoryViewMessenger()
{
}

TParaObjectPrototype* TKinokoHistoryViewMessenger::Clone(void)
{
    return new TKinokoHistoryViewMessenger(_ViewBuilder, _AnalysisSequenceBuilder);
}

void TKinokoHistoryViewMessenger::Construct(const string& ClassName, vector<TParaValue*>& ArgumentList) throw(TScriptException)
{
#if 0
    static bool IsFirst = true;
    if (IsFirst) {
	cerr << "WARNING: History is obsolete; use Trend instead." << endl;
	IsFirst = false;
    }
#endif

    if (ArgumentList.size() < 4) {
	throw TScriptException("History::History(): too few argument[s]");
    }
    if (
	(! ArgumentList[1]->IsNumeric()) ||
	(! ArgumentList[2]->IsNumeric()) ||
	(! ArgumentList[3]->IsLong())
    ){
	throw TScriptException("History::History(): invalid argument");
    }

    _Name = this->ObjectName();
    string Title = ArgumentList[0]->AsString();
    double MinValue = ArgumentList[1]->AsDouble();
    double MaxValue = ArgumentList[2]->AsDouble();
    int NumberOfSamples = ArgumentList[3]->AsLong();

    double WindowWidth = NumberOfSamples;
    if (ArgumentList.size() > 4) {
        if (! ArgumentList[4]->IsNumeric()) {
	    throw TScriptException(
	        "History::History(string name, double range_min, double range_max, int number_of_samples, double window_width): invalid argument"
	    );
	}
	WindowWidth = ArgumentList[4]->AsLong();
    }

    _History = new TKameHistory(NumberOfSamples);
    _History->SetTitle(Title);

    _HistoryView = _ViewFactory->CreateHistoryView(
	_History, MinValue, MaxValue, WindowWidth
    );
    _HistoryView->SetName(_Name);

    _ViewBuilder->AddView(_HistoryView);

#if 1
    //... for backward compatibility ...//
    if (ClassName == "MeanValueHistory") {
        _HistoryView->SetDisplayValue(TKinokoHistoryView::FieldId_Mean);
    }
    else if (ClassName == "TotalValueHistory") {
        _HistoryView->SetDisplayValue(TKinokoHistoryView::FieldId_Sum);
    }
    else if (ClassName == "EntryCountHistory") {
        _HistoryView->SetDisplayValue(TKinokoHistoryView::FieldId_Counts);
    }
#endif
}

int TKinokoHistoryViewMessenger::InvokeMethodByName(const string& MethodName, vector<TParaValue*>& ArgumentList, TParaValue& ReturnValue) throw(TScriptException)
{
    if ((MethodName == "drawCounts") || (MethodName == "drawCounts")) {
	return DrawCounts(ArgumentList, ReturnValue);
    }
    else if ((MethodName == "drawSum") || (MethodName == "drawSum")) {
	return DrawSum(ArgumentList, ReturnValue);
    }
    else if ((MethodName == "drawMean") || (MethodName == "drawMean")) {
	return DrawMean(ArgumentList, ReturnValue);
    }
    else if ((MethodName == "drawDeviation") || (MethodName == "drawDeviation")) {
	return DrawDeviation(ArgumentList, ReturnValue);
    }
    else if (MethodName == "setDisplayStatistics") {
	return SetDisplayStatistics(ArgumentList, ReturnValue);
    }
    else if ((MethodName == "setYScaleLog") || (MethodName == "setLogY")) {
	return SetYScaleLog(ArgumentList, ReturnValue);
    }
    else if ((MethodName == "setYScaleLinear") || (MethodName == "setLinearY")) {
	return SetYScaleLinear(ArgumentList, ReturnValue);
    }
    else if (MethodName == "setAxisTitle") {
	return SetAxisTitle(ArgumentList, ReturnValue);
    }
    else if (MethodName == "hold") {
	return Hold(ArgumentList, ReturnValue);
    }
    else if (MethodName == "setOperationRange") {
	return SetOperationRange(ArgumentList, ReturnValue);
    }
    else if (MethodName == "enableAlarm") {
	return EnableAlarm(ArgumentList, ReturnValue);
    }
    else if (MethodName == "disableAlarm") {
	return DisableAlarm(ArgumentList, ReturnValue);
    }
    else if ((MethodName == "fillOne") || (MethodName == "FillOne")) {
	throw TScriptException(
	    "History::fillOne()",
	    "interface not available (use fill() instead)"
	);
    }

    return TKinokoViewMessenger::InvokeMethodByName(MethodName, ArgumentList, ReturnValue);
}

TKinokoView* TKinokoHistoryViewMessenger::View(void)
{
    return _HistoryView;
}

TKinokoDataAnalyzer* TKinokoHistoryViewMessenger::CreateAnalysisAction(vector<TKinokoDataElementMessenger*> DataElementMessengerList)
{
    TKinokoHistoryWriter* AnalysisAction = new TKinokoHistoryWriter(
	_History,
	DataElementMessengerList[0]->SectionPath(),
	DataElementMessengerList[0]->DataElementSpecifier()
    );
    AnalysisAction->SetTargetName(_Name);

    return AnalysisAction;
}

int TKinokoHistoryViewMessenger::DrawCounts(std::vector<TParaValue*>& ArgumentList, TParaValue& ReturnValue) throw(TScriptException)
{
    try {
	_ViewBuilder->AddAction(new TKinokoHistoryViewDrawAction(
            _HistoryView, TKinokoHistoryView::FieldId_Counts
	));
    }
    catch (TKinokoException &e) {
	throw TScriptException(e.Message());
    }

    return 1;
}

int TKinokoHistoryViewMessenger::DrawSum(std::vector<TParaValue*>& ArgumentList, TParaValue& ReturnValue) throw(TScriptException)
{
    try {
	_ViewBuilder->AddAction(new TKinokoHistoryViewDrawAction(
            _HistoryView, TKinokoHistoryView::FieldId_Sum
	));
    }
    catch (TKinokoException &e) {
	throw TScriptException(e.Message());
    }

    return 1;
}

int TKinokoHistoryViewMessenger::DrawMean(std::vector<TParaValue*>& ArgumentList, TParaValue& ReturnValue) throw(TScriptException)
{
    try {
	_ViewBuilder->AddAction(new TKinokoHistoryViewDrawAction(
            _HistoryView, TKinokoHistoryView::FieldId_Mean
	));
    }
    catch (TKinokoException &e) {
	throw TScriptException(e.Message());
    }

    return 1;
}

int TKinokoHistoryViewMessenger::DrawDeviation(std::vector<TParaValue*>& ArgumentList, TParaValue& ReturnValue) throw(TScriptException)
{
    try {
	_ViewBuilder->AddAction(new TKinokoHistoryViewDrawAction(
            _HistoryView, TKinokoHistoryView::FieldId_Deviation
	));
    }
    catch (TKinokoException &e) {
	throw TScriptException(e.Message());
    }

    return 1;
}

int TKinokoHistoryViewMessenger::SetDisplayStatistics(std::vector<TParaValue*>& ArgumentList, TParaValue& ReturnValue) throw(TScriptException)
{
    int StatisticsFlag = 0;

    for (unsigned i = 0; i < ArgumentList.size(); i++) {
	string StatName = ArgumentList[i]->AsString();
	if (StatName == "counts") {
	    StatisticsFlag |= TKinokoHistoryView::Stat_Counts;
	}
	if (StatName == "sum") {
	    StatisticsFlag |= TKinokoHistoryView::Stat_Sum;
	}
	if (StatName == "mean") {
	    StatisticsFlag |= TKinokoHistoryView::Stat_Mean;
	}
	if ((StatName == "deviation") || (StatName == "rms")) {
	    StatisticsFlag |= TKinokoHistoryView::Stat_Deviation;
	}
    }

    _HistoryView->SetDisplayStatistics(StatisticsFlag);

    return 1;
}

int  TKinokoHistoryViewMessenger::SetYScaleLog(vector<TParaValue*>& ArgumentList, TParaValue& ReturnValue) throw(TScriptException)
{
    _HistoryView->SetYScaleLog();
    return 1;
}

int  TKinokoHistoryViewMessenger::SetYScaleLinear(vector<TParaValue*>& ArgumentList, TParaValue& ReturnValue) throw(TScriptException)
{
    _HistoryView->SetYScaleLinear();
    return 1;
}

int TKinokoHistoryViewMessenger::SetAxisTitle(vector<TParaValue*>& ArgumentList, TParaValue& ReturnValue) throw(TScriptException)
{
    string XTitle, YTitle;
    if (ArgumentList.size() > 0) {
	XTitle = ArgumentList[0]->AsString();
    }
    if (ArgumentList.size() > 1) {
	YTitle = ArgumentList[1]->AsString();
    }

    _HistoryView->SetAxisTitle(XTitle, YTitle);

    return 1;
}

int TKinokoHistoryViewMessenger::Hold(std::vector<TParaValue*>& ArgumentList, TParaValue& ReturnValue) throw(TScriptException)
{
    try {
        _ViewBuilder->AddAction(
	    new TKinokoHistoryViewHoldAction(_HistoryView, _History)
	);
    }
    catch (TKinokoException &e) {
	throw TScriptException(e.Message());
    }

    _HistoryView->DisableAutoHold();

    return 1;
}

int TKinokoHistoryViewMessenger::SetOperationRange(std::vector<TParaValue*>& ArgumentList, TParaValue& ReturnValue) throw(TScriptException)
{
    if (
	(ArgumentList.size() < 2) ||
	(! ArgumentList[0]->IsNumeric()) ||
	(! ArgumentList[1]->IsNumeric())
    ){
	throw TScriptException(
	    "History::setOperationRange(float lower_bound, float upper_bound)"
	    "invalid argument[s]"
	);
    }

    double LowerBound = ArgumentList[0]->AsDouble();
    double UpperBound = ArgumentList[1]->AsDouble();

    try {
	_ViewBuilder->AddAction(
	    new TKinokoHistoryViewSetOperationRangeAction(
		_HistoryView, LowerBound, UpperBound
	    )
	);
    }
    catch (TKinokoException &e) {
	throw TScriptException(e.Message());
    }

    return 1;
}

int TKinokoHistoryViewMessenger::EnableAlarm(std::vector<TParaValue*>& ArgumentList, TParaValue& ReturnValue) throw(TScriptException)
{
    string Message;
    if (ArgumentList.size() > 0) {
	Message = ArgumentList[0]->AsString();
    }

    try {
	_ViewBuilder->AddAction(
	    new TKinokoHistoryViewEnableAlarmAction(
		_HistoryView, _ViewBuilder->EventEmitter(), Message
	    )
	);
    }
    catch (TKinokoException &e) {
	throw TScriptException(e.Message());
    }

    return 1;
}

int TKinokoHistoryViewMessenger::DisableAlarm(std::vector<TParaValue*>& ArgumentList, TParaValue& ReturnValue) throw(TScriptException)
{
    try {
	_ViewBuilder->AddAction(
	    new TKinokoHistoryViewDisableAlarmAction(_HistoryView)
	);
    }
    catch (TKinokoException &e) {
	throw TScriptException(e.Message());
    }

    return 1;
}
#endif


TKinokoWaveViewMessenger::TKinokoWaveViewMessenger(TKinokoViewBuilder* ViewBuilder, TKinokoAnalysisSequenceBuilder* AnalysisSequenceBuilder)
: TKinokoViewMessenger(ViewBuilder, AnalysisSequenceBuilder)
{
    _WaveView = 0;
    _Ntuple = 0;
}

TKinokoWaveViewMessenger::~TKinokoWaveViewMessenger()
{
}

TParaObjectPrototype* TKinokoWaveViewMessenger::Clone(void)
{
    return new TKinokoWaveViewMessenger(_ViewBuilder, _AnalysisSequenceBuilder);
}

void TKinokoWaveViewMessenger::Construct(const string& ClassName, vector<TParaValue*>& ArgumentList) throw(TScriptException)
{
    if (ArgumentList.size() < 5) {
	throw TScriptException("Wave::Wave(): too few argument[s]");
    }
    if (
	(! ArgumentList[1]->IsNumeric()) ||
	(! ArgumentList[2]->IsNumeric()) ||
	(! ArgumentList[3]->IsNumeric()) ||
	(! ArgumentList[4]->IsNumeric())
    ){
	throw TScriptException("Wave::Wave(): invalid argument");
    }

    _Name = this->ObjectName();
    string Title = ArgumentList[0]->AsString();
    double XMin = ArgumentList[1]->AsDouble();
    double XMax = ArgumentList[2]->AsDouble();
    double YMin = ArgumentList[3]->AsDouble();
    double YMax = ArgumentList[4]->AsDouble();
    
    _Ntuple = new TKameNtuple();
    _Ntuple->SetTitle(Title);

    _WaveView = _ViewFactory->CreateWaveView(_Ntuple, XMin, XMax, YMin, YMax);
    _WaveView->SetName(_Name);

    _ViewBuilder->AddView(_WaveView);
}

int TKinokoWaveViewMessenger::InvokeMethodByName(const string& MethodName, vector<TParaValue*>& ArgumentList, TParaValue& ReturnValue) throw(TScriptException)
{
    if (MethodName == "setAxisTitle") {
	return SetAxisTitle(ArgumentList, ReturnValue);
    }

    return TKinokoViewMessenger::InvokeMethodByName(MethodName, ArgumentList, ReturnValue);
}

TKinokoView* TKinokoWaveViewMessenger::View(void)
{
    return _WaveView;
}

TKinokoDataAnalyzer* TKinokoWaveViewMessenger::CreateAnalysisAction(vector<TKinokoDataElementMessenger*> DataElementMessengerList)
{
    TKinokoWaveWriter* AnalysisAction = new TKinokoWaveWriter(
	_Ntuple,
	DataElementMessengerList[0]->SectionPath(),
	DataElementMessengerList[0]->DataElementSpecifier()
    );
    AnalysisAction->SetTargetName(_Name);

    return AnalysisAction;
}

int TKinokoWaveViewMessenger::SetAxisTitle(vector<TParaValue*>& ArgumentList, TParaValue& ReturnValue) throw(TScriptException)
{
    string XTitle, YTitle;
    if (ArgumentList.size() > 0) {
	XTitle = ArgumentList[0]->AsString();
    }
    if (ArgumentList.size() > 1) {
	YTitle = ArgumentList[1]->AsString();
    }

    _WaveView->SetAxisTitle(XTitle, YTitle);

    return 1;
}



TKinokoMapViewMessenger::TKinokoMapViewMessenger(TKinokoViewBuilder* ViewBuilder, TKinokoAnalysisSequenceBuilder* AnalysisSequenceBuilder)
: TKinokoViewMessenger(ViewBuilder, AnalysisSequenceBuilder)
{
    _MapView = 0;
    _Ntuple = 0;
}

TKinokoMapViewMessenger::~TKinokoMapViewMessenger()
{
}

TParaObjectPrototype* TKinokoMapViewMessenger::Clone(void)
{
    return new TKinokoMapViewMessenger(_ViewBuilder, _AnalysisSequenceBuilder);
}

void TKinokoMapViewMessenger::Construct(const string& ClassName, vector<TParaValue*>& ArgumentList) throw(TScriptException)
{
    if (ArgumentList.size() < 7) {
	throw TScriptException("Map::Map(): too few argument[s]");
    }
    if (
	(! ArgumentList[0]->IsString()) ||
	(! ArgumentList[1]->IsNumeric()) ||
	(! ArgumentList[2]->IsNumeric()) ||
	(! ArgumentList[3]->IsNumeric()) ||
	(! ArgumentList[4]->IsNumeric()) ||
	(! ArgumentList[5]->IsNumeric()) ||
	(! ArgumentList[6]->IsNumeric())
    ){
	throw TScriptException("Map::Map(): invalid argument[s]");
    }

    _Name = this->ObjectName();
    string Title = ArgumentList[0]->AsString();
    double XMin = ArgumentList[1]->AsDouble();
    double XMax = ArgumentList[2]->AsDouble();
    double YMin = ArgumentList[3]->AsDouble();
    double YMax = ArgumentList[4]->AsDouble();
    double ZMin = ArgumentList[5]->AsDouble();
    double ZMax = ArgumentList[6]->AsDouble();
    
    _Ntuple = new TKameNtuple();
    _Ntuple->SetTitle(Title);

    _MapView = _ViewFactory->CreateMapView(
	_Ntuple, XMin, XMax, YMin, YMax, ZMin, ZMax
    );
    _MapView->SetName(_Name);

    _ViewBuilder->AddView(_MapView);
}

int TKinokoMapViewMessenger::InvokeMethodByName(const string& MethodName, vector<TParaValue*>& ArgumentList, TParaValue& ReturnValue) throw(TScriptException)
{
    if ((MethodName == "addPoint") || (MethodName == "AddPoint")) {
	return AddPoint(ArgumentList, ReturnValue);
    }
    else if ((MethodName == "setPointSize") || (MethodName == "SetPointSize")) {
	return SetPointSize(ArgumentList, ReturnValue);
    }
    else if ((MethodName == "fill") || (MethodName == "Fill")) {
	throw TScriptException(
	    "Map::fill()",
	    "interface not available (use fillOne() instead)"
	);
    }
    else {
	return TKinokoViewMessenger::InvokeMethodByName(MethodName, ArgumentList, ReturnValue);
    }
}

TKinokoView* TKinokoMapViewMessenger::View(void)
{
    return _MapView;
}

TKinokoDataAnalyzer* TKinokoMapViewMessenger::CreateAnalysisAction(vector<TKinokoDataElementMessenger*> DataElementMessengerList)
{
    TKinokoMapWriter* AnalysisAction = new TKinokoMapWriter(
	_Ntuple,
	DataElementMessengerList[0]->SectionPath(),
	DataElementMessengerList[0]->DataElementSpecifier()
    );
    AnalysisAction->SetTargetName(_Name);

    return AnalysisAction;
}

int TKinokoMapViewMessenger::AddPoint(vector<TParaValue*>& ArgumentList, TParaValue& ReturnValue) throw(TScriptException)
{
    if (ArgumentList.size() < 3) {
	throw TScriptException("Map::addPoint(): too few argument[s]");
    }
    if (
	(! ArgumentList[0]->IsLong()) || 
	(! ArgumentList[1]->IsNumeric()) ||
	(! ArgumentList[2]->IsNumeric())
    ){
	throw TScriptException("Map::addPoint(): invalid argument");
    }

    long Address = ArgumentList[0]->AsLong();
    double PositionX = ArgumentList[1]->AsDouble();
    double PositionY = ArgumentList[2]->AsDouble();

    _MapView->AddPoint(Address, PositionX, PositionY);

    return 1;
}

int TKinokoMapViewMessenger::SetPointSize(vector<TParaValue*>& ArgumentList, TParaValue& ReturnValue) throw(TScriptException)
{
    if (ArgumentList.size() < 1) {
	throw TScriptException("Map::setPointSize(): too few argument[s]");
    }
    if (! ArgumentList[0]->IsNumeric()) {
	throw TScriptException("Map::setPointSize(): invalid argument");
    }

    double PointSize = ArgumentList[0]->AsDouble();
    _MapView->SetPointSize(PointSize);

    return 1;
}



TKinokoTabularViewMessenger::TKinokoTabularViewMessenger(TKinokoViewBuilder* ViewBuilder, TKinokoAnalysisSequenceBuilder* AnalysisSequenceBuilder)
: TKinokoViewMessenger(ViewBuilder, AnalysisSequenceBuilder)
{
    _TabularView = 0;
    _Ntuple = 0;
}

TKinokoTabularViewMessenger::~TKinokoTabularViewMessenger()
{
}

TParaObjectPrototype* TKinokoTabularViewMessenger::Clone(void)
{
    return new TKinokoTabularViewMessenger(_ViewBuilder, _AnalysisSequenceBuilder);
}

void TKinokoTabularViewMessenger::Construct(const string& ClassName, vector<TParaValue*>& ArgumentList) throw(TScriptException)
{
    if (ArgumentList.size() < 1) {
	throw TScriptException("Tabular::Tabular(): too few argument[s]");
    }

    _Name = this->ObjectName();
    string Title = ArgumentList[0]->AsString();

    int NumberOfColumns = 1;
    if (ArgumentList.size() >= 2) {
	if (! ArgumentList[1]->IsLong()) {
	    throw TScriptException("Tabular::Tabular(): invalid argument");
	}
	NumberOfColumns = ArgumentList[1]->AsLong();
    }

    _Ntuple = new TKameNtuple();
    _Ntuple->SetTitle(Title);

    _TabularView = _ViewFactory->CreateTabularView(_Ntuple, NumberOfColumns);
    _TabularView->SetName(_Name);

    _ViewBuilder->AddView(_TabularView);
}

int TKinokoTabularViewMessenger::InvokeMethodByName(const string& MethodName, vector<TParaValue*>& ArgumentList, TParaValue& ReturnValue) throw(TScriptException)
{
    if ((MethodName == "fill") || (MethodName == "Fill")) {
	throw TScriptException(
	    "Tabular::fill()",
	    "interface not available (use fillOne() instead)"
	);
    }

    return TKinokoViewMessenger::InvokeMethodByName(MethodName, ArgumentList, ReturnValue);
}

TKinokoView* TKinokoTabularViewMessenger::View(void)
{
    return _TabularView;
}

TKinokoDataAnalyzer* TKinokoTabularViewMessenger::CreateAnalysisAction(vector<TKinokoDataElementMessenger*> DataElementMessengerList)
{
    TKinokoTabularWriter* AnalysisAction = new TKinokoTabularWriter(
	_Ntuple,
	DataElementMessengerList[0]->SectionPath(),
	DataElementMessengerList[0]->DataElementSpecifier()
    );
    AnalysisAction->SetTargetName(_Name);

    return AnalysisAction;
}



TKinokoPictureViewMessenger::TKinokoPictureViewMessenger(TKinokoViewBuilder* ViewBuilder, TKinokoAnalysisSequenceBuilder* AnalysisSequenceBuilder)
: TKinokoViewMessenger(ViewBuilder, AnalysisSequenceBuilder)
{
    _PictureView = 0;
}

TKinokoPictureViewMessenger::~TKinokoPictureViewMessenger()
{
}

TParaObjectPrototype* TKinokoPictureViewMessenger::Clone(void)
{
    return new TKinokoPictureViewMessenger(_ViewBuilder, _AnalysisSequenceBuilder);
}

void TKinokoPictureViewMessenger::Construct(const string& ClassName, vector<TParaValue*>& ArgumentList) throw(TScriptException)
{
    _Name = this->ObjectName();

    string Title;
    double XMin = 0, XMax = 1, YMin = 0, YMax = 1;
    if (ArgumentList.size() == 0) {
	;
    }
    else if (ArgumentList.size() == 1) {
	if (! ArgumentList[0]->IsString()) {
	    throw TScriptException("Picture::Picture(): invalid argument[s]");
	}
	Title = ArgumentList[0]->AsString();
    }
    else if (ArgumentList.size() == 5) {
	if (
	    (! ArgumentList[1]->IsNumeric()) ||
	    (! ArgumentList[2]->IsNumeric()) ||
	    (! ArgumentList[3]->IsNumeric()) ||
	    (! ArgumentList[4]->IsNumeric())
	){
	    throw TScriptException("Picture::Picture(): invalid argument[s]");
	}
	XMin = ArgumentList[1]->AsDouble();
	XMax = ArgumentList[2]->AsDouble();
	YMin = ArgumentList[3]->AsDouble();
	YMax = ArgumentList[4]->AsDouble();
    }
    else {
	throw TScriptException(
	    "Picture::Picture(): invalid number of argument[s]"
	);
    }
    
    _PictureView = _ViewFactory->CreatePictureView(
	Title, XMin, XMax, YMin, YMax
    );
    _PictureView->SetName(_Name);

    _ViewBuilder->AddView(_PictureView);
}

int TKinokoPictureViewMessenger::InvokeMethodByName(const string& MethodName, vector<TParaValue*>& ArgumentList, TParaValue& ReturnValue) throw(TScriptException)
{
    if (
	(MethodName == "fill") || (MethodName == "Fill") ||
	(MethodName == "fillOne") || (MethodName == "FillOne")
    ){
	throw TScriptException(
	    "Picture::fill()", "interface not available"
	);
    }

    return TKinokoViewMessenger::InvokeMethodByName(MethodName, ArgumentList, ReturnValue);
}

TKinokoView* TKinokoPictureViewMessenger::View(void)
{
    return _PictureView;
}

TKinokoDataAnalyzer* TKinokoPictureViewMessenger::CreateAnalysisAction(vector<TKinokoDataElementMessenger*> DataElementMessengerList)
{
    return 0;
}



TKinokoSchematicViewMessenger::TKinokoSchematicViewMessenger(TKinokoViewBuilder* ViewBuilder, TKinokoAnalysisSequenceBuilder* AnalysisSequenceBuilder)
: TKinokoViewMessenger(ViewBuilder, AnalysisSequenceBuilder)
{
    _SchematicView = 0;
    _Ntuple = 0;
}

TKinokoSchematicViewMessenger::~TKinokoSchematicViewMessenger()
{
}

TParaObjectPrototype* TKinokoSchematicViewMessenger::Clone(void)
{
    return new TKinokoSchematicViewMessenger(
	_ViewBuilder, _AnalysisSequenceBuilder
    );
}

void TKinokoSchematicViewMessenger::Construct(const string& ClassName, vector<TParaValue*>& ArgumentList) throw(TScriptException)
{
    _Name = this->ObjectName();

    string Title;
    if (ArgumentList.size() >= 1) {
	Title = ArgumentList[0]->AsString();
    }

    float XMin = 0, XMax = 1, YMin = 0, YMax = 1;
    if (ArgumentList.size() >= 5) {
	if (
	    (! ArgumentList[1]->IsNumeric()) ||
	    (! ArgumentList[2]->IsNumeric()) ||
	    (! ArgumentList[3]->IsNumeric()) ||
	    (! ArgumentList[4]->IsNumeric())
	){
	    throw TScriptException(
		"Schematic::Schematic(string title, float x0, float y0, float x1, float y1): invalid argument[s]"
	    );
	}

	XMin = ArgumentList[1]->AsDouble();
	XMax = ArgumentList[2]->AsDouble();
	YMin = ArgumentList[3]->AsDouble();
	YMax = ArgumentList[4]->AsDouble();
    }
    
    _Ntuple = new TKameNtuple();
    _Ntuple->SetTitle(Title);

    _SchematicView = _ViewFactory->CreateSchematicView(
	_Ntuple, XMin, XMax, YMin, YMax
    );
    _SchematicView->SetName(_Name);

    _ViewBuilder->AddView(_SchematicView);
}

int TKinokoSchematicViewMessenger::InvokeMethodByName(const string& MethodName, vector<TParaValue*>& ArgumentList, TParaValue& ReturnValue) throw(TScriptException)
{
    if (MethodName == "setColorScale") {
	return SetColorScale(ArgumentList, ReturnValue);
    }
    else if (MethodName == "putTextItem") {
	return PutTextItem(ArgumentList, ReturnValue);
    }
    else if (MethodName == "putRectangleItem") {
	return PutRectangleItem(ArgumentList, ReturnValue);
    }
    else if (MethodName == "putCircleItem") {
	return PutCircleItem(ArgumentList, ReturnValue);
    }
    else if (MethodName == "fill") {
	throw TScriptException(
	    "Schematic::fill()",
	    "interface not available (use fillOne() instead)"
	);
    }
    else {
	return TKinokoViewMessenger::InvokeMethodByName(MethodName, ArgumentList, ReturnValue);
    }
}

TKinokoView* TKinokoSchematicViewMessenger::View(void)
{
    return _SchematicView;
}

TKinokoDataAnalyzer* TKinokoSchematicViewMessenger::CreateAnalysisAction(vector<TKinokoDataElementMessenger*> DataElementMessengerList)
{
    TKinokoMapWriter* AnalysisAction = new TKinokoMapWriter(
	_Ntuple,
	DataElementMessengerList[0]->SectionPath(),
	DataElementMessengerList[0]->DataElementSpecifier()
    );
    AnalysisAction->SetTargetName(_Name);

    return AnalysisAction;
}

int TKinokoSchematicViewMessenger::SetColorScale(vector<TParaValue*>& ArgumentList, TParaValue& ReturnValue) throw(TScriptException)
{
    if (ArgumentList.size() < 2) {
	throw TScriptException(
	    "Schematic::setColorScale(float min, float max, ...): too few argument[s]");
    }
    if (
	(! ArgumentList[0]->IsNumeric()) ||
	(! ArgumentList[1]->IsNumeric())
    ){
	throw TScriptException(
	    "Schematic::setColorScale(float min, float max, ...): invalid argument");
    }

    double Min = ArgumentList[0]->AsDouble();
    double Max = ArgumentList[1]->AsDouble();
    
    string Type;
    if (ArgumentList.size() >= 3) {
	Type = ArgumentList[2]->AsString();
    }
    else {
	Type = "rainbow";
    }

    _SchematicView->SetColorScale(Min, Max, Type);

    return 1;
}

int TKinokoSchematicViewMessenger::PutTextItem(vector<TParaValue*>& ArgumentList, TParaValue& ReturnValue) throw(TScriptException)
{
    if (ArgumentList.size() < 3) {
	throw TScriptException(
	    "Schematic::putTextItem(int address, float x0, float y0): too few argument[s]");
    }
    if (
	(! ArgumentList[0]->IsLong()) || 
	(! ArgumentList[1]->IsNumeric()) ||
	(! ArgumentList[2]->IsNumeric())
    ){
	throw TScriptException(
	    "Schematic::putTextItem(int address, float x0, float y0): invalid argument");
    }

    long Address = ArgumentList[0]->AsLong();
    double X0 = ArgumentList[1]->AsDouble();
    double Y0 = ArgumentList[2]->AsDouble();
    
    string Format = "";
    if (ArgumentList.size() >= 4) {
	Format = ArgumentList[3]->AsString();
    }

    _SchematicView->PutTextItem(Address, X0, Y0, Format);

    return 1;
}

int TKinokoSchematicViewMessenger::PutRectangleItem(vector<TParaValue*>& ArgumentList, TParaValue& ReturnValue) throw(TScriptException)
{
    if (ArgumentList.size() < 5) {
	throw TScriptException(
	    "Schematic::putRectangleItem(int address, float x0, float y0, float x1, float y1): too few argument[s]");
    }
    if (
	(! ArgumentList[0]->IsLong()) || 
	(! ArgumentList[1]->IsNumeric()) ||
	(! ArgumentList[2]->IsNumeric()) ||
	(! ArgumentList[3]->IsNumeric()) ||
	(! ArgumentList[4]->IsNumeric())
    ){
	throw TScriptException(
	    "Schematic::putRectangleItem(int address, float x0, float y0, float x1, float y1): invalid argument");
    }

    long Address = ArgumentList[0]->AsLong();
    double X0 = ArgumentList[1]->AsDouble();
    double Y0 = ArgumentList[2]->AsDouble();
    double X1 = ArgumentList[3]->AsDouble();
    double Y1 = ArgumentList[4]->AsDouble();

    _SchematicView->PutRectangleItem(Address, X0, Y0, X1, Y1);

    return 1;
}

int TKinokoSchematicViewMessenger::PutCircleItem(vector<TParaValue*>& ArgumentList, TParaValue& ReturnValue) throw(TScriptException)
{
    if (ArgumentList.size() < 4) {
	throw TScriptException(
	    "Schematic::putCircleItem(int address, float x, float y, float r): too few argument[s]");
    }
    if (
	(! ArgumentList[0]->IsLong()) || 
	(! ArgumentList[1]->IsNumeric()) ||
	(! ArgumentList[2]->IsNumeric()) ||
	(! ArgumentList[3]->IsNumeric())
    ){
	throw TScriptException(
	    "Schematic::putCircleItem(int address, float x, float y, float r): invalid argument");
    }

    long Address = ArgumentList[0]->AsLong();
    double X0 = ArgumentList[1]->AsDouble();
    double Y0 = ArgumentList[2]->AsDouble();
    double Radius = ArgumentList[3]->AsDouble();

    _SchematicView->PutCircleItem(Address, X0, Y0, Radius);

    return 1;
}
