/* kdfmerge.cc */
/* Created by Enomoto Sanshiro on 9 November 2001. */
/* Last updated by Enomoto Sanshiro on 26 November 2001. */


#include <cstdlib>
#include <iostream>
#include <string>
#include <vector>
#include "MushFile.hh"
#include "MushFileSystem.hh"
#include "MushArgumentList.hh"
#include "KinokoKdfStorage.hh"
#include "KinokoDataProcessor.hh"

using namespace std;


class TKinokoKdfMerge {
  public:
    TKinokoKdfMerge(const TMushArgumentList& ArgumentList);
    virtual ~TKinokoKdfMerge();
    virtual void Start(void) throw(TKinokoException);
  protected:
    vector<string> _InputFileNameList;
    bool _IsOverWriteAllowed;
    string _OutputFileName;
    vector<TKinokoStorage*> _InputStorageList;
    TKinokoStorage* _OutputStorage;
    vector<TKinokoInputStream*> _InputStreamList;
    TKinokoOutputStream* _OutputStream;
};



TKinokoKdfMerge::TKinokoKdfMerge(const TMushArgumentList& ArgumentList)
{
    for (int i = 0; i < ArgumentList.NumberOfParameters() - 1; i++) {
	_InputFileNameList.push_back(ArgumentList[i]);
    }
    _OutputFileName = ArgumentList[_InputFileNameList.size()];
    _IsOverWriteAllowed = ArgumentList.IsOptionSpecified("-f");

    _OutputStorage = 0;
}

TKinokoKdfMerge::~TKinokoKdfMerge()
{
    for (unsigned i = 0; i < _InputStorageList.size(); i++) {
	delete _InputStorageList[i];
    }

    delete _OutputStorage;
}

void TKinokoKdfMerge::Start(void) throw(TKinokoException)
{
    string InputFiles;
    for (unsigned i = 0; i < _InputFileNameList.size(); i++) {
	string InputFileName = _InputFileNameList[i];
	TKinokoStorage* InputStorage = new TKinokoKdfStorage(InputFileName);
	TKinokoInputStream* InputStream = InputStorage->GetInputStream();

	try {
	    TKinokoStorageHeader InputStorageHeader;
	    InputStorage->ReadHeader(InputStorageHeader);
	}
	catch (TKinokoException &e) {
	    /* version number mismatch ? */
	    cout << "WARNING: " << e << endl << endl;
	}

	_InputStorageList.push_back(InputStorage);
	_InputStreamList.push_back(InputStream);
	InputFiles += " " + InputFileName;
    }

    if (
	(! _IsOverWriteAllowed) &&
	(TMushFileAttribute(_OutputFileName).IsReadable())
    ){
	throw TKinokoException(
	    "file already exists: " + _OutputFileName + "\n" +
	    "use -f option to overwrite"
	);
    }

    _OutputStorage = new TKinokoKdfStorage(_OutputFileName);
    if (_IsOverWriteAllowed) {
	_OutputStorage->AllowOverWrite();
    }

    TKinokoStorageHeader StorageHeader;
    StorageHeader.AddEntry("Creator", "kdfmerge");
    StorageHeader.AddEntry("DateTime", TMushDateTime().AsString());
    StorageHeader.AddEntry("UserName", TMushUser().Name());
    StorageHeader.AddEntry("Comment", "created by kdfmerge from" + InputFiles);
    _OutputStorage->WriteHeader(StorageHeader);

    _OutputStream = _OutputStorage->GetOutputStream();

    void* Packet;
    int PacketSize, TotalPacketSize;
    do {
	TotalPacketSize = 0;
	for (unsigned i = 0; i < _InputStreamList.size(); i++) {
	    if ((PacketSize = _InputStreamList[i]->NextEntry(Packet)) > 0) {
		_OutputStream->Write(Packet, PacketSize);
		_InputStreamList[i]->Flush(Packet);
		TotalPacketSize += PacketSize;
	    }
	}
    } while (TotalPacketSize > 0);
}


int main(int argc, char** argv)
{
    TMushArgumentList ArgumentList(argc, argv);
    if (ArgumentList.NumberOfParameters() < 3) {
	cerr << "Usage: " << argv[0];
	cerr << " InputFile1 InputFile2 ... OutputFile" << endl;
        return EXIT_FAILURE;
    }

    try {
	TKinokoKdfMerge(ArgumentList).Start();
    }
    catch (TKinokoException& e) {
	cerr << "ERROR: " << e << endl;
        return EXIT_FAILURE;
    }

    return 0;
}
