/* KinokoCanvas3dPlotObject.cc */
/* Created by Enomoto Sanshiro on 11 July 2000. */
/* Last updated by Enomoto Sanshiro on 4 November 2002. */


#include <algorithm>
#include <cmath>
#include "KinokoCanvasPlotObject.hh"
#include "KinokoCanvas3dPlotObject.hh"

using namespace std;


#define clip(x, lower, upper) (min(max((lower), (x)), (upper)))


TKinokoCanvas3dPlotObject::TKinokoCanvas3dPlotObject(TKinokoCanvas* Canvas, bool IsMonochrome)
: TKinokoCanvasFramedObject(Canvas), TKinokoCanvasPlotObject(Canvas)
{
    _ZMin = 0;
    _ZMax = 1.0;
    _ZWidth = (_ZMax - _ZMin);

    _LogZMin = (_ZMin > 0) ? log10(_ZMin) : log10(0.5);
    _LogZMax = (_ZMax > 0) ? log10(_ZMax) : 0;
    _LogZWidth = _LogZMax - _LogZMin;

    _IsZScaleLog = false;
    _IsMonochrome = IsMonochrome;

    SetZCoordinate(0, 1);

    if (IsMonochrome) {
	_ColorScale = new TKinokoCanvasGrayColorScale(_Canvas);
    }
    else {
	_ColorScale = new TKinokoCanvasWhiteBackgroundRainbowColorScale(_Canvas);
    }
}

TKinokoCanvas3dPlotObject::~TKinokoCanvas3dPlotObject()
{
    delete _ColorScale;
}

TKinokoShellObject* TKinokoCanvas3dPlotObject::Clone(void)
{
    return new TKinokoCanvas3dPlotObject(_Canvas, _IsMonochrome);
}

void TKinokoCanvas3dPlotObject::Initialize(void)
{
    TKinokoCanvasPlotObject::Initialize();
    _ColorScale->Initialize();
}

vector<string> TKinokoCanvas3dPlotObject::ActionList(void)
{
    vector<string> ActionList = TKinokoCanvasPlotObject::ActionList();

    ActionList.push_back("setZScaleLinear");
    ActionList.push_back("setZScaleLog");

    return ActionList;
}

int TKinokoCanvas3dPlotObject::ProcessDrawingAction(const string& ActionName)
{
    int Result = 1;
    if (ActionName == "setZScaleLinear") {
	SetZScaleLinear();
    }
    else if (ActionName == "setZScaleLog") {
	SetZScaleLog();
    }
    else {
	Result = TKinokoCanvasPlotObject::ProcessDrawingAction(ActionName);
    }

    return Result;
}

int TKinokoCanvas3dPlotObject::SetZCoordinate(float ZMin, float ZWidth)
{
    _ZMinOrg = ZMin;
    _ZWidthOrg = ZWidth;

    if (_IsZScaleLog) {
	if (ZMin <= 0) {
	    float ZMax = ZMin + ZWidth;
	    float ZMaxForLog = (ZMax <= 0) ? 1.0 : ZMax;
	    float ZMinForLog = min(ZMaxForLog / 100.0, 0.5);
	    ZMin = ZMinForLog;
	    ZWidth = ZMaxForLog - ZMinForLog;
	}

	_LogZMin = log10(ZMin);
	_LogZMax = log10(ZMin + ZWidth);
	_LogZWidth = _LogZMax - _LogZMin;
    }

    _ZMin = ZMin;
    _ZWidth = ZWidth;
    _ZMax = ZMin + ZWidth;

    return 1;
}

int TKinokoCanvas3dPlotObject::SetZScaleLinear(void)
{
    if (_IsZScaleLog) {
	_IsZScaleLog = false;
	SetZCoordinate(_ZMinOrg, _ZWidthOrg);
    }

    return 1;
}

int TKinokoCanvas3dPlotObject::SetZScaleLog(void)
{
    if (! _IsZScaleLog) {
	_IsZScaleLog = true;
	SetZCoordinate(_ZMinOrg, _ZWidthOrg);
    }

    return 1;
}

int TKinokoCanvas3dPlotObject::ProcessSetCommand(const string& Name, const string& Value)
{
    int Result = 1;

    if (Name == "zscale") {
	if (Value == "log") {
	    SetZScaleLog();
	}
	else {
	    SetZScaleLinear();
	}
    }
    else {
	Result = TKinokoCanvasPlotObject::ProcessSetCommand(Name, Value);
    }

    return Result;
}

int TKinokoCanvas3dPlotObject::ProcessDrawingCommand(const string& Command, istream& InputStream)
{
    int Result = 0;

    if (Command == "scatterhist") {
	Result = Process2dHistCommand(InputStream, Type_Scatter);
    }
    else if (Command == "boxhist") {
	Result = Process2dHistCommand(InputStream, Type_Box);
    }
    else if (Command == "colorhist") {
	Result = Process2dHistCommand(InputStream, Type_Color);
    }
    else {
	Result = TKinokoCanvasPlotObject::ProcessDrawingCommand(Command, InputStream);
    }

    return Result;
}

int TKinokoCanvas3dPlotObject::ProcessFrameCommand(istream& InputStream)
{
    int Result = TKinokoCanvasPlotObject::ProcessFrameCommand(InputStream);
    if (Result == 0) {
	return Result;
    }

    Result = 0;

    float ZMin, ZMax;
    if (InputStream >> ZMin >> ZMax) {
	if (ZMax > ZMin) {
	    Result = SetZCoordinate(ZMin, ZMax - ZMin);
	}
    }

    return Result;
}

int TKinokoCanvas3dPlotObject::Process2dHistCommand(istream& InputStream, int Type)
{
    // syntax: 
    //   'hist' x_min x_step x_bins y_min y_step z00 z01 ...

    if (! _IsFrameValid) {
	return 1;
    }

    int Result = 0;

    float xmin, xstep, ymin, ystep;
    int xbins;
    if (! (InputStream >> xmin >> xstep >> xbins >> ymin >> ystep)) {
	return 0;
    }

    int OldColorIndex = _ImageArea->SetColor(_ColorIndex);
    int OldLineWidth = _ImageArea->SetLineWidth(_LineWidth);
    int OldLineStyleIndex = _ImageArea->SetLineStyle(_LineStyleIndex);

    float cx0, cy0, cx1, cy1;
    float x0, y0, x = xmin, y = ymin, z;
    int DataCount = 0;
    while (InputStream >> z) {
	if (DataCount++ % xbins == 0) {
	    x = x0 = xmin;

	    y0 = y;
	    y += ystep;
	    cy0 = CanvasYOf(y0);
	    cy1 = CanvasYOf(y);
	}
	x0 = x;
	x += xstep;
	cx0 = CanvasXOf(x0);
	cx1 = CanvasXOf(x);

	if ((x0 < _XMin) || (x > _XMax) || (y0 < _YMin) || (y > _YMax)) {
	    continue;
	}
	 
	float NormalizedZ;
	if (_IsZScaleLog) {
	    if (z > 0) {
		float LogZ = clip((float) log10(z), _LogZMin, _LogZMax);
		NormalizedZ = (LogZ - _LogZMin) / _LogZWidth;
	    }
	    else {
		NormalizedZ = 0;
	    }
	}
	else {
	    z = clip(z, _ZMin, _ZMax);
	    NormalizedZ = (z - _ZMin) / _ZWidth;
	}
	
	if (Type == Type_Scatter) {
	    float Area = (cx1 - cx0 + 1) * (cy0 - cy1 + 1);
	    int NumberOfPoints = (int) (Area * NormalizedZ);
	    for (int i = 0; i < NumberOfPoints; i++) {
		_ImageArea->DrawPoint(
		    cx0  + ((cx1 - cx0) * drand48()),
		    cy0  + ((cy1 - cy0) * drand48())
		);
	    }
	}
	else if (Type == Type_Box) {
	    float Ratio = sqrt(fabs(NormalizedZ));
	    if (Ratio > 0) {
		float HalfWidth = Ratio * (cx1 - cx0) / 2;
		float HalfHeight = Ratio * (cy0 - cy1) / 2;
		
		float cx = (cx0 + cx1) / 2;
		float cy = (cy0 + cy1) / 2;
		_ImageArea->DrawRect(
		    (cx - HalfWidth), (cy - HalfHeight),
		    (cx + HalfWidth), (cy + HalfHeight) 
		); 
	    }
	}
	else if (Type == Type_Color) {
	    _ImageArea->SetColor(_ColorScale->ColorIndexOf(NormalizedZ));
	    _ImageArea->DrawRectFill(cx0, cy1, cx1, cy0);
	}
    }

    _ImageArea->SetColor(OldColorIndex);
    _ImageArea->SetLineWidth(OldLineWidth);
    _ImageArea->SetLineStyle(OldLineStyleIndex);

    TKinokoCanvasPlotObject::DrawFrame();

    return Result = 1;
}



TKinokoCanvasZScaled3dPlotObject::TKinokoCanvasZScaled3dPlotObject(TKinokoCanvas* Canvas, bool IsMonochrome)
: TKinokoCanvasFramedObject(Canvas),
  TKinokoCanvas3dPlotObject(Canvas, IsMonochrome), 
  TKinokoCanvasColorScaledObject(
      Canvas, 
      (IsMonochrome ? (TKinokoCanvasColorScale*) new TKinokoCanvasGrayColorScale(Canvas) : (TKinokoCanvasColorScale*) new TKinokoCanvasWhiteBackgroundRainbowColorScale(Canvas))
  )
{
}

TKinokoCanvasZScaled3dPlotObject::~TKinokoCanvasZScaled3dPlotObject()
{
}

TKinokoShellObject* TKinokoCanvasZScaled3dPlotObject::Clone(void)
{
    return new TKinokoCanvasZScaled3dPlotObject(_Canvas, _IsMonochrome);
}

void TKinokoCanvasZScaled3dPlotObject::Initialize(void)
{
    TKinokoCanvas3dPlotObject::Initialize();
    TKinokoCanvasColorScaledObject::Initialize();
}

int TKinokoCanvasZScaled3dPlotObject::ProcessCommand(const std::string& Command, std::istream& InputStream)
{
    int Result = 0;

    if (Command == "drawzscalelabel") {
	Result = ProcessDrawZScaleLabelCommand(InputStream);
    }
    else {
	Result = TKinokoCanvas3dPlotObject::ProcessCommand(Command, InputStream);
    }

    return Result;
}

int TKinokoCanvasZScaled3dPlotObject::ProcessSetCommand(const string& Name, const string& Value)
{
    int Result = 1;

    if (Name == "ztitle") {
	_ZTitle = Value;
    }
    else if (Name == "zscalelabel") {
	if (Value == "disabled") {
	    DisableZScaleLabel();
	}
	else {
	    EnableZScaleLabel();
	}
    }
    else {
	Result = TKinokoCanvas3dPlotObject::ProcessSetCommand(Name, Value);
    }

    return Result;
}

int TKinokoCanvasZScaled3dPlotObject::ProcessFrameCommand(istream& InputStream)
{
    int Result = TKinokoCanvas3dPlotObject::ProcessFrameCommand(InputStream);
    if (Result) {
	float ZMin = TKinokoCanvas3dPlotObject::_ZMin;
	float ZWidth = TKinokoCanvas3dPlotObject::_ZWidth;
	TKinokoCanvasColorScaledObject::SetZCoordinate(ZMin, ZWidth);
    }

    return Result;
}

int TKinokoCanvasZScaled3dPlotObject::ProcessDrawZScaleLabelCommand(istream& InputStream)
{
    if (! _IsFrameValid) {
	return 1;
    }

    int Result = 0;

    int OldColorIndex = _ImageArea->SetColor(_ColorIndexList[Color_Foreground]);
    int OldLineWidth = _ImageArea->SetLineWidth(1);
    int OldLineStyleIndex = _ImageArea->SetLineStyle();
    int OldFontIndex = _ImageArea->SetFont(_FontIndexList[Font_Label]);

    int x1 = _FrameOffsetX + _FrameWidth + _ZScaleWidth;
    int y1 = _FrameOffsetY + _FrameHeight;
    int x0 = x1 - _ZScaleBoxWidth;
    int y0 = _FrameOffsetY;
    int ScaleHeight = y1 - y0;

    float z;
    if ((InputStream >> z) && ((z >= _XMin) || (z <= _XMax))) {
	float y;
	if (! TKinokoCanvasColorScaledObject::_IsZScaleLog) {
	    double ZMin = TKinokoCanvasColorScaledObject::_ZMin;
	    double ZWidth = TKinokoCanvasColorScaledObject::_ZWidth;
	    y = y1 - (ScaleHeight * (z - ZMin) / ZWidth);
	}
	else {
	    double LogZMin = TKinokoCanvasColorScaledObject::_LogZMin;
	    double LogZWidth = TKinokoCanvasColorScaledObject::_LogZWidth;
	    y = y1 - (ScaleHeight * (log10(z) - LogZMin) / LogZWidth);
	}

	string Label;
	InputStream >> ws;
	if (getline(InputStream, Label)) {
	    _ImageArea->DrawText(x0 - 3 , y, Label, "cr");
	}
	else {
	    //... use typesetter 
	    _ImageArea->DrawNumberText(x0 - 3 , y, z, "cr");
	}

	Result = 1;
    }

    _ImageArea->SetColor(OldColorIndex);
    _ImageArea->SetLineWidth(OldLineWidth);
    _ImageArea->SetLineStyle(OldLineStyleIndex);
    _ImageArea->SetFont(OldFontIndex);

    return Result;
}

int TKinokoCanvasZScaled3dPlotObject::SetZScaleLinear(void)
{
    TKinokoCanvas3dPlotObject::SetZScaleLinear();
    TKinokoCanvasColorScaledObject::SetZScaleLinear();

    return 1;
}

int TKinokoCanvasZScaled3dPlotObject::SetZScaleLog(void)
{
    TKinokoCanvas3dPlotObject::SetZScaleLog();
    TKinokoCanvasColorScaledObject::SetZScaleLog();

    return 1;
}

int TKinokoCanvasZScaled3dPlotObject::DrawFrame(void)
{
    int Result = TKinokoCanvas3dPlotObject::DrawFrame();
    if (Result > 0) {
	TKinokoCanvasColorScaledObject::DrawFrame();
    }

    return Result;
}
