/* KinokoCanvasDataScale.cc */
/* Created by Enomoto Sanshiro on 11 July 2000. */
/* Last updated by Enomoto Sanshiro on 7 December 2003. */


#include <iostream>
#include <iomanip>
#include <sstream>
#include <string>
#include <algorithm>
#include <cmath>
#include <climits>
#include "MushMisc.hh"
#include "KinokoCanvasTypesetter.hh"
#include "KinokoCanvasDataScale.hh"

using namespace std;


TKinokoCanvasDataScale::TKinokoCanvasDataScale(float Min, float Max, int MaxNumberOfDivisions)
{
    _Min = min(Min, Max);
    _Max = max(Min, Max);
    _MaxNumberOfDivisions = MaxNumberOfDivisions;

    _LabelScalingFactor = 1;

    _Width = _Max - _Min;
    _FirstDivision = _Min;
    _LastDivision = _Max;
    _DivisionStep = -1;
    _NumberOfDivisions = -1;
    _TickStep = -1;
    _NumberOfTicks = -1;

    _UsedMin = _Min;
    _UsedMax = _Max;
}

TKinokoCanvasDataScale::~TKinokoCanvasDataScale()
{
}

float TKinokoCanvasDataScale::Min(void) const
{
    return _Min;
}

float TKinokoCanvasDataScale::Max(void) const
{
    return _Max;
}

float TKinokoCanvasDataScale::Width(void) const
{
    return _Width;
}

TKinokoCanvasDataScale& TKinokoCanvasDataScale::Rescale(float Min, float Max, int MaxNumberOfDivisions)
{
    _Min = min(Min, Max);
    _Max = max(Min, Max);
    _MaxNumberOfDivisions = MaxNumberOfDivisions;

    _Width = Max - Min;

    // recalculate steps only when the range has been changed //
    if (
	(fabs(_Min - _UsedMin) > _Width * 1.0e-5) ||
	(fabs(_Max - _UsedMax) > _Width * 1.0e-5) ||
	(_MaxNumberOfDivisions == 0)
    ){
	_FirstDivision = _Min;
	_LastDivision = _Max;
	_DivisionStep = -1;
	_NumberOfDivisions = -1;
	_NumberOfTicks = -1;
	_UsedMin = _Min;
	_UsedMax = _Max;
	_LabelScalingFactor = 1;
    }

    return *this;
}

string TKinokoCanvasDataScale::FactorizeScaleLabel(void)
{
    string FactorLabel;

    double RangeScale = max(fabs(_Min), fabs(_Max));
    if ((RangeScale >= 1.0e+4) || (RangeScale <= 9.9e-2)) {
	double ScalingOrder = floor(log10(1.01 * RangeScale));
	double ScalingFactor = pow(10, ScalingOrder);
	if (fabs(RangeScale/ScalingFactor) < 2) {
	    ScalingOrder -= 1;
	    ScalingFactor /= 10.0;
	}
	
	ostringstream os;
	os << "{#times}10^{" << ScalingOrder << "}";
	FactorLabel = os.str();
	
	_LabelScalingFactor = ScalingFactor;
    }

    return FactorLabel;
}

int TKinokoCanvasDataScale::NumberOfDivisions(void)
{
    if (_NumberOfDivisions < 0) {
	pair<float, float> StepData = DecideSteps(_Min, _Max, _MaxNumberOfDivisions);
	_FirstDivision = StepData.first;
	_DivisionStep = StepData.second;

	_NumberOfDivisions = (int) ((_Max - _FirstDivision) / _DivisionStep) + 1;
	_LastDivision = _FirstDivision + (_NumberOfDivisions-1) * _DivisionStep;
    }

    return _NumberOfDivisions;
}

float TKinokoCanvasDataScale::DivisionValueOf(int DivisionIndex)
{
    if (_NumberOfDivisions < 0) {
	NumberOfDivisions();
    }

    double Value = _FirstDivision + DivisionIndex * _DivisionStep;
    if (fabs(Value / _DivisionStep) < 1.0e-6) {
	Value = 0;
    }

    return Value;
}

string TKinokoCanvasDataScale::DivisionLabelOf(int DivisionIndex)
{
    float Value = DivisionValueOf(DivisionIndex) / _LabelScalingFactor;

    ostringstream os;
    os << Value;

    return os.str();
}

int TKinokoCanvasDataScale::NumberOfTicks(void)
{
    if (_NumberOfTicks >= 0) {
	return _NumberOfTicks;
    }

    if (_NumberOfDivisions < 0) {
	NumberOfDivisions();
    }
    
    if (_NumberOfTicks < 0) {
	int Exponent = (int) log10(1.01 * _DivisionStep);
	if (_DivisionStep < 0.99) {
	    Exponent -= 1;
	}
	int Factor = (int) ((1.01 * _DivisionStep) / pow(10.0, Exponent));
	if (Factor == 1) {
	    Factor = 5;
	}
	else if (Factor == 2) {
	    Factor = 4;
	}
	else if (Factor > 5) {
	    Factor /= 2;
	}

	_TickStep = _DivisionStep / Factor;
	_NumberOfTicks = (_NumberOfDivisions - 1) * Factor + 1;
	int NumberOfLeadingTicks = (int) ((_FirstDivision - _Min) / _TickStep);
	int NumberOfTrailingTicks = (int) ((_Max - _LastDivision) / _TickStep);
	_FirstTick = _FirstDivision - _TickStep * NumberOfLeadingTicks;
	_NumberOfTicks += NumberOfLeadingTicks + NumberOfTrailingTicks;
    }

    return _NumberOfTicks;
}

float TKinokoCanvasDataScale::TickValueOf(int TickIndex)
{
    if (_NumberOfTicks < 0) {
	NumberOfTicks();
    }

    return _FirstTick + TickIndex * _TickStep;
}

pair<float, float> TKinokoCanvasDataScale::DecideSteps(float Min, float Max, int MaxNumberOfSteps)
{
    int NumberOfSteps = max(2, MaxNumberOfSteps / 2 + 1);

    pair<float, float> Step;  // pair of first-step-value and step-width
    float Width = Max - Min;

    // order-normalization to make the width fit in range [N/2, 20*N]
    if (Width > 20 * NumberOfSteps) {
	Step = DecideSteps(Min / 10, Max / 10, MaxNumberOfSteps);
	Step.first *= 10;
	Step.second *= 10;
	return Step;
    }
    else if (Width < NumberOfSteps / 2.0) {
	Step = DecideSteps(Min * 10, Max * 10, MaxNumberOfSteps);
	Step.first /= 10;
	Step.second /= 10;
	return Step;
    }

    // factor-normalization(5) to make the width fit in range [N/2, 5*N)
    else if (Width > 4.99 * NumberOfSteps) {
	// I*5 normalization: once at most //
	// here the upper limit should be greater than 5*N to avoid
	// subsequent I*0.5 normalization which result in ugly 
	// I*2.5 normalization
	Step = DecideSteps(Min / 5, Max / 5, MaxNumberOfSteps);
	Step.first *= 5;
	Step.second *= 5;
	return Step;
    }

    // factor-normalization(2) to make the width fit in range (N, 2*N]
    else if (Width > 2.01 * NumberOfSteps) {
	// I*2 normalization: twice at most //
	Step = DecideSteps(Min / 2, Max / 2, MaxNumberOfSteps);
	Step.first *= 2;
	Step.second *= 2;
	return Step;
    }
    else if (Width < 0.99 * NumberOfSteps) {
	// I*0.5 normalization: once at most //
	Step = DecideSteps(Min * 2, Max * 2, MaxNumberOfSteps);
	Step.first /= 2;
	Step.second /= 2;
	return Step;
    }

    if ((Min < -LONG_MAX) || (Min > LONG_MAX)) {
	Step.first = Min;
    }
    else {
	Step.first = ((long) Min < Min) ? ((long) Min + 1) : ((long) Min);
    }

    Step.second = 1;

    return Step;
}



TKinokoCanvasLogDataScale::TKinokoCanvasLogDataScale(float Min, float Max, int NumberOfDivisions)
: TKinokoCanvasDataScale(Min, Max, NumberOfDivisions)
{
    _IsUsingNormalScale = false;
    
    //...  to be removed
    if ((Min <= 0) || (Max <= Min)) {
	cerr << "WARNING: TKinokoCanvasLogDataScale::TKinokoCavansLogDataScale(): ";
	cerr << "invalid log scale range: ";
	cerr << "(" << Min << ", " << Max << ")" << endl;
    }
    if (_Min <= 0) {
	_Min = 0.5;
    }
    if (_Max <= _Min) {
	_Min = _Max / 100.0;
    }
}

TKinokoCanvasLogDataScale::~TKinokoCanvasLogDataScale()
{
}

TKinokoCanvasDataScale& TKinokoCanvasLogDataScale::Rescale(float Min, float Max, int NumberOfDivisions)
{
    //...  to be removed
    if ((Min <= 0) || (Max <= Min)) {
	cerr << "WARNING: TKinokoCanvasLogDataScale::Rescale(): ";
	cerr << "invalid log scale range: ";
	cerr << "(" << Min << ", " << Max << ")" << endl;
    }
    if (Min <= 0) {
	Min = 0.5;
    }
    if (Max <= Min) {
	Min = Max / 10.0;
    }

    return TKinokoCanvasDataScale::Rescale(Min, Max, NumberOfDivisions);
}

string TKinokoCanvasLogDataScale::FactorizeScaleLabel(void)
{
    return "";
}

int TKinokoCanvasLogDataScale::NumberOfDivisions(void)
{
    if (_NumberOfDivisions >= 0) {
	return _NumberOfDivisions;
    }

    float LogMin = log10(_Min);
    float LogMax = log10(_Max);

    int ILogMin = (int) (((int) LogMin < LogMin) ? (LogMin + 1) : LogMin);
    float LogWidth = LogMax - ILogMin;
    
    if (LogWidth > 1) {
	int LogDivisionStep = (int) ((LogWidth-1) / _MaxNumberOfDivisions) + 1;
	_NumberOfDivisions = (int) (LogWidth / LogDivisionStep) + 1;
	_DivisionStep = pow(10.0, (double) LogDivisionStep);
	_FirstDivision = pow(10.0, (double) ILogMin);
	_LastDivision = _FirstDivision * pow(_DivisionStep, _NumberOfDivisions - 1);
	_IsUsingNormalScale = false;
    }
    else {
	_NumberOfDivisions = -1;
	_NumberOfDivisions = TKinokoCanvasDataScale::NumberOfDivisions();
	_IsUsingNormalScale = true;
    }

    return _NumberOfDivisions;
}

float TKinokoCanvasLogDataScale::DivisionValueOf(int DivisionIndex)
{
    if (_NumberOfDivisions < 0) {
	NumberOfDivisions();
    }

    if (_IsUsingNormalScale) {
	return TKinokoCanvasDataScale::DivisionValueOf(DivisionIndex);
    }

    return _FirstDivision * pow(_DivisionStep, DivisionIndex);
}

string TKinokoCanvasLogDataScale::DivisionLabelOf(int DivisionIndex)
{
    if ((_Max / _Min) <= 100) {
	return TKinokoCanvasDataScale::DivisionLabelOf(DivisionIndex);
    }
    
    float Value = DivisionValueOf(DivisionIndex) / _LabelScalingFactor;
    return TKinokoCanvasMathTextComposer::PowerOf10TextOf(Value);
}

int TKinokoCanvasLogDataScale::NumberOfTicks(void)
{
    if (_NumberOfTicks >= 0) {
	return _NumberOfTicks;
    }

    if (_NumberOfDivisions < 0) {
	NumberOfDivisions();
    }

    if (! _IsUsingNormalScale) {
	double OrderOfMin = floor(log10(1.01 * _Min));
	double OrderOfMax = floor(log10(1.01 * _Max));
	double FactorOfMin = ceil(0.99 * (_Min / pow(10.0, OrderOfMin)));
	double FactorOfMax = floor(1.01 * (_Max / pow(10.0, OrderOfMax)));

	_TickIndexOffset = (int) ((FactorOfMin - 1) + 0.1);
	_FirstTick = pow(10.0, OrderOfMin);
	_NumberOfTicks = (int) (9 * (OrderOfMax - OrderOfMin) + (FactorOfMax - 1) + 0.1) - _TickIndexOffset;
    }
    else {
	_NumberOfTicks = -1;
	_NumberOfTicks = TKinokoCanvasDataScale::NumberOfTicks();
    }

    return _NumberOfTicks;
}

float TKinokoCanvasLogDataScale::TickValueOf(int TickIndex)
{
    if (_NumberOfTicks < 0) {
	NumberOfTicks();
    }

    if (_IsUsingNormalScale) {
	return TKinokoCanvasDataScale::TickValueOf(TickIndex);
    }

    TickIndex += _TickIndexOffset;

    return (TickIndex % 9 + 1) * _FirstTick * pow(10.0, TickIndex / 9);
}



TKinokoCanvasTimeDataScale::TKinokoCanvasTimeDataScale(float Min, float Max, int MaxNumberOfDivisions)
: TKinokoCanvasDataScale(Min, Max, MaxNumberOfDivisions)
{
    _Epoch = 0;
    _Unit = 0;
    _Step = 0;

    _NumberOfDivisions = -1;
}

TKinokoCanvasTimeDataScale::~TKinokoCanvasTimeDataScale()
{
}

void TKinokoCanvasTimeDataScale::SetTimeTick(long Epoch, const string& Format, int Unit, int Step)
{
    _Epoch = Epoch;
    _Format = Format;
    _Unit = Unit;
    _Step = Step;

    _NumberOfDivisions = -1;
}

TKinokoCanvasDataScale& TKinokoCanvasTimeDataScale::Rescale(float Min, float Max, int MaxNumberOfDivisions)
{
    return TKinokoCanvasDataScale::Rescale(Min, Max, MaxNumberOfDivisions);
}

string TKinokoCanvasTimeDataScale::FactorizeScaleLabel(void)
{
    return "";
}

int TKinokoCanvasTimeDataScale::NumberOfDivisions(void)
{
    if (_NumberOfDivisions < 0) {
	SetDivisions();
    }

    return _NumberOfDivisions;
}

float TKinokoCanvasTimeDataScale::DivisionValueOf(int DivisionIndex)
{
    if (_NumberOfDivisions < 0) {
	SetDivisions();
    }

    return _DivisionList[DivisionIndex].first;
}

string TKinokoCanvasTimeDataScale::DivisionLabelOf(int DivisionIndex)
{
    if (_NumberOfDivisions < 0) {
	SetDivisions();
    }

    return _DivisionList[DivisionIndex].second;
}

int TKinokoCanvasTimeDataScale::NumberOfTicks(void)
{
    if (_NumberOfDivisions < 0) {
	SetDivisions();
    }

    return _TickList.size();
}

float TKinokoCanvasTimeDataScale::TickValueOf(int TickIndex)
{
    if (_NumberOfDivisions < 0) {
	SetDivisions();
    }

    return _TickList[TickIndex];
}

void TKinokoCanvasTimeDataScale::SetDivisions()
{
    double Range = _Max - _Min;

    int ThisUnit = _Unit;
    if (ThisUnit == TimeUnit_Undefined) {
	if (Range < 5 * 60) {
	    ThisUnit = TimeUnit_Second;
	}
	else if (Range < 3.0 * 60*60) {
	    ThisUnit = TimeUnit_Minute;
	}
	else if (Range < 3.0 * 24*60*60) {
	    ThisUnit = TimeUnit_Hour;
	}
	else if (Range < 2.0 * 30*24*60*60) {
	    ThisUnit = TimeUnit_Day;
	}
	else if (Range < 2.0 * 365*24*60*60) {
	    ThisUnit = TimeUnit_Month;
	}
	else {
	    ThisUnit = TimeUnit_Year;
	}
    }	

    double UnitWidth;
    string DefaultFormat;
    switch (ThisUnit) {
      case TimeUnit_Second:
	UnitWidth = 1;
	DefaultFormat = "%S";
	break;
      case TimeUnit_Minute:
	UnitWidth = 60;
	DefaultFormat = "%M";
	break;
      case TimeUnit_Hour:
	UnitWidth = 3600;
	DefaultFormat = "%H";
	break;
      case TimeUnit_Day:
	UnitWidth = 86400;
	DefaultFormat = "%d";
	break;
      case TimeUnit_Month:
	UnitWidth = 30*86400;
	DefaultFormat = "%m";
	break;
      case TimeUnit_Year:
	UnitWidth = 365.25*86400;
	DefaultFormat = "%y";
	break;
      default:
	UnitWidth = 1;
	DefaultFormat = "%S";
	break;
    }

    int ThisStep = _Step;
    if (ThisStep == 0) {
	int NumberOfUnitTicks = max((int) (Range / UnitWidth), 1);
	ThisStep = (NumberOfUnitTicks - 1) /_MaxNumberOfDivisions + 1;
    }

    string ThisFormat = _Format;
    if (ThisFormat.empty()) {
	ThisFormat = DefaultFormat;
    }

    _DivisionList.erase(_DivisionList.begin(), _DivisionList.end());
    _TickList.erase(_TickList.begin(), _TickList.end());

    TMushDateTime TickDate(_Epoch + (int) _Min);
    int Year = TickDate.Year();
    int Month = TickDate.Month();
    int Day = TickDate.Day();
    int Hour = TickDate.Hour();
    int Minute = TickDate.Min();
    int Second = TickDate.Sec();

    int StepToNextDivision = 0;
    while (1) {
	switch (ThisUnit) {
	  case TimeUnit_Second:
	    TickDate.Set(Year, Month, Day, Hour, Minute, Second);
	    Second++;
	    break;
	  case TimeUnit_Minute:
	    TickDate.Set(Year, Month, Day, Hour, Minute, Second=0);
	    Minute++;
	    break;
	  case TimeUnit_Hour:
	    TickDate.Set(Year, Month, Day, Hour, Minute=0, Second=0);
	    Hour++;
	    break;
	  case TimeUnit_Day:
	    TickDate.Set(Year, Month, Day, Hour=0, Minute=0, Second=0);
	    Day++;
	    break;
	  case TimeUnit_Month:
	    TickDate.Set(Year, Month, Day=1, Hour=0, Minute=0, Second=0);
	    Month++;
	    break;
	  case TimeUnit_Year:
	    TickDate.Set(Year, Month=1, Day=1, Hour=0, Minute=0, Second=0);
	    Year++;
	    break;
	  default:
	    TickDate.Set(_Epoch + (long) (_Max + 0.1*Range));
	    break;
	}
	RegulateDate(Year, Month, Day, Hour, Minute, Second);

	double Value = TickDate.AsLong() - _Epoch;
	if (Value < _Min) {
	    continue;
	}
	if (Value > _Max) {
	    break;
	}

	if (StepToNextDivision <= 0) {
	    string Label = TickDate.AsString(ThisFormat);
	    _DivisionList.push_back(make_pair(Value, Label));
	    StepToNextDivision = ThisStep;
	}
	_TickList.push_back(Value);
	StepToNextDivision--;
    }

    _NumberOfDivisions = _DivisionList.size();
}

inline static int LastDayOfMonth(int Year, int Month)
{
    static const int LastDay[] = {
	0, 
	31, 28, 31, 30, 31, 30,
	31, 31, 30, 31, 30, 31
    };

    if ((Month == 2) && (Year % 4 == 0) && (Year % 100 != 0)) {
	return 29;
    }
    else {
	return LastDay[Month];
    }
}

void TKinokoCanvasTimeDataScale::RegulateDate(int& Year, int& Month, int& Day, int& Hour, int& Min, int& Sec)
{
    if (Sec >= 60) {
	Min += Sec / 60;
	Sec %= 60;
    }
    if (Min >= 60) {
	Hour += Min / 60;
	Min %= 60;
    }
    if (Hour >= 24) {
	Day += Hour / 24;
	Hour %= 24;
    }

    if (Month > 12) {
	Year += Month / 12;
	Month %= 12;
    }

    while (Day > LastDayOfMonth(Year, Month)) {
	Day -= LastDayOfMonth(Year, Month);
	Month++;

	if (Month > 12) {
	    Month -= 12;
	    Year++;
	}
    }
}



TKinokoCanvasElapseDataScale::TKinokoCanvasElapseDataScale(float Min, float Max, int MaxNumberOfDivisions)
: TKinokoCanvasDataScale(Min, Max, MaxNumberOfDivisions)
{
}

TKinokoCanvasElapseDataScale::~TKinokoCanvasElapseDataScale()
{
}

string TKinokoCanvasElapseDataScale::FactorizeScaleLabel(void)
{
    return "";
}

std::string TKinokoCanvasElapseDataScale::DivisionLabelOf(int DivisionIndex)
{
    int Value = (int) (DivisionValueOf(DivisionIndex) / _LabelScalingFactor);

    int Sec = Value % 60;
    int Min = (Value / 60) % 60;
    int Hour = Value / 3600;

    bool ShowsHour = (_Max >= 3600);
    bool ShowsMin = (_Max >= 60);
    bool ShowsSec = ((_Max - _Min) < 3600);

    ostringstream os;
    if (ShowsHour) {
	os << Hour;
	os << ":" << setfill('0') << setw(2);
	ShowsMin = true;
    }
    if (ShowsMin) {
	os << Min;
	if (ShowsSec) {
	    os << "'" << setfill('0') << setw(2);
	}
    }
    if (ShowsSec) {
	os << Sec;
    }
    
    return os.str();
}

pair<float, float> TKinokoCanvasElapseDataScale::DecideSteps(float Min, float Max, int MaxNumberOfSteps)
{
    static const int Minute = 60;
    static const int Hour = 60 * Minute;

    pair<float, float> Step;  // pair of first-step-value and step-width
    float Width = Max - Min;

    if (Width > 1.2*Hour) {
	Step = TKinokoCanvasDataScale::DecideSteps(
	    Min / Hour, Max / Hour, MaxNumberOfSteps
	);
	Step.first *= Hour;
	Step.second *= Hour;
    }
    else if (Width > 1.2*Minute) {
	Step = TKinokoCanvasDataScale::DecideSteps(
	    Min / Minute, Max / Minute, MaxNumberOfSteps
	);
	Step.first *= Minute;
	Step.second *= Minute;
    }
    else {
	Step = TKinokoCanvasDataScale::DecideSteps(
	    Min, Max, MaxNumberOfSteps
	);
    }

    // thin out, since time labels can be longer than usual //
    MaxNumberOfSteps = max(2, (int) (0.7 * MaxNumberOfSteps));
    int NumberOfSteps = (int) (Width / Step.second) + 1;
    if (NumberOfSteps > MaxNumberOfSteps) {
	Step.second *= 2;
    }

    return Step;
}
